//----------------------------------------------------------------------------------------
// Name:        exclusion_dialog.cpp
// Purpose:     Exclusion lists dialog
// Author:      Robert O'Connor
// Modified by:
// Created:     2001/10/20
// Copyright:   (c) Robert O'Connor ( rob@medicalmnemonics.com )
// Licence:     GPL
// RCS-ID:      $Id: exclusion_dialog.cpp,v 1.8 2001/12/29 01:59:09 robertoconnor Exp $
//----------------------------------------------------------------------------------------

// ---------------------------------------------------------------------------------------
// Headers
// ---------------------------------------------------------------------------------------

#ifdef __GNUG__
    #pragma implementation "exclusion_dialog.cpp"
    #pragma interface "exclusion_dialog.cpp"
#endif
// ---------------------------------------------------------------------------------------
// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// For all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWindows headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif
// ---------------------------------------------------------------------------------------
// ---------------------------------------------------------------------------------------
#include "wx/xrc/xmlres.h"              // XRC XML resouces
// ---------------------------------------------------------------------------------------
#include "exclusion_dialog.h"
#include "blocked_dialog.h"
#include "configuration.h"

// ---------------------------------------------------------------------------------------
// Private variables
// ---------------------------------------------------------------------------------------

exclusion_dialog* the_exclusion_dialog = NULL;

// ---------------------------------------------------------------------------------------
// Internal constants
// ---------------------------------------------------------------------------------------

// Used in automatically sizing the listctrl's column widths to use full width
// without causing a horizontal scrollbar in the listctrl object window.
#define TOTAL_SPACING_WIDTH     21  // Sum of pixel spacing between all columns + scrollbar

// Custom version of XMLCTRL macros for the list controls, since they will be 
// used so much in the code.
#ifdef __WXDEBUG__
#define EXCLUSION_EXTENSIONS_LISTCTRL \
    ( wxDynamicCast( ( *the_exclusion_dialog ).FindWindow( XMLID( "exclusion_dialog_extensions_tab_listctrl" ) ), wxListCtrl) )
#else
#define EXCLUSION_EXTENSIONS_LISTCTRL \
    ( ( wxListCtrl* )( ( *the_exclusion_dialog ).FindWindow( XMLID( "exclusion_dialog_extensions_tab_listctrl" ) ) ) )
#endif

#ifdef __WXDEBUG__
#define EXCLUSION_URLS_LISTCTRL \
    ( wxDynamicCast( ( *the_exclusion_dialog ).FindWindow( XMLID( "exclusion_dialog_urls_tab_listctrl" ) ), wxListCtrl) )
#else
#define EXCLUSION_URLS_LISTCTRL \
    ( ( wxListCtrl* )( ( *the_exclusion_dialog ).FindWindow( XMLID( "exclusion_dialog_urls_tab_listctrl" ) ) ) )
#endif

// Notebook pages indexes for context help and which listctrl to add/update.
enum {
    EXCLUSION_EXTENSIONS_PAGE    = 0,
    EXCLUSION_URLS_PAGE
};

// ---------------------------------------------------------------------------------------
// Event table: connect the events to the handler functions to process them
// ---------------------------------------------------------------------------------------

BEGIN_EVENT_TABLE(exclusion_dialog, wxDialog)
    EVT_BUTTON( XMLID( "exclusion_dialog_extensions_tab_add_button" ), exclusion_dialog::on_extensions_tab_add_button )
    EVT_BUTTON( XMLID( "exclusion_dialog_extensions_tab_edit_button" ), exclusion_dialog::on_extensions_tab_edit_button )
    EVT_BUTTON( XMLID( "exclusion_dialog_extensions_tab_delete_button" ), exclusion_dialog::on_extensions_tab_delete_button )

    EVT_BUTTON( XMLID( "exclusion_dialog_urls_tab_add_button" ), exclusion_dialog::on_urls_tab_add_button )
    EVT_BUTTON( XMLID( "exclusion_dialog_urls_tab_edit_button" ), exclusion_dialog::on_urls_tab_edit_button )
    EVT_BUTTON( XMLID( "exclusion_dialog_urls_tab_delete_button" ), exclusion_dialog::on_urls_tab_delete_button )

    // A double-click or enter key on a listctrl row. Same action as if a respective 'edit button' click.
    EVT_LIST_ITEM_ACTIVATED( XMLID( "exclusion_dialog_extensions_tab_listctrl" ), exclusion_dialog::on_extensions_tab_edit_button ) 
    EVT_LIST_ITEM_ACTIVATED( XMLID( "exclusion_dialog_urls_tab_listctrl" ), exclusion_dialog::on_urls_tab_edit_button ) 
   
    EVT_NOTEBOOK_PAGE_CHANGED( -1, exclusion_dialog::OnPageChanged )
    
    EVT_UPDATE_UI( XMLID( "exclusion_dialog_extensions_tab_listctrl" ), exclusion_dialog::on_update_ui_extensions_tab_listctrl )
    EVT_UPDATE_UI( XMLID( "exclusion_dialog_urls_tab_listctrl" ), exclusion_dialog::on_update_ui_urls_tab_listctrl )
    EVT_BUTTON( wxID_OK, exclusion_dialog::OnOK )
END_EVENT_TABLE()

//----------------------------------------------------------------------------------------
// Non-event handler functions
//----------------------------------------------------------------------------------------

void init_a_exclusion_dialog( wxWindow* parent, wxString channel_section )
{
    the_exclusion_dialog = new exclusion_dialog; 
    wxTheXmlResource->LoadDialog( the_exclusion_dialog, parent, "exclusion_dialog" );
    
    // Set up the listctrls
    the_exclusion_dialog->listctrl_init();
    
    the_exclusion_dialog->m_channel_section = channel_section;
    
    // Set the selected notebook page index to 0 (EXCLUSION_EXTENSIONS_PAGE),
    // since always initially show page 0 when start a dialog.
    the_exclusion_dialog->m_selected_exclusion_notebookpage_index = EXCLUSION_EXTENSIONS_PAGE;
    
    the_exclusion_dialog->ShowModal();
    
}

//Initialize the list control. EXCLUSION_LISTCTRL macro is defined above.
void exclusion_dialog::listctrl_init()
{
    int ctrl_width;
    int ctrl_height;
    
    // TODO: Save the column widths to a textfile. Error correct if they vanish a column
    // by making it too thin.
    EXCLUSION_EXTENSIONS_LISTCTRL->SetSingleStyle( wxLC_REPORT );
    EXCLUSION_EXTENSIONS_LISTCTRL->GetSize( &ctrl_width, &ctrl_height );    
    EXCLUSION_URLS_LISTCTRL->SetSingleStyle( wxLC_REPORT );
    EXCLUSION_URLS_LISTCTRL->GetSize( &ctrl_width, &ctrl_height ); 
        
    EXCLUSION_EXTENSIONS_LISTCTRL->InsertColumn( EXCLUSION_COLUMN, _T( "File Extension" ), wxLIST_FORMAT_LEFT, ( ctrl_width-TOTAL_SPACING_WIDTH-140 )  );
    EXCLUSION_EXTENSIONS_LISTCTRL->InsertColumn( ACTION_COLUMN, _T( "Action" ), wxLIST_FORMAT_LEFT, ( 70 )  );
    EXCLUSION_EXTENSIONS_LISTCTRL->InsertColumn( PRIORITY_COLUMN, _T( "Priority" ), wxLIST_FORMAT_LEFT, 70 );
    EXCLUSION_URLS_LISTCTRL->InsertColumn( EXCLUSION_COLUMN, _T( "URL" ), wxLIST_FORMAT_LEFT, ( ctrl_width-TOTAL_SPACING_WIDTH-140)  );
    EXCLUSION_URLS_LISTCTRL->InsertColumn( ACTION_COLUMN, _T( "Action" ), wxLIST_FORMAT_LEFT, ( 70)  );
    EXCLUSION_URLS_LISTCTRL->InsertColumn( PRIORITY_COLUMN, _T( "Priority" ), wxLIST_FORMAT_LEFT, 70 );

    // To speed up inserting we hide the control temporarily
    EXCLUSION_EXTENSIONS_LISTCTRL->Hide();
    EXCLUSION_URLS_LISTCTRL->Hide();

    // Load the exclusions into the rows of the listctrl
    the_exclusion_dialog->listctrl_load_rows();
    wxLogDebug( "Finished loading list control rows");
    
    // Finished inserting. Show the control
    EXCLUSION_EXTENSIONS_LISTCTRL->Show();
    EXCLUSION_URLS_LISTCTRL->Show();
}

void exclusion_dialog::listctrl_load_rows()
{    
    wxString string;                // Holds the line of text read from the TextFile
    wxTextFile textfile;            // The textfile to open
    wxString filename;              // The textfile's filename
    bool successful;                // Whether opening the file was successful.
    long extensions_row_number = 0; // Row number of extensions listctrl to insert the info
    long urls_row_number = 0;       // Row number of urls listctrl to insert the info

    wxString exclusion;             // The string to exlude
    wxString action;                // The action of excluded item (+ or -)
    wxString priority;              // The priority of the excluded item

    // Get the list of exclusion list filenames from config file
    if ( m_channel_section == "DEFAULT" ) {
        filename = the_configuration->Read( get_os_configuration_section() << "exclusion_lists", _T("exclusionlist.txt") );
    } else {
        filename = the_configuration->Read( m_channel_section << "exclusion_lists", _T("exclusionlist.txt") );
    }
    
    // Possible (if a power user) to have multiple of them separated by semicolons.
    // Shorten the filename to whatever is in front of the first semi-colon.
    // TODO: could examine if contains a ';' to test for multiple files, and if so
    // then pop up a multiple choice dialog asking which one they want to edit.
    filename = filename.BeforeFirst(';');
    wxLogDebug( "Name of exclusionlist filename is %s", filename.c_str() );
    
    // Keep the filename in a class variable for later
    m_exclusion_filename = filename;
    
    // Try to open this file
    successful = textfile.Open( filename );    
    // If can't open it, try to create a new one here.
    if (! successful ) {
        successful = textfile.Create( filename );
    }
    wxLogDebug( "Ability to open or create file is %ld", successful );
    
    // Can't open or create the file, must be a bad filename or something else wrong with
    // it. TODO: Add some error handling for that.

    // A loop to load up the lines one by one from the texttextfile.
    for ( string = textfile.GetFirstLine(); !textfile.Eof(); string = textfile.GetNextLine() ) {
        // Ignore lines that are empty or start with a '#' as they are comments
        if ( ! ( string.StartsWith( "#", NULL ) || ( string == _T("") ) ) ) {        
            // Entries are of the format
            // 0:-:.*\.mp3$
            // Priority : Include(+) or Exclude(-) : Item to exclude (can contain a ':')
            priority = string.BeforeFirst(':');
            // Since exclusion can contain a ':' (for example http:), can use AfterLast
            // yet. Therefore, first reduce the string to what is after the first ':'
            string = string.AfterFirst(':');
            // Now the first ':' of remaining string safely divides the action and exclusion
            action = string.BeforeFirst(':');        
            exclusion = string.AfterFirst(':');   
            // More descriptive to have "Include" / "Exclude" that just a +/-, so convert them
            if ( action == '-') {
                action = _T( "Exclude" );
            } else {
                action = _T( "Include" );
            }
            
            // If there is no "http" in the string, treat it as a file extension.
            // and stick it into the file extension listctrl. TODO: If people are putting
            // non-http URLS in there, then need to have a commented section divider in the 
            // file, like #--------BEGIN URLS----------.
            if ( ! exclusion.Contains( "http" ) ) {
                EXCLUSION_EXTENSIONS_LISTCTRL->InsertItem( extensions_row_number, exclusion, 0 );
                EXCLUSION_EXTENSIONS_LISTCTRL->SetItem( extensions_row_number, ACTION_COLUMN, action );
                EXCLUSION_EXTENSIONS_LISTCTRL->SetItem( extensions_row_number, PRIORITY_COLUMN, priority );              
                extensions_row_number++; 
            // Else put in in the url tab.
            } else {
                EXCLUSION_URLS_LISTCTRL->InsertItem( urls_row_number, exclusion, 0 );
                EXCLUSION_URLS_LISTCTRL->SetItem( urls_row_number, ACTION_COLUMN, action );
                EXCLUSION_URLS_LISTCTRL->SetItem( urls_row_number, PRIORITY_COLUMN, priority );
                urls_row_number++;
            }        
        }
    }
    // Closes the file and frees the memory.
    textfile.Close();
}


// Deletes a row (item)
void exclusion_dialog::listctrl_delete_selected_row() 
{    
    // Selected_item_id of -1 means that nothing is selected, so just abort
    if ( listctrl_get_selected_item_id() == -1 ) {
        return;
    }
    
    // Make a page specific message for the confirm dialog
    wxString confirmation_string;
    if ( m_selected_exclusion_notebookpage_index == EXCLUSION_EXTENSIONS_PAGE ) {
        confirmation_string = _T( "Are you sure you wish to delete this " \
                                  "file extension from your exclusion list?" );
    } else {
        confirmation_string = _T( "Are you sure you wish to delete this " \
                                  "URL from your exclusion list?" );
    }
        
    // Create a confirm dialog
    wxMessageDialog confirm_dlg( this, confirmation_string, _T( "Confirm delete" ),
                                 wxYES_NO|wxYES_DEFAULT|wxICON_QUESTION );
    
    // If they say yes, then delete it, and mark the file as dirty so will know
    // that it needs a save.
    if ( confirm_dlg.ShowModal() == wxID_YES ) {
        get_selected_listctrl()->DeleteItem( listctrl_get_selected_item_id() );
        m_exclusion_file_is_dirty = TRUE; 
    }
}


// Inserts an empty row (item), returning the row number of the new insertion
// ready to be passed to the blocked dialog
long exclusion_dialog::listctrl_add_empty_row() 
{
    long new_inserted_row_index;
    int last_row;
    
    // Get the number of the last row
    last_row = the_blocked_dialog->m_selected_listctrl->GetItemCount();     
    new_inserted_row_index = ( get_selected_listctrl() )->InsertItem( (long) last_row, "", 0 );
    ( get_selected_listctrl() )->SetItem( (long) last_row, ACTION_COLUMN, _T( "Exclude" ) );
    ( get_selected_listctrl() )->SetItem( (long) last_row, PRIORITY_COLUMN, "0" );
    return new_inserted_row_index;
}


// Return which listctrl to insert/edit the new item
wxListCtrl* exclusion_dialog::get_selected_listctrl() 
{
    // m_selected_exclusion_notebookpage_index was set during page change events
    // (The wxNotebook->GetSelected() method doesn't seem to work as in 
    // wx docs, so use this method, as recommended in the /samples/controls                        
    if (  m_selected_exclusion_notebookpage_index == EXCLUSION_EXTENSIONS_PAGE ) {
        return EXCLUSION_EXTENSIONS_LISTCTRL;
    } else {
        return EXCLUSION_URLS_LISTCTRL;
    }                
}


// Returns what row is currently selected
long exclusion_dialog::listctrl_get_selected_item_id()
{   
    long           selected_item_id;  // Will hold the index of the selected row.  
    wxListCtrl*    selected_listctrl; // Listctrl of the selected row
    
    wxLogDebug( "Entering listctrl_get_selected_item_id function" );                                   
    
    selected_listctrl = the_exclusion_dialog->get_selected_listctrl();
    
    selected_item_id = selected_listctrl->GetNextItem( -1, 
                                                       wxLIST_NEXT_ALL,
                                                       wxLIST_STATE_SELECTED );
                                                       
    wxLogDebug( "Value of selected item_id is %ld", selected_item_id );         
    
    // If no rows that fit the flag of being in a selected state, return -1 and stop.
    if ( selected_item_id == -1 ) {
        // TODO: May want to error handle here.
        return -1;
    } else {   
        return selected_item_id;
    }
}

void exclusion_dialog::listctrl_save_rows( wxListCtrl* listctrl )
{
    wxListItem     info;             // To look up the info for the item.
    long           item = -1;        // Item numbers of listctrl grid cells. Getting next
                                     // from row '-1' allows inclusion of first selected row.
    wxString       exclusion_string; // String extracted from exclusion column
    wxString       action_string;    // String extracted from action column
    wxString       priority_string;  // String extracted from priority column
    wxString       final_string;     // The final output string to write to the textfile.
                                    
    wxLogDebug( "Entering save rows function" );   
    
    // Recommended loop for listctrl.                                 
    for ( ;; ) {
        item = listctrl->GetNextItem( item, wxLIST_NEXT_ALL );
        
        wxLogDebug( "Value of item is %ld", item );  
        
        // No more rows in the item, so break loop.
        if ( item == -1 )
            break; 

        // Look up the properties of wxListItem--first thing is to set the id number
        // (m_itemId) to our current item so we know what item we are talking about
        // in the listctrl grid.
        info.m_itemId = item;
        // Set text mask (don't know why that is here, but seems to be always used).
        info.m_mask = wxLIST_MASK_TEXT;

        // Set the column of the cell to look for information, to the exclsuion column
        info.m_col = EXCLUSION_COLUMN;
        // Get the properties of that cell
        listctrl->GetItem( info );
        exclusion_string = info.m_text;
        // Abort this interation of the loop if this exclusion is blank
        if ( exclusion_string == "" ) {
            continue;
        }
        
        // Same strategy for the action column
        info.m_col = ACTION_COLUMN;
        listctrl->GetItem( info );        
        if ( info.m_text == _T( "Include" ) ) {
            action_string = "+";
        } else {
            action_string = "-";
        }

        // And same strategy for the priority column
        info.m_col = PRIORITY_COLUMN;
        listctrl->GetItem( info );
        priority_string = info.m_text;
        
        // Assemble the bits into the final string into the ':' format used
        // as separators in the exclusion list.
        final_string = priority_string + ':' + action_string + ':' + exclusion_string;
        wxLogDebug( "Final_string to append to exclusion file: %s", final_string.c_str() );  
        // Add the final string as a line in the file.
        m_textfile.AddLine( final_string );        
     }
}

void exclusion_dialog::save_exclusion_file()
{
    bool successful = wxRemoveFile( m_exclusion_filename );
    wxLogDebug( "Ability to delete file is %ld", successful );    
    
    successful = m_textfile.Create( m_exclusion_filename );
    wxLogDebug( "Ability to create file is %ld", successful );  
    
    m_textfile.AddLine( "##" );    
    m_textfile.AddLine( "## Leading and trailing white space is ignored." );
    m_textfile.AddLine( "## emty lines or lines starting with a \'#\' are considered comments" );
    m_textfile.AddLine( "## everything else should be of the form:" );
    m_textfile.AddLine( "##  <prio>:<action>:<regexp>" );
    m_textfile.AddLine( "## where:" );
    m_textfile.AddLine( "##  <prio>: is an integer (negative numbers are also valid) specifing" );
    m_textfile.AddLine( "##          the priority.  Rules with higher priorities are considered" );
    m_textfile.AddLine( "##          before rules with lower priorities.  Rules of equal priority" );
    m_textfile.AddLine( "##          are considered in the sequence that they appear in the file." );
    m_textfile.AddLine( "##  <action>: is either a plus or a minus sign.  Plus means \'include this" );
    m_textfile.AddLine( "##            document\' while minus means \'do not include this document\'." );
    m_textfile.AddLine( "##  <regexp>: a valid regular expression (as known, e.g. from perl)." );
    m_textfile.AddLine( "##" );
    m_textfile.AddLine( "" );
    m_textfile.AddLine( "# all rules in this file have priority 0, so you can use easily" );
    m_textfile.AddLine( "# override things by using higher priorities." );
    m_textfile.AddLine( "" );
    m_textfile.AddLine( "## ----------------------------------------------------------------------" );
    m_textfile.AddLine( "## Ignore files that we know we cannot handle:" );
    m_textfile.AddLine( "##" );    
    m_textfile.AddLine( "" );    
    
    // Write the blocked extensions
    the_exclusion_dialog->listctrl_save_rows( EXCLUSION_EXTENSIONS_LISTCTRL );
    
    m_textfile.AddLine( "## ----------------------------------------------------------------------" );   
    m_textfile.AddLine( "## Ignore known advertisement sites:" );
    m_textfile.AddLine( "##" );
    
    // Write the blocked URLs
    the_exclusion_dialog->listctrl_save_rows( EXCLUSION_URLS_LISTCTRL );
    
    m_textfile.AddLine( "## ----------------------------------------------------------------------" );   
   
    // Write the changes. NOTE: Similar to the way the_configuration->Write() works, 
    // m_textfile.AddLine(...) doesn't actually add the lines to the file on disk,
    // until there is a specific flushing called m_textfile.Write(), which flushes
    // all the AddLines to the disk.
    m_textfile.Write();
    
    // Closes the file and frees the memory.
    m_textfile.Close();
}


//----------------------------------------------------------------------------------------
// Event handlers
//----------------------------------------------------------------------------------------

void exclusion_dialog::on_extensions_tab_add_button( wxCommandEvent &event )
{
    long new_item_id;
    
    new_item_id = listctrl_add_empty_row() ;
    init_a_blocked_dialog( this, get_selected_listctrl(), new_item_id );
}


void exclusion_dialog::on_extensions_tab_edit_button( wxCommandEvent &event )
{
    if ( listctrl_get_selected_item_id() != -1 ) {
        init_a_blocked_dialog( this, get_selected_listctrl(), listctrl_get_selected_item_id() );
    }
}


void exclusion_dialog::on_extensions_tab_delete_button( wxCommandEvent &event )
{
    listctrl_delete_selected_row();
}


void exclusion_dialog::on_urls_tab_add_button( wxCommandEvent &event )
{
    long new_item_id;
    
    new_item_id = listctrl_add_empty_row() ;
    init_a_blocked_dialog( this, get_selected_listctrl(), new_item_id );
}


void exclusion_dialog::on_urls_tab_edit_button( wxCommandEvent &event )
{
    if ( listctrl_get_selected_item_id() != -1 ) {
        init_a_blocked_dialog( this, get_selected_listctrl(), listctrl_get_selected_item_id() );
    }
}


void exclusion_dialog::on_urls_tab_delete_button( wxCommandEvent &event )
{
    listctrl_delete_selected_row();
}


void exclusion_dialog::OnPageChanged( wxNotebookEvent &event )
{
    // Gets the index of the notebook page, and sets it to the class variable
    m_selected_exclusion_notebookpage_index = event.GetSelection();
    wxLogDebug( "m_selected_exclusion_notebookpage_index=%ld",
                m_selected_exclusion_notebookpage_index );
}  


// Update the enabled/disabled state of the edit/delete buttons depending on 
// whether a row (item) is selected in the listctrl
void exclusion_dialog::on_update_ui_extensions_tab_listctrl( wxUpdateUIEvent &event )
{
    bool enabled = (bool) EXCLUSION_EXTENSIONS_LISTCTRL->GetSelectedItemCount();

    XMLCTRL( *the_exclusion_dialog, "exclusion_dialog_extensions_tab_edit_button", wxButton )
        ->Enable( enabled );

    XMLCTRL( *the_exclusion_dialog, "exclusion_dialog_extensions_tab_delete_button", wxButton )
        ->Enable( enabled );
}


// Update the enabled/disabled state of the edit/delete buttons depending on 
// whether a row (item) is selected in the listctrl
void exclusion_dialog::on_update_ui_urls_tab_listctrl( wxUpdateUIEvent &event )
{
    bool enabled = (bool) EXCLUSION_URLS_LISTCTRL->GetSelectedItemCount();

    XMLCTRL( *the_exclusion_dialog, "exclusion_dialog_urls_tab_edit_button", wxButton )
        ->Enable( enabled );

    XMLCTRL( *the_exclusion_dialog, "exclusion_dialog_urls_tab_delete_button", wxButton )
        ->Enable( enabled );    
}



// Override wxDialog's default behavior for clicking an OK button.
void exclusion_dialog::OnOK( wxCommandEvent& event )
{
    // Save the file. TODO: Only save if modified.
    if ( m_exclusion_file_is_dirty == TRUE ) {
        the_exclusion_dialog->save_exclusion_file();
    }
    // Get rid of the modal dialog. Not transferring any info from this modal's control
    // to a parent dialog, so don't have to bother with wxWindow::Validate or 
    // wxWindow::TransferDataFromWindow. TODO: could return whether or not the list control
    // needs to be redrawn (since changed channel name or update time), or use a global.
    EndModal( wxID_OK );
}


