//----------------------------------------------------------------------------------------
// Name:        main_dialog.cpp
// Purpose:     Main dialog
// Author:      Robert O'Connor
// Modified by:
// Created:     2001/10/20
// Copyright:   (c) Robert O'Connor ( rob@medicalmnemonics.com )
// Licence:     GPL
// RCS-ID:      $Id: main_dialog.cpp,v 1.13 2002/01/04 03:34:27 robertoconnor Exp $
//----------------------------------------------------------------------------------------

// ---------------------------------------------------------------------------------------
// Headers
// ---------------------------------------------------------------------------------------
#include "setup.h"              // Plucker-Desktop compilation settings:
                                // modify to turn on/off certain code sections 
#ifdef __GNUG__
    #pragma implementation "main_dialog.cpp"
    #pragma interface "main_dialog.cpp"
#endif
// ---------------------------------------------------------------------------------------

// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// For all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWindows headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif
// ---------------------------------------------------------------------------------------
#include "wx/image.h"
#include "wx/listctrl.h"                // wxListCtrl
#include "wx/radiobut.h"                // wxRadioButton
#include "wx/dir.h"                     // wxDir
#if ( USE_MSW_DIALOG_TASKBAR_ICON ) 
    #include "wx/msw/private.h"         // Dialog/taskbar icon for W95
#endif
// ---------------------------------------------------------------------------------------
#include "wx/xrc/xmlres.h"              // XRC XML resouces
// ---------------------------------------------------------------------------------------
#include "main_dialog.h"
#include "configuration.h"
#include "install_dialog.h"
#include "channel_dialog.h"
#include "progress_dialog.h"
#include "exclusion_dialog.h"
#include "splashscreen.h"
#include "startup_tips.h"
#include "utils_datetime.h"
#include "showcase_dialog.h"
#include "utils_string.h"
#include "utils_internet.h"
#include "utils_controls.h"

// ---------------------------------------------------------------------------------------
// Resources
// ---------------------------------------------------------------------------------------

// The application icon
#if defined(__WXGTK__) || defined(__WXMOTIF__)
    #include "resource/icons_application/application_icon.xpm"
#endif

// ---------------------------------------------------------------------------------------
// Private variables
// ---------------------------------------------------------------------------------------

main_dialog* the_main_dialog = NULL;

// ---------------------------------------------------------------------------------------
// Internal constants
// ---------------------------------------------------------------------------------------

// Easier code maintenance if want to add/rearrangement of listctrl's columns.
// SECTION_COLUMN needs to be 0 in current code as GetNextItem only seems to look at 
// the zero column.
enum {
    SECTION_COLUMN      = 0,    // Directory channel is stored in
    NAME_COLUMN,                // Channel's name
    DUE_COLUMN                  // When channel is due
};

// Used in automatically sizing the listctrl's column widths to use full width
// without causing a horizontal scrollbar in the listctrl object window.
#define TOTAL_SPACING_WIDTH     7   // Sum of pixel spacing between all columns

// Custom version of XMLCTRL macro for the list control, since it will be 
// used so much in the code.
#ifdef __WXDEBUG__
#define MAIN_LISTCTRL \
    ( wxDynamicCast( ( *the_main_dialog ).FindWindow( XMLID( "main_dialog_channels_tab_listctrl" ) ), wxListCtrl) )
#else
#define MAIN_LISTCTRL \
    ( ( wxListCtrl* )( ( *the_main_dialog ).FindWindow( XMLID( "main_dialog_channels_tab_listctrl" ) ) ) )
#endif

// ---------------------------------------------------------------------------------------
// Event table: connect the events to the handler functions to process them
// ---------------------------------------------------------------------------------------

BEGIN_EVENT_TABLE(main_dialog, wxDialog)        
    EVT_BUTTON( XMLID( "main_dialog_close_button" ), main_dialog::on_quit )

    EVT_UPDATE_UI( XMLID( "main_dialog_channels_tab_listctrl" ), main_dialog::on_update_ui_channels_tab_listctrl )
    EVT_BUTTON( XMLID( "main_dialog_channels_tab_add_button" ), main_dialog::on_channels_tab_add_button )
    EVT_BUTTON( XMLID( "main_dialog_channels_tab_configure_button" ), main_dialog::on_channels_tab_configure_button )
    EVT_BUTTON( XMLID( "main_dialog_channels_tab_delete_button" ), main_dialog::on_channels_tab_delete_button )
    EVT_BUTTON( XMLID( "main_dialog_channels_tab_update_selected_button" ), main_dialog::on_channels_tab_update_selected_button )
    EVT_BUTTON( XMLID( "main_dialog_channels_tab_update_due_button" ), main_dialog::on_channels_tab_update_due_button )
    EVT_BUTTON( XMLID( "main_dialog_channels_tab_update_all_button" ), main_dialog::on_channels_tab_update_all_button )
    // A double-click or enter key on a listctrl row.
    EVT_LIST_ITEM_ACTIVATED( XMLID( "main_dialog_channels_tab_listctrl" ), main_dialog::on_channels_tab_configure_button ) 
    
    EVT_BUTTON( XMLID( "main_dialog_basic_tab_channel_defaults_button" ), main_dialog::on_basic_tab_channel_defaults_button )
    EVT_UPDATE_UI( XMLID( "main_dialog_basic_tab_use_proxy_checkbox" ), main_dialog::on_update_ui_basic_tab_use_proxy_checkbox )

    EVT_BUTTON( XMLID( "main_dialog_basic_tab_install_software_button" ), main_dialog::on_basic_tab_install_software_button )

    EVT_LISTBOX( XMLID( "main_dialog_advanced_tab_section_listbox" ), main_dialog::on_advanced_tab_section_listbox )
    EVT_UPDATE_UI( XMLID( "main_dialog_advanced_tab_startup_tips_checkbox" ), main_dialog::on_update_ui_advanced_tab_startup_tips_checkbox )
    EVT_UPDATE_UI( XMLID( "main_dialog_advanced_tab_console_radiobutton" ), main_dialog::on_update_ui_advanced_tab_console_radiobutton )
    EVT_UPDATE_UI( XMLID( "main_dialog_advanced_tab_external_editor_radiobutton" ), main_dialog::on_update_ui_advanced_tab_external_editor_radiobutton )
    EVT_BUTTON( XMLID( "main_dialog_advanced_tab_external_editor_button" ), main_dialog::on_advanced_tab_external_editor_button )
    EVT_BUTTON( XMLID( "main_dialog_advanced_tab_download_showcase_listings_button" ), main_dialog::on_advanced_tab_download_showcase_listings_button )
    EVT_BUTTON( XMLID( "main_dialog_advanced_tab_showcase_details_button" ), main_dialog::on_advanced_tab_showcase_details_button )
    EVT_BUTTON( XMLID( "main_dialog_advanced_tab_exclusion_lists_button" ), main_dialog::on_advanced_tab_exclusion_lists_button )
    EVT_BUTTON( XMLID( "main_dialog_advanced_tab_helpers_button" ), main_dialog::on_advanced_tab_helpers_button )

    EVT_UPDATE_UI( XMLID( "main_dialog_autoupdate_tab_after_hours_radiobutton" ), main_dialog::on_update_ui_autoupdate_tab_after_hours_radiobutton )
    
    // We have to specify a EVT_CLOSE in a dialog-only app to correctly terminate
    // the program, when hit the 'X' button (or equivalent) in dialog's top corner.
    EVT_CLOSE( main_dialog::on_quit )
END_EVENT_TABLE()

//----------------------------------------------------------------------------------------
// Non-event handler functions
//----------------------------------------------------------------------------------------

// Initialize an instance of the dialog. No parent window argument on this one (vs. rest 
// of app's dialogs), since the main dialog is a top-level-only dialog, so has no parent
void init_a_main_dialog()
{
    the_main_dialog = new main_dialog; 
      
    wxTheXmlResource->LoadDialog( the_main_dialog, NULL, "main_dialog" );
    
    // Display the splashscreen. Unfortunately, seems to have to be after main_dialog
    // created, so that main dialog will be the top window. TODO: find out why it can't 
    // be in plucker-desktop.cpp main app loadup so that show_splashscreen can go up
    // to before loading up the resources, where it belongs ( problem there is that the
    // Destroy() in the splashscreen destructor takes out the main form too for some reason ). 
    show_splashscreen();
    
    // Set an icon for the dialog
    the_main_dialog->set_icon( wxICON( appicon ) );
    // Show dialog
    the_main_dialog->Show( TRUE );
    // Set up the listctrl
    the_main_dialog->listctrl_init();
    // Initialize (create) the advanced tab's panels
    the_main_dialog->advanced_tab_init();
    // Read and set the saved values for the controls from the configuration file
    the_main_dialog->read_configuration();
            
    // Destroy the splashscreen
    destroy_splashscreen();
    
    // Show startup tips
    show_startup_tips();
}

// Gives an icon to a dialog-only application, as dialogues aren't supposed to have 
// an icon by default. From frame.h, using macros from /msw/private.h 
void main_dialog::set_icon( const wxIcon& icon )
{
m_icon = icon;
#if ( USE_MSW_DIALOG_TASKBAR_ICON ) 
    if ( m_icon.Ok() ) {
        SendMessage( GetHwnd(), WM_SETICON, (WPARAM)TRUE, (LPARAM)(HICON) m_icon.GetHICON() );
    }
#endif
}

void main_dialog::advanced_tab_init()
{
    // Create the advanced_splitterwindow. Use -1 id number, since not need to reference it.
    // Made it 1x1 so not see it when start dialog, when it is on top of channel_tab,
    // since.not yet remapped to the unknown control
    // TODO: Check if CLIP_CHILDREN will work here.
     
    the_advanced_splitterwindow = new wxSplitterWindow( this, -1,
                                                wxDefaultPosition, wxSize( 1, 1 ),
                                                wxSP_NOBORDER /*wxCLIP_CHILDREN*/ );
                                                
    // Attach to unknown control placeholder [see showcase_dialog.cpp for notes on 
    // how these unknown controls are done].
    wxTheXmlResource->AttachUnknownControl( _T( "main_dialog_advanced_tab_splitterwindow" ),
                                            the_advanced_splitterwindow );
    wxLogDebug( "Attached unknown control (main_dialog_advanced_tab_splitterwindow)" );
    
    // Set bordersize of splitterwindow to zero
    XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_splitterwindow", wxSplitterWindow )
        ->SetBorderSize( 0 );
    wxLogDebug( "Set advanced_tab_splitterwindow border to zero" );  
        
    // Get the size of the splitterwindow, so know how big to set the panels.
    the_advanced_splitterwindow_size = XMLCTRL( *the_main_dialog, 
         "main_dialog_advanced_tab_splitterwindow", wxSplitterWindow )->GetSize();          
    wxLogDebug( "Got size of main_dialog_advanced_tab_splitterwindow" );     
        
    // Build and add the panelflip pages to the splitterwindow. If change order,
    // change the enum in the variables above.
    advanced_tab_add_panelflip_page( _T( "General" ),
                                     "main_dialog_advanced_tab_general_panel" );
    
    advanced_tab_add_panelflip_page( _T( "Spidering" ),
                                     "main_dialog_advanced_tab_spidering_panel" );
    
    advanced_tab_add_panelflip_page( _T( "HTML editor" ),
                                     "main_dialog_advanced_tab_editor_panel" );
    
    advanced_tab_add_panelflip_page( _T( "Plucker showcase" ),
                                     "main_dialog_advanced_tab_showcase_panel" ); 
                                     
    // These panels have OS specific options, so only show the one for the user's OS.
#ifdef __WXMSW__
    advanced_tab_add_panelflip_page( _T( "Windows integration" ),
                                     "main_dialog_advanced_tab_os_msw_panel" );
    
    advanced_tab_add_panelflip_page( _T( "Supporting applications" ),
                                     "main_dialog_advanced_tab_applications_msw_panel" );
#endif
#ifdef __WXMAC__
    advanced_tab_add_panelflip_page( _T( "Macintosh integration" ),
                                     "main_dialog_advanced_tab_os_msw_panel" );
    
    advanced_tab_add_panelflip_page( _T( "Supporting applications" ),
                                     "main_dialog_advanced_tab_applications_msw_panel" );
#endif    
#ifdef __WXGTK__
    advanced_tab_add_panelflip_page( _T( "Linux integration" ),
                                     "main_dialog_advanced_tab_os_msw_panel" );

    advanced_tab_add_panelflip_page( _T( "Supporting applications" ),
                                     "main_dialog_advanced_tab_applications_msw_panel" );
#endif

    // Set the splitterwindow to initially display the topmost panel (0), and similarly
    // set the listbox sidebar controller to topmost.    
    XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_splitterwindow", wxSplitterWindow )
        ->Initialize( the_advanced_panelflip_array[ 0 ]->panel );    
        
    the_advanced_panelflip_array[ 0 ]->panel->Show( TRUE );        
    
    XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_section_listbox", wxListBox )
        ->SetSelection( 0 );
    
    the_advanced_splitterwindow_displayed_page_index = 0;
        
    wxLogDebug( "Initialized to the spidering panel" );            
}  

void main_dialog::advanced_tab_add_panelflip_page( wxString title, wxString resource_name )
{
    // Make a new element for the array and load up each bit of it with the proper
    // properties, then add it to the end of the array.
    panelflip_page *new_page    = new panelflip_page();    
    new_page->title             = title; 
    the_advanced_panelflip_array.Add( new_page );
    wxLogDebug( "Added panelflip_page. title=%s, resource_name=%s",
                 title.c_str(), resource_name.c_str() );
    // TODO: Need to delete the *new_page object now that we're done with it?             
    
    // Get the index of the element now.
    int newest_page_index = the_advanced_panelflip_array.GetCount() - 1;    
    wxLogDebug( "panelflip_array: newest_page_index=%i", newest_page_index );
    
    // Create it's panel and load from the resource.
    the_advanced_panelflip_array[ newest_page_index ]->panel = new wxPanel;    
    wxTheXmlResource->LoadPanel( the_advanced_panelflip_array[ newest_page_index ]->panel,
                                 the_advanced_splitterwindow,
                                 resource_name );                                 
    wxLogDebug( "Built a panelflip_panel" );
    
    // Set the panel's size, so that it expands to fill the full size. TODO: can this
    // be replaced with a SetSizer or something so that don't need to write a separate 
    // on size function.   
    the_advanced_panelflip_array[ newest_page_index ]->panel
        ->SetSize( the_advanced_splitterwindow_size );
    wxLogDebug( "Set advanced tab panels' size to parent splitterwinodow size" );
    
    // Initially hide this panel. so won't peek through the active window
    // when clicking away on the active window.
    the_advanced_panelflip_array[ newest_page_index ]->panel->Show( FALSE ); 
    wxLogDebug( "Hid panel" );
    
    // Append the panel's title to the sidebar listbox
    XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_section_listbox", wxListBox )  
        ->Append( title );
    wxLogDebug( "Appended title '%s' to listbox", title.c_str() );    
}

void main_dialog::advanced_tab_set_panelflip_page( int page_index )
{
    // Get a pointer to the old window that is about to be replaced. Using GetWindow1 for
    // splitters with just 1 managed window, returns that only managed window.
    wxWindow* old_window;    
    old_window = the_advanced_splitterwindow->GetWindow1();
    
    // Now swap the old_window with the new window
    the_advanced_splitterwindow->ReplaceWindow( old_window,
                                 the_advanced_panelflip_array[ page_index ]->panel );
                                 
    // Show the new window, hide the old one;
    the_advanced_panelflip_array[ page_index ]->panel->Show( TRUE );
    old_window->Show( FALSE );
    
    the_advanced_splitterwindow_displayed_page_index = page_index;
}

//Initialize the list control. MAIN_LISTCTRL macro is defined above.
void main_dialog::listctrl_init()
{
    int ctrl_width;
    int ctrl_height;
    
    // TODO: Save the column widths to a file. Error correct if they vanish a column
    // by making it too thin.
    MAIN_LISTCTRL->SetSingleStyle( wxLC_REPORT );
    MAIN_LISTCTRL->GetSize( &ctrl_width, &ctrl_height ); 
    
    MAIN_LISTCTRL->InsertColumn( SECTION_COLUMN, _T( "Section" ), wxLIST_FORMAT_LEFT, 0 );
    MAIN_LISTCTRL->InsertColumn( NAME_COLUMN, _T( "Channel name" ), wxLIST_FORMAT_LEFT, ( ctrl_width-100-TOTAL_SPACING_WIDTH ) );
    MAIN_LISTCTRL->InsertColumn( DUE_COLUMN, _T( "Due" ), wxLIST_FORMAT_LEFT, 100 );
    
    // To speed up inserting we hide the control temporarily
    MAIN_LISTCTRL->Hide();

    // Load the channels into the rows of the table
    the_main_dialog->listctrl_load_rows();
    
    // Finished inserting. Show the control
    MAIN_LISTCTRL->Show();    
} 

void main_dialog::listctrl_load_rows()
{
    wxString configuration_section;   // Section of channel in configuration file
    int row_number           = 0;     // List row to insert channel information    
    long index               = 0;     // Index counter for looping through the sections
    bool found_section       = FALSE; // To monitor success of getting first/next group
    wxFile file;                      // To make a home.html if missing.
    
    wxLogDebug( "Starting to load channel sections into listctrl rows" );
    
    // Make a "channels" directory to hold home.html, cache and other future stuff
    // for each entry
    if ( ! wxDirExists( get_plucker_directory( CHANNELS ) ) ) {
        wxMkdir( get_plucker_directory( CHANNELS ), 0777);
    }
    wxLogDebug( "Finished testing/trying to make channel directory" );
    
    // TODO: also make the default channel one, in case in is gone.
    
    found_section = the_configuration->GetFirstGroup( configuration_section,
                                        index );
    while ( found_section ) {
        // Load up the each channel section (group) from the
        // plucker.ini/pluckerrc file
        wxLogDebug( "Configuration section is %s",
            configuration_section.c_str() );
        if ( is_channel_configuration_section( configuration_section ) ) {
            // TODO: check for illegal characters (spaces, etc) in the
            // section name, and rename the section in the config file
            // as necessary before loading it into listctrl.
            
            wxString doc_name;

            doc_name = configuration_section + "/doc_name";
            
            // If no doc_name key, then use the db_name key
            // The db_name key is the depreciated equivalent of doc_name key
            if ( ! the_configuration->Exists( doc_name ) )
                doc_name = configuration_section + "/db_name";
            
            // If no db_name key either, the the doc_file key as the doc_name also
            if ( ! the_configuration->Exists( doc_name ) ) 
                doc_name = configuration_section + "/doc_file";
            
            // The no db_name or doc_file key, then use the db_file key.
            // The db_file key is the depreciated equivalent of doc_file key
            if ( ! the_configuration->Exists( doc_name ) ) 
                doc_name = configuration_section + "/db_file";
                                      
            // Only include channels with an assigned document name
            if ( the_configuration->Exists( doc_name ) ) {
                wxString channel_name;
                wxString due_date;
                wxString home_html;

                // Put the section_name as first zero-width column so
                // know what section of configuration file to call when
                // configure/update the channel.
                MAIN_LISTCTRL->InsertItem( row_number, configuration_section ); 
            
                // Read the current channel name and stuff it into a
                // row. TODO: The 0 in 3rd argument of InsertItem put
                // as the image index for a little glyph later on.
                channel_name = the_configuration->Read( doc_name,
                                                    _T( "Unnamed Channel" ) );
                // If it was a filename key, then strip off path and the 
                // extension, so only basename left
                if ( doc_name == configuration_section + "/doc_file" ||
                     doc_name == configuration_section + "/db_file" ) {
                    wxSplitPath( channel_name.c_str(), NULL, &channel_name, NULL ); 
                }
                MAIN_LISTCTRL->SetItem( row_number, NAME_COLUMN,
                                channel_name, 0 );
            
                // Read the due date and put it in its column.
                // TODO: Put the actual item that will go in here
                // (probably will be a converted date).
                due_date = the_configuration->Read( configuration_section +
                            "/schedule_refresh_interval", _T( "Never" ) ); 
                MAIN_LISTCTRL->SetItem( row_number, DUE_COLUMN, due_date );     
            
                // Make a "channel" and a "channel/cache" directory for
                // each entry
                if (! wxDirExists( get_plucker_directory( CHANNELS ) + '/' +
                        configuration_section ) ) {
                    wxMkdir( get_plucker_directory( CHANNELS ) + '/' +
                        configuration_section, 0777 );
                    wxLogDebug( "Finished testing/trying to make channel-specific directory" );
                    wxMkdir( get_plucker_directory( CHANNELS ) + '/' +
                        configuration_section + "/cache", 0777 );
                    wxLogDebug( "Finished testing/trying to make channel's cache directory" );
                }
                // If no home.html in the channel directory, make one of
                // those too. TODO: Copy from the default home.html instead
                // of making an empty one.
                home_html = get_plucker_directory( CHANNELS ) + '/' +
                            configuration_section + '/' + "home.html";
                if ( ! wxFileExists( home_html ) ) {
                    file.Create( home_html );
                    file.Close();
                    wxLogDebug( "Finished trying to make channel's home.html" );
                }            
            
                // Increase the row number where we will insert new one.            
                row_number++;
            }
        }
        found_section = the_configuration->GetNextGroup( configuration_section,
                                            index );
    }
}

// Get an array of sections names from channel rows currently selected in the listctrl
wxArrayString main_dialog::listctrl_get_selected_channel_sections()
{   
    wxArrayString  channel_sections; // Will hold the array of selected sections to return.  
    wxString       string;          // Temp string for clarity.
    wxListItem     info;            // To look up the info for the item.
    long           item = -1;       // Item numbers of listctrl grid cells. Getting next
                                    // from row '-1' allows inclusion of first selected row.
    
    wxLogDebug( "Entering selected channels function" );                                   
    for ( ;; )
    {
        item = MAIN_LISTCTRL->GetNextItem( item, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED );
        
        wxLogDebug( "Value of item is %ld", item );  
        
        // No more rows that fit the flag of being in a selected state, so break loop.
        if ( item == -1 )
            break; 

        // This item is selected, so add its directory column value to the array.        
        // Look up the properties of wxListItem--first thing is to set the id number
        // (m_itemId) to our current item so we know what item we are talking about
        // in the listctrl grid.
        info.m_itemId = item;
        
        wxLogDebug( "m_itemId is %ld", info.m_itemId );
        wxLogDebug( "m_col is %ld", info.m_col );
        // For some reason, looking up the the m_text returns a blank, but the GetItemText
        // function works properly.
        string = MAIN_LISTCTRL->GetItemText( item );
        wxLogDebug( "GetItemText is %s", string.c_str() );
        // If this selected item has a property of being in the SECTION_COLUMN
        // (as checked by looking at m_col), then we want to add item's text label  
        // (as checked by looking at m_text) to our array.
        if ( info.m_col == SECTION_COLUMN ) {    
            channel_sections.Add( string );
        }
    }    
    return channel_sections;
}

// Read and set the saved values for the controls from the configuration file
void main_dialog::read_configuration()
{
    wxString string;
    int i = 0;
    
    //------Basic Tab-------------------------------------------------------------------
    
    string = the_configuration->Read( "/DEFAULT/http_proxy", _T("") );
    // Check the 'Use proxy' checkbox if there is some values for http_proxy.
    if ( string != "") {
        XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_use_proxy_checkbox", wxCheckBox )
        ->SetValue( TRUE );
    }
    // Want to parse out the http://www.myproxy.com:port into separate boxes for URL and
    // port number, since that is the better GUI approach. 
    // Start by seeing if there is 2 ':' in the address
    if ( string.Freq(':') != 1 ) {
        // Fill the server port with everything before the last ':' (takes care of an http:
        // but also https: case).
        XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_server_textctrl", wxTextCtrl )
            ->SetValue( string.BeforeLast(':') );
        // Port is everything after the second ':'.    
        XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_port_textctrl", wxTextCtrl )
            ->SetValue( string.AfterLast(':') ); 
    // Else just have a http://www.myproxy.com address, or some kind of bad URL.
    // Stick it all into the server box.
    } else {    
        XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_server_textctrl", wxTextCtrl )
            ->SetValue( string );    
    }     
    
    XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_login_textctrl", wxTextCtrl )
        ->SetValue( the_configuration->Read( "/DEFAULT/http_proxy_user", _T("") ) );
    
    XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_password_textctrl", wxTextCtrl )
        ->SetValue( the_configuration->Read( "/DEFAULT/http_proxy_pass", _T("") ) );
    
    wxLogDebug ( "Finished reading basic tab preferences" );
    
   //------Advanced Tab:GENERAL PAGE----------------------------------------------------

    XMLCTRL( *(the_advanced_panelflip_array[ GENERAL_PAGE ]->panel), "main_dialog_advanced_tab_splashscreen_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "/PLUCKER-DESKTOP/show_splashscreen", 1L ) );

    wxLogDebug ( "Finished reading advanced tab general preferences" );

    XMLCTRL( *(the_advanced_panelflip_array[ GENERAL_PAGE ]->panel), "main_dialog_advanced_tab_startup_tips_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "/PLUCKER-DESKTOP/show_startup_tips", 1L ) );

    wxLogDebug ( "Finished reading advanced tab general preferences" );

    XMLCTRL( *(the_advanced_panelflip_array[ GENERAL_PAGE ]->panel), "main_dialog_advanced_tab_startup_tips_choice", wxChoice )
        ->SetSelection( (int) the_configuration->Read( "/PLUCKER-DESKTOP/startup_tips_type", 0L ) );     

    wxLogDebug ( "Finished reading advanced tab general preferences" );

    XMLCTRL( *(the_advanced_panelflip_array[ GENERAL_PAGE ]->panel), "main_dialog_advanced_tab_tooltips_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "/PLUCKER-DESKTOP/show_tooltips", 1L ) );

    wxLogDebug ( "Finished reading advanced tab general preferences" );

    //------Advanced Tab:SPIDERING PAGE---------------------------------------------------
    
    XMLCTRL( *(the_advanced_panelflip_array[ SPIDERING_PAGE ]->panel), "main_dialog_advanced_tab_verbosity_choice", wxChoice )
        ->SetSelection( (int) the_configuration->Read( "/DEFAULT/verbosity", 1L ) );     
    
    wxLogDebug ( "Finished reading advanced tab general preferences" );
  
    if ( (bool) the_configuration->Read( "/PLUCKER-DESKTOP/show_progress_dialog", 1L ) ) {
        XMLCTRL( *(the_advanced_panelflip_array[ SPIDERING_PAGE ]->panel), "main_dialog_advanced_tab_progress_radiobutton", wxRadioButton )
            ->SetValue( TRUE );
    } else {
        XMLCTRL( *(the_advanced_panelflip_array[ SPIDERING_PAGE ]->panel), "main_dialog_advanced_tab_console_radiobutton", wxRadioButton )
            ->SetValue( TRUE );
    }   
    
    wxLogDebug ( "Finished reading advanced tab general preferences" );
    
    XMLCTRL( *(the_advanced_panelflip_array[ SPIDERING_PAGE ]->panel), "main_dialog_advanced_tab_close_console_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "/WINDOWS/close_on_exit", 0L ) );        

    wxLogDebug ( "Finished reading advanced tab spidering preferences" );

    //------Advanced Tab:EDITOR PAGE-----------------------------------------------------
 
    if ( the_configuration->Read( "/PLUCKER-DESKTOP/html_editor", _T("") ) == "" ) {
        XMLCTRL( *(the_advanced_panelflip_array[ EDITOR_PAGE ]->panel), "main_dialog_advanced_tab_builtin_editor_radiobutton", wxRadioButton )
            ->SetValue( TRUE );
    } else {
        XMLCTRL( *(the_advanced_panelflip_array[ EDITOR_PAGE ]->panel), "main_dialog_advanced_tab_external_editor_radiobutton", wxRadioButton )
            ->SetValue( TRUE );
    }    
    XMLCTRL( *(the_advanced_panelflip_array[ EDITOR_PAGE ]->panel), "main_dialog_advanced_tab_external_editor_textctrl", wxTextCtrl )
        ->SetValue( the_configuration->Read( "/PLUCKER-DESKTOP/html_editor", _T("") ) );    
    
    wxLogDebug ( "Finished reading advanced tab general preferences" );
                
    // Basic toolset is choice index 0, Advanced toolset is choice index 1   
    int toolset_type = the_configuration->Read( "/PLUCKER-DESKTOP/editor_use_advanced_toolbars", 0L );
    XMLCTRL( *(the_advanced_panelflip_array[ EDITOR_PAGE ]->panel), "main_dialog_advanced_tab_editor_toolbar_choice", wxChoice )
        ->SetSelection( toolset_type );      
    
    wxLogDebug ( "Finished reading advanced tab general preferences" );
        
    XMLCTRL( *(the_advanced_panelflip_array[ EDITOR_PAGE ]->panel), "main_dialog_advanced_tab_editor_insert_linefeeds_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "/PLUCKER-DESKTOP/editor_tools_insert_linefeeds", 0L ) );

    wxLogDebug ( "Finished reading advanced tab editor preferences" );

    //------Advanced Tab:SHOWCASE PAGE----------------------------------------------
        
    // Horizontal layout is choice index 0, Vertical layout is choice index 1
    int layout = the_configuration->Read( "/PLUCKER-DESKTOP/showcase_layout_type", 1L );     
    XMLCTRL( *(the_advanced_panelflip_array[ SHOWCASE_PAGE ]->panel), "main_dialog_advanced_tab_showcase_layout_choice", wxChoice )
        ->SetSelection( layout );    
    
    wxLogDebug ( "Finished reading advanced tab general preferences" );
    
    // Name/URL only is choice index 0, All fields is choice index 1
    int insert_all_parameters = the_configuration->Read( "/PLUCKER-DESKTOP/showcase_insert_all_parameters", 1L );     
    XMLCTRL( *(the_advanced_panelflip_array[ SHOWCASE_PAGE ]->panel), "main_dialog_advanced_tab_showcase_insert_all_parameters_choice", wxChoice )
        ->SetSelection( insert_all_parameters );
        
    wxLogDebug ( "Finished reading advanced tab general preferences" );

    XMLCTRL( *(the_advanced_panelflip_array[ SHOWCASE_PAGE ]->panel), "main_dialog_advanced_tab_showcase_images_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "/PLUCKER-DESKTOP/showcase_show_images", 0L ) );        

    wxLogDebug ( "Finished reading advanced tab showcase preferences" );

    //------Advanced Tab:OS PAGE---------------------------------------------------  

#ifdef __WXMSW__
    XMLCTRL( *(the_advanced_panelflip_array[ OS_PAGE ]->panel), "main_dialog_advanced_tab_startup_launch_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "/PLUCKER-DAEMON/launch_on_startup", 0L ) );
#endif
#ifdef __WXMAC__

#endif    
#ifdef __WXGTK__

#endif

    wxLogDebug ( "Finished reading advanced tab OS preferences" );
         
    //------Autoupdate Tab--------------------------------------------------------------
 
    i = (long) the_configuration->Read( "/PLUCKER-DAEMON/autoupdate_mode", 1L );
    switch (i) {
        case 0:
            XMLCTRL( *the_main_dialog, "main_dialog_autoupdate_tab_never_radiobutton", wxRadioButton )
                ->SetValue( TRUE );
            break;
        case 1:
            XMLCTRL( *the_main_dialog, "main_dialog_autoupdate_tab_after_hours_radiobutton", wxRadioButton )
                ->SetValue( TRUE );
            break;
        case 2:
            XMLCTRL( *the_main_dialog, "main_dialog_autoupdate_tab_immediately_radiobutton", wxRadioButton )
                ->SetValue( TRUE );
            break;
    }        
 
    wxString autoupdate_deferred_spider_time_string;
    wxDateTime autoupdate_deferred_spider_time;
    autoupdate_deferred_spider_time_string = the_configuration->Read( "/PLUCKER-DAEMON/autoupdate_deferred_spider_time", "19:00:00" );
    autoupdate_deferred_spider_time = rfc_string_to_time( autoupdate_deferred_spider_time_string );
    int choice_index = autoupdate_deferred_spider_time.GetHour();
    XMLCTRL( *the_main_dialog, "main_dialog_autoupdate_tab_after_hours_choice", wxChoice )
        ->SetSelection( choice_index );    
    
    wxLogDebug ( "Finished reading autoupdate preferences" );
}

// Write out the values of the dialog's controls to the configuration file.
void main_dialog::write_configuration()
{
    bool b;
    long lng;
    wxString string;
         
    //------Basic Tab-------------------------------------------------------------------
    
    // If 'use proxy' checkbox is checked, write out the info, else write out blanks
    if ( XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_use_proxy_checkbox", wxCheckBox )
            ->GetValue() == TRUE ) {        
        string = XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_server_textctrl", wxTextCtrl )->GetValue();              
        if ( XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_port_textctrl", wxTextCtrl )->GetValue() != "") {
            string += ( ":" + XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_port_textctrl", wxTextCtrl )->GetValue() );
        }
        the_configuration->Write( "/DEFAULT/http_proxy", string );
                 
        the_configuration->Write( "/DEFAULT/http_proxy_user", 
            XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_login_textctrl", wxTextCtrl )->GetValue() );
           
        the_configuration->Write( "/DEFAULT/http_proxy_pass", 
            XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_password_textctrl", wxTextCtrl )->GetValue() );        
    } else {
        the_configuration->Write( "/DEFAULT/http_proxy", _T("") );
        
        the_configuration->Write( "/DEFAULT/http_proxy_user", _T("") );
        
        the_configuration->Write( "/DEFAULT/http_proxy_pass", _T("") );             
    }

    wxLogDebug ( "Finished writing basic tab preferences" );
    
    //------Advanced Tab:GENERAL PAGE---------------------------------------------------  

    the_configuration->Write( "/PLUCKER-DESKTOP/show_splashscreen",
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ GENERAL_PAGE ]->panel), "main_dialog_advanced_tab_splashscreen_checkbox", wxCheckBox )->GetValue() );
         
    the_configuration->Write( "/PLUCKER-DESKTOP/show_startup_tips",
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ GENERAL_PAGE ]->panel), "main_dialog_advanced_tab_startup_tips_checkbox", wxCheckBox )->GetValue() );

    the_configuration->Write( "/PLUCKER-DESKTOP/startup_tips_type", 
        (long) XMLCTRL( *(the_advanced_panelflip_array[ GENERAL_PAGE ]->panel), "main_dialog_advanced_tab_startup_tips_choice", wxChoice )->GetSelection() );
    
    the_configuration->Write( "/PLUCKER-DESKTOP/show_tooltips",
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ GENERAL_PAGE ]->panel), "main_dialog_advanced_tab_tooltips_checkbox", wxCheckBox )->GetValue() );

    wxLogDebug ( "Finished writing advanced tab general preferences" );
    
    //------Advanced Tab:SPIDERING PAGE-------------------------------------------------
    
    the_configuration->Write( "/DEFAULT/verbosity", 
        (long) XMLCTRL( *(the_advanced_panelflip_array[ SPIDERING_PAGE ]->panel), "main_dialog_advanced_tab_verbosity_choice", wxChoice )->GetSelection() );
    
    the_configuration->Write( "/WINDOWS/close_on_exit", 
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ SPIDERING_PAGE ]->panel), "main_dialog_advanced_tab_close_console_checkbox", wxCheckBox )->GetValue() );

    if ( XMLCTRL( *(the_advanced_panelflip_array[ SPIDERING_PAGE ]->panel), "main_dialog_advanced_tab_progress_radiobutton", wxRadioButton )->GetValue() ) {
        b = 1; 
    } else {
        b = 0;
    }
    the_configuration->Write( "/PLUCKER-DESKTOP/show_progress_dialog", b );

    wxLogDebug ( "Finished writing advanced tab spidering preferences" );
    
    //------Advanced Tab:EDITOR PAGE---------------------------------------------------

    if ( XMLCTRL( *(the_advanced_panelflip_array[ EDITOR_PAGE ]->panel), "main_dialog_advanced_tab_builtin_editor_radiobutton", wxRadioButton )
            ->GetValue() == TRUE ) {
        the_configuration->Write( "/PLUCKER-DESKTOP/html_editor", _T("") );   
    } else {
        the_configuration->Write ( "/PLUCKER-DESKTOP/html_editor",
            XMLCTRL( *(the_advanced_panelflip_array[ EDITOR_PAGE ]->panel), "main_dialog_advanced_tab_external_editor_textctrl", wxTextCtrl )->GetValue() );   
    }
    
    // Basic toolset is choice index 0, advancedis choice index 1, so just cast to boolean directly.
    the_configuration->Write( "/PLUCKER-DESKTOP/editor_use_advanced_toolbars", 
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ EDITOR_PAGE ]->panel), "main_dialog_advanced_tab_editor_toolbar_choice", wxChoice )->GetSelection() );     
    
    the_configuration->Write( "/PLUCKER-DESKTOP/editor_tools_insert_linefeeds",
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ EDITOR_PAGE ]->panel), "main_dialog_advanced_tab_editor_insert_linefeeds_checkbox", wxCheckBox )->GetValue() );

    wxLogDebug ( "Finished writing advanced tab editor preferences" );
  
    //------Advanced Tab:SHOWCASE PAGE----------------------------------------------
        
    // Horizontal layout is choice index 0, vertical layout is index 1, so just cast to boolean directly.
    the_configuration->Write( "/PLUCKER-DESKTOP/showcase_layout_type", 
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ SHOWCASE_PAGE ]->panel), "main_dialog_advanced_tab_showcase_layout_choice", wxChoice )->GetSelection() );     
   
    // Name/URL only is choice index 0, all parameters is index 1, so just cast to boolean directly.
    the_configuration->Write( "/PLUCKER-DESKTOP/showcase_insert_all_parameters", 
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ SHOWCASE_PAGE ]->panel), "main_dialog_advanced_tab_showcase_insert_all_parameters_choice", wxChoice )->GetSelection() );     
 
    the_configuration->Write( "/PLUCKER-DESKTOP/showcase_show_images",
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ SHOWCASE_PAGE ]->panel), "main_dialog_advanced_tab_showcase_images_checkbox", wxCheckBox )->GetValue() );

    wxLogDebug ( "Finished writing advanced tab showcase preferences" );
  
    //------Advanced Tab:OS PAGE---------------------------------------------------  

#ifdef __WXMSW__
    the_configuration->Write( "/PLUCKER-DAEMON/launch_on_startup",
        (bool) XMLCTRL( *(the_advanced_panelflip_array[ OS_PAGE ]->panel), "main_dialog_advanced_tab_startup_launch_checkbox", wxCheckBox )->GetValue() );
#endif
#ifdef __WXMAC__

#endif    
#ifdef __WXGTK__

#endif

    wxLogDebug ( "Finished writing advanced tab OS preferences" );
  
    //------Autoupdate Tab--------------------------------------------------------------
    
    if ( XMLCTRL( *the_main_dialog, "main_dialog_autoupdate_tab_never_radiobutton", wxRadioButton )->GetValue() )
       lng = 0;
    else if ( XMLCTRL( *the_main_dialog, "main_dialog_autoupdate_tab_after_hours_radiobutton", wxRadioButton )->GetValue() )
       lng = 1;
    else  
       lng = 2; 
    the_configuration->Write( "/PLUCKER-DAEMON/autoupdate_mode", lng );       
    
    int choice_index;   
    wxDateTime::wxDateTime_t autoupdate_deferred_spider_hour;
    // Bizarre: Cygwin coughs unless initialize this to Now();
    wxDateTime autoupdate_deferred_spider_time = wxDateTime::Now();
    wxString autoupdate_deferred_spider_time_string;
    choice_index = XMLCTRL( *the_main_dialog, "main_dialog_autoupdate_tab_after_hours_choice", wxChoice )
                      ->GetSelection();                      
    autoupdate_deferred_spider_hour = (wxDateTime::wxDateTime_t) choice_index;
    // Sets hours, minutes, seconds to zero
    autoupdate_deferred_spider_time.ResetTime();
    autoupdate_deferred_spider_time_string = time_to_rfc_string( autoupdate_deferred_spider_time );
    the_configuration->Write( "/PLUCKER-DAEMON/autoupdate_deferred_spider_time", autoupdate_deferred_spider_time_string );
    
    wxLogDebug ( "Finished writing autoupdate preferences" );        
}

// Shut down the dialog-only application
void main_dialog::on_quit( wxCommandEvent& WXUNUSED( event ) )
{
    // Write out the values of the dialog's controls to the configuration file.
    the_main_dialog->write_configuration();
    wxLogDebug ( "Wrote configuration" );
    // Shut down the configuration object, so not memory leak
    finish_configuration();
    wxLogDebug ( "Destroyed configuration object" );
    // Use Destroy() instead of Close() on the dialog, since unlike normal with a
    // wxDialog, we need to destroy, not just hide.
    Destroy();
    wxLogDebug ( "Destroyed main dialog. Successful program termination!" );
}

//----------------------------------------------------------------------------------------
// Event handlers
//----------------------------------------------------------------------------------------
void main_dialog::on_update_ui_channels_tab_listctrl( wxUpdateUIEvent &event )
{
    bool enabled = (bool) MAIN_LISTCTRL->GetSelectedItemCount();

    XMLCTRL( *the_main_dialog, "main_dialog_channels_tab_configure_button", wxButton )
        ->Enable( enabled );

    XMLCTRL( *the_main_dialog, "main_dialog_channels_tab_delete_button", wxButton )
        ->Enable( enabled );
        
    XMLCTRL( *the_main_dialog, "main_dialog_channels_tab_update_selected_button", wxButton )
        ->Enable( enabled );
} 
 
void main_dialog::on_channels_tab_add_button( wxCommandEvent &event )
{
    // First write the updated main_dialog settings to the configuration, since want the most 
    // recent changes to be used (such as showcase layout, what type of editor toolbar to use, etc).
    the_main_dialog->write_configuration();}

// Note: also used on the list control double-click event
void main_dialog::on_channels_tab_configure_button( wxCommandEvent &event )
{    
    // First write the updated main_dialog settings to the configuration, since want the most 
    // recent changes to be used (such as showcase layout, what type of editor toolbar to use, etc).
    the_main_dialog->write_configuration();

    // Make a list of the directory columns that are currently selected
    wxArrayString channel_sections = the_main_dialog->listctrl_get_selected_channel_sections();    
    
    // If array isn't empty, boot up the channel dialog telling it what directory want
    if ( channel_sections.GetCount() != 0 ) {
        init_a_channel_dialog( this, channel_sections.Item( 0 ) );
    }
}

void main_dialog::on_channels_tab_delete_button( wxCommandEvent &event )
{
    wxMessageDialog dlg( this,
                         _T( "Are you sure you wish to permanently delete the selected channel?" ),
                         _T( "Confirm channel delete" ),
                         wxYES_NO|wxYES_DEFAULT|wxICON_QUESTION );
    dlg.ShowModal();
}

void  main_dialog::on_channels_tab_update_selected_button( wxCommandEvent &event )
{
    // First write the updated main_dialog settings to the configuration, since want the most 
    // recent changes to be used (such as proxy settings, spidering status verbosity, etc).
    the_main_dialog->write_configuration();

    // Make a list of the directory columns that are currently selected
    wxArrayString channel_sections = the_main_dialog->listctrl_get_selected_channel_sections();    
    wxString string;
    
    // If array isn't empty, update it, telling it what section we want    
    // TODO: remap to progress dialog if specified, proper location of plucker-build.
    if ( channel_sections.GetCount() != 0 ) {
        // Execute plucker-build in a shell window, passing it the channel section of 
        // the configuration file (to look for to override any similar settings in the 
        // DEFAULT section) .
        string << "plucker-build" << " -s " << channel_sections.Item( 0 );
        wxShell( string );
    }
}

void  main_dialog::on_channels_tab_update_due_button( wxCommandEvent &event )
{
    // First write the updated main_dialog settings to the configuration, since want the most 
    // recent changes to be used (such as proxy settings, spidering status verbosity, etc).
    the_main_dialog->write_configuration();
    
    init_a_progress_dialog( this );
}

void  main_dialog::on_channels_tab_update_all_button( wxCommandEvent &event )
{
    // First write the updated main_dialog settings to the configuration, since want the most 
    // recent changes to be used (such as proxy settings, spidering status verbosity, etc).
    the_main_dialog->write_configuration();
   
    init_a_progress_dialog( this );
}

void main_dialog::on_basic_tab_install_software_button( wxCommandEvent &event )
{
    // First write the updated main_dialog settings to the configuration, since want the most 
    // recent changes to be used (such as proxy settings, spidering status verbosity, etc).
    the_main_dialog->write_configuration();
   
    init_a_install_dialog( this );
}

void main_dialog::on_update_ui_basic_tab_use_proxy_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_use_proxy_checkbox", wxCheckBox )
            ->GetValue();
    XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_server_textctrl", wxTextCtrl )
            ->Enable( enabled );
    XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_port_textctrl", wxTextCtrl )
            ->Enable( enabled );
    XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_login_textctrl", wxTextCtrl )
            ->Enable( enabled );
    XMLCTRL( *the_main_dialog, "main_dialog_basic_tab_proxy_password_textctrl", wxTextCtrl )
            ->Enable( enabled );      
}

void main_dialog::on_advanced_tab_section_listbox( wxCommandEvent &event )
{
    // Get the index of the selected page in the listbox
    int page_index;     
    page_index = XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_section_listbox", wxListBox )
                    ->GetSelection();
                    
    // Don't switch pages, if the item clicked is the one that is already selected  
    // (since it will cause it to switch to a blank panel ).
    if ( page_index != the_advanced_splitterwindow_displayed_page_index ) {
        // It is a new page. Execute the panelflip widget's setpage function.
        advanced_tab_set_panelflip_page( page_index );
    }
}

void main_dialog::on_update_ui_advanced_tab_startup_tips_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_startup_tips_checkbox", wxCheckBox )
            ->GetValue();
    XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_startup_tips_choice", wxChoice )
            ->Enable( enabled );
}

void main_dialog::on_update_ui_advanced_tab_external_editor_radiobutton( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_external_editor_radiobutton", wxRadioButton )
            ->GetValue();
    XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_external_editor_textctrl", wxTextCtrl )
            ->Enable( enabled );
    XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_external_editor_button", wxButton )
            ->Enable( enabled );
}


void main_dialog::on_update_ui_advanced_tab_console_radiobutton( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_console_radiobutton", wxRadioButton )
            ->GetValue();
    XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_close_console_checkbox", wxCheckBox )
            ->Enable( enabled );
}


void main_dialog::on_advanced_tab_external_editor_button( wxCommandEvent &event )
{
    // [utils_controls.cpp]    
    textctrl_filename_selection( XMLCTRL( *the_main_dialog, "main_dialog_advanced_tab_external_editor_textctrl", wxTextCtrl ),
                                 _T( "Choose external editor" ),
                                 "",
                                 "",
                                 FILE_DIALOG_WILDCARDS_EXECUTABLES,
                                 false,
                                 ';' ); 
}


void main_dialog::on_basic_tab_channel_defaults_button( wxCommandEvent &event )
{
    // Make a list of the directory columns that are currently selected
    wxArrayString channel_sections = the_main_dialog->listctrl_get_selected_channel_sections();
    
    // Boot up the channel dialog, but this time telling section is [DEFAULT] , not a channel directory
    init_a_channel_dialog( this, "DEFAULT" );
}


void main_dialog::on_advanced_tab_download_showcase_listings_button( wxCommandEvent &event )
{

    
    // Allow user to select a choice. Make the array of listbox choices
    wxArrayString showcase_listing_urls_array;
    wxString url;
    wxString showcase_listing_urls_string = the_configuration->Read( "/PLUCKER-DESKTOP/showcase_listings_urls",
             "http://www.afflatus.com/rob.md/projects/plucker/showcase/plucker_showcase.htm" );    
    // Converts the read string to an array of strings.
    string_to_arraystring( showcase_listing_urls_string, ';', &showcase_listing_urls_array, true );
    
    // Show the single choice dialog to get which url to download
    int selected_index; 
    selected_index = wxGetSingleChoiceIndex( _T( "Download Plucker showcase listings from:" ),
                                             _T( "Select download URL" ),
                                             showcase_listing_urls_array,
                                             this );    
    // If hit cancel button (-1) then abort here, so not overwrite old one.
    // Else get the url from the selected item in index and keep going.
    if ( selected_index == -1 ) {
        return;
    } else {
        wxString url = showcase_listing_urls_array.Item( (size_t) selected_index );
    }
    
    // Generate a local file
    wxString local_filename;    
    wxFile downloaded_file;    
    local_filename = get_plucker_directory( RESOURCES ) << "/plucker_showcase.htm";
    // If plucker_showcase.htm got deleted, will also replace it with a new one.
    downloaded_file.Create( local_filename, TRUE, wxS_DEFAULT );
    // Have to set the mode as at least write, or else can't write to it.
    downloaded_file.Open( local_filename, wxFile::read_write );
    
    // Call function to download file from remote url [see utils_internet.cpp]
    bool downloaded_successfully;
    downloaded_successfully = download_url_to_local_file( url, local_filename );
        
    // Let user know if was a problem and then abort
    if ( ! downloaded_successfully ) {
        wxLogMessage( _T( "Unable to download showcase listings to local file." ) );
    } else {
        // Replace the old textfile with the new one
        wxMessageBox( _T( "Successfully downloaded new showcase listings." ) );
    }    
    downloaded_file.Close();
}


void main_dialog::on_advanced_tab_showcase_details_button( wxCommandEvent &event )
{   
    // Make the array of listbox indexes to have selected
    wxArrayInt selected_index_numbers;
    wxString showcase_details_fields_string = the_configuration->Read( "/PLUCKER-DESKTOP/showcase_dialog_details_fields", "7" );    
     
    // Converts the read string to an array of integers.
    string_to_arrayint( showcase_details_fields_string, ';', &selected_index_numbers, true );
  
    // Show the dialog 
    size_t number_of_selected_items = wxGetMultipleChoices( selected_index_numbers,
                                        _T( "Fields to display in details pane:" ),
                                        _T( "Select desired fields" ),
                                        WXSIZEOF( showcase_field_names ), showcase_field_names,
                                        this );
    if ( number_of_selected_items != 0 )  {
        wxString details_pane_configuration_string;
        for ( size_t n = 0; n < number_of_selected_items; n++ ) {
            details_pane_configuration_string += wxString::Format( wxT( "%u;" ), selected_index_numbers[ n ] );
        }
        // Strips off the trailing ';'
        details_pane_configuration_string.RemoveLast();
        wxLogDebug( "Wrote out showcase_dialog_details_fields to config as=" + details_pane_configuration_string );
        the_configuration->Write( "/PLUCKER-DESKTOP/showcase_dialog_details_fields", details_pane_configuration_string );
        the_configuration->Flush();
    }
}


void main_dialog::on_advanced_tab_exclusion_lists_button( wxCommandEvent &event )
{
    init_a_exclusion_dialog( this, "DEFAULT" );
}


void main_dialog::on_advanced_tab_helpers_button( wxCommandEvent &event )
{
}


void main_dialog::on_update_ui_autoupdate_tab_after_hours_radiobutton( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_main_dialog, "main_dialog_autoupdate_tab_after_hours_radiobutton", wxRadioButton )
            ->GetValue();
    XMLCTRL( *the_main_dialog, "main_dialog_autoupdate_tab_after_hours_choice", wxChoice )
            ->Enable( enabled );
}
