//----------------------------------------------------------------------------------------
// Name:        utils_controls.cpp
// Purpose:     Utils for wx Controls
// Author:      Robert O'Connor
// Modified by:
// Created:     2001/10/20
// Copyright:   (c) Robert O'Connor ( rob@medicalmnemonics.com )
// Licence:     GPL
// RCS-ID:      $Id: utils_controls.cpp,v 1.2 2002/01/04 03:34:27 robertoconnor Exp $
//----------------------------------------------------------------------------------------

// ---------------------------------------------------------------------------------------
// Headers
// ---------------------------------------------------------------------------------------

#ifdef __GNUG__
    #pragma implementation "utils_controls.cpp"
    #pragma interface "utils_controls.cpp"
#endif
// ---------------------------------------------------------------------------------------
// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// For all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWindows headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif
// ---------------------------------------------------------------------------------------
#include "wx/colordlg.h"            // wxColorDialog
#include "wx/filedlg.h"             // wxFileDialog
// ---------------------------------------------------------------------------------------
#include "utils_controls.h"
#include "utils_string.h"

//----------------------------------------------------------------------------------------
// Non-event handler functions
//----------------------------------------------------------------------------------------

// Fixes a a wxSplitterWindow sash position that is is such that it exceeds a
// minimum pane width of the right/bottom window, or is in fact set beyond the
// wxSplitterWindow dimension. If so, it is out of range, it backtracks the 
// sash it to the minimum pane width. Returns whether or not it needed 
// to be fixed.
bool splitterwindow_fix_sash_out_of_range( wxSplitterWindow* splitterwindow )
{   
    // If its not split, impossible to be out of range, so abort
    if ( ! splitterwindow->IsSplit() ) {
        wxLogDebug( "Window not split, so sash fix aborting." );
        return FALSE;
    }   
  
    int ctrl_width;
    int ctrl_height;    
    splitterwindow->GetSize( &ctrl_width, &ctrl_height ); 
 
    int split_mode = splitterwindow->GetSplitMode();
    
    int window_maximum;
    if ( split_mode == wxSPLIT_VERTICAL ) {
        window_maximum = ctrl_width;
    } else {
        window_maximum = ctrl_height;
    }
    
    int sash_position = splitterwindow->GetSashPosition();
    // Minimum pane size is zero if was never set.
    int minimum_pane_size = splitterwindow->GetMinimumPaneSize();
    // Check to see if the sash position is legal, ie, within the minimum pane size.    
    if ( sash_position < ( window_maximum - minimum_pane_size ) ) {
        // It's okay, so abort and return FALSE.
        wxLogDebug( "Sash within allowable parameters, so sash fix aborting." );
        return FALSE;
    } else {
        // Ilegal, so set the sash back so its within a minimum pane width.
        // TRUE redraws the sash.
        splitterwindow->SetSashPosition( ( window_maximum - minimum_pane_size ), TRUE );
        wxLogDebug( "Sash exceeds allowable parameters, so fixing sash." );
        // Return TRUE, since it needed a fix.
        return TRUE;
    }
}


// When a "Choose color..." button beside a hex color textctrl is clicked, show
// a color dialog, initialize to textctrl's previous value if possible, get the
// color, convert to hex, put it back in the textctrl. Return false if there
// was a problem.
bool textctrl_hex_color_selection ( wxTextCtrl* target_color_textctrl )
{
    wxColour col;
    wxColourData color_data;
    
    // Set the default selected color to the hex one already in the textbox, 
    // if possible:
    // 1. Extract the old hex value
    wxString old_hex_color_string = target_color_textctrl->GetValue();    
    // 2. Check to see if valid [utils_string.cpp]
    if ( is_valid_hex_color_string( old_hex_color_string ) ) {
        wxLogDebug( "Valid old_hex_color_string=" + old_hex_color_string );
        // Convert textboxes previous color from its hex value [utils_string.cpp]
        col = hex_string_to_color( old_hex_color_string );
    } else {
        // MSW/Cygwin (at least) needs to have an initialization to something other
        // than black in order to get the matrix part of the dialog going properly.
        col = wxColour( "MEDIUM BLUE" );
    }
    // Put that value in color_data for the initial color of the color dialog.        
    color_data.SetColour( col );
   
    // Tells MSW to show the full color dialog, not just the smaller one. No
    // effect on other platforms.
    color_data.SetChooseFull( TRUE );

    // Show the color dialog, using the parent of the textctrl as parent of the 
    // new color dialog
    wxColourDialog color_dialog( target_color_textctrl->GetParent(), &color_data );
    // Set the title
    color_dialog.SetTitle( _T( "Choose the colour" ) );
    // Only do the action if returned hitting an OK button.
    if ( color_dialog.ShowModal() == wxID_OK ) {
        wxColourData returned_color_data = color_dialog.GetColourData();
        col = returned_color_data.GetColour();
        // Convert to hex [utils_string.cpp]
        wxString output_hex_color_string = color_to_hex_string( col );
        // Set the textbox to that hex value
        target_color_textctrl->SetValue( output_hex_color_string );
    }
    return true;
}


// When a "Choose filename..." button beside a filename textctrl is clicked, show
// a filename dialog, initialize to textctrl's previous value if possible, get the
// filename, convert to hex, put it back in the textctrl. Return false if there
// was a problem.
bool textctrl_filename_selection 
    ( 
    wxTextCtrl* target_filename_textctrl,       // TextCtrl to read/write filename(s)
    wxString message,                           // Message for the textctrl
    wxString default_file,                      // Default filename
    wxString default_directory,                 // Default directory
    wxString wildcard,                          // Wildcard filters
    bool allow_multiple,                        // Allow multiple selections
    wxChar multiple_selection_delimiter         // Delimiter in textctrl to use to 
                                                // separate multiple files (usu ;)
    )
{
    // Extract the old filename value to override default file and default directory
    wxString old_filename_string = target_filename_textctrl->GetValue();   
    if ( allow_multiple == true ) {
        // Can only set the first entry to set dir/file.
        old_filename_string = old_filename_string.BeforeFirst ( multiple_selection_delimiter );
    }
    
    wxString old_directory;
    wxString old_base;
    wxString old_extension;
    
    // Must check for non-null string before calling wxSplitPath
    if ( old_filename_string != "" ) {
        wxSplitPath( old_filename_string, &old_directory, &old_base, &old_extension );
    }
    
    // If there was a directory part, make that default directory
    if ( old_directory != "" ) {
        default_directory = old_directory;
    }
    
    // If was a base, make that default filename
    if ( old_base != "" ) {
        default_file = old_base;
        // If was an extension, add a '.' and the extension to filename
        if ( old_extension != "" ) {
            default_file += '.' + old_extension;
        } 
    }

    wxFileDialog file_dialog( target_filename_textctrl->GetParent(), 
                              message,
                              default_directory,
                              default_file,
                              wildcard );

    // Set multiple style if multiple
    if ( allow_multiple == true ) {
        file_dialog.SetStyle ( wxMULTIPLE );
    }    
                                  
    wxString output_string;
    // Show the filename dialog modally, and only do the action if returned
    // hitting an OK button.
    if ( file_dialog.ShowModal() == wxID_OK ) {
        if ( allow_multiple == true ) {
            // Get the array of selections, and loop through writing to output_string 
            // and a delimiter at the end of each.
            wxArrayString selected_paths;
            file_dialog.GetPaths( selected_paths );
            for ( size_t i=0; i < selected_paths.Count(); i++ ) {
                output_string += selected_paths.Item( i ) + multiple_selection_delimiter;
            }
            // Remove the last delimiter
            output_string.RemoveLast();       
        } else {
            output_string = file_dialog.GetPath();
        }
        target_filename_textctrl->SetValue( output_string );
    }
    return true;
}