//----------------------------------------------------------------------------------------
// Name:        utils_datetime.cpp
// Purpose:     Utils for date and time
// Author:      Robert O'Connor
// Modified by:
// Created:     2001/12/28
// Copyright:   (c) Robert O'Connor ( rob@medicalmnemonics.com )
// Licence:     GPL
// RCS-ID:      $Id: utils_datetime.cpp,v 1.3 2002/01/02 01:20:39 robertoconnor Exp $
//----------------------------------------------------------------------------------------

// Many things in time in Cygwin won't work, when parsing a date. It adds or subtracts hours 
// for timezones when parsing a date, and most attempts to stop this won't work: 
// 
// output_datetime.ParseDateTime( output_datetime_string );
// wxLogDebug( "output_datetime parsed by ParseDateTime is=%s", output_datetime.Format().c_str() );
// output_datetime.ParseRfc822Date( output_datetime_string );
// wxLogDebug( "output_datetime parsed by ParseRfc822Date is=%s", output_datetime.Format().c_str() );
// 
// Tm objects won't work either, in correcting for timezone: 
// wxDateTime::Tm tm_representation;
// tm_representation = output_datetime.GetTm( 13 );
// wxDateTime final_date_time( tm_representation );
// wxLogDebug( "final_date_time is=%s", final_date_time.Format().c_str() );
// 
// MakeGMT() does nothing and ToGMT() makes a screwy date
// output_datetime.MakeGMT( FALSE );
// wxLogDebug( "Cygwin adjustment: after MakeGMT(), output_datetime=%s", output_datetime.Format().c_str() );
// output_datetime.ToGMT( FALSE );
// wxLogDebug( "Cygwin adjustment: after ToGMT(), output_datetime=%s", output_datetime.Format().c_str() );
// 
// Cygwin can extract the timezone with this, but the answer is difficult to understand:
//    tzset();
//    wxLogDebug( "_timezone: %d", _timezone );  


// ---------------------------------------------------------------------------------------
// Headers
// ---------------------------------------------------------------------------------------
#include "setup.h"              // Plucker-Desktop compilation settings:
                                // modify to turn on/off certain code sections 
// ---------------------------------------------------------------------------------------
#ifdef __GNUG__
    #pragma implementation "utils_datetime.cpp"
    #pragma interface "utils_datetime.cpp"
#endif
// ---------------------------------------------------------------------------------------
// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// For all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWindows headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif
// ---------------------------------------------------------------------------------------
#include "utils_datetime.h"

//----------------------------------------------------------------------------------------
// Non-event handler functions
//----------------------------------------------------------------------------------------

// Converts a "Sat, 30 Jun 2001 10:30:00" string to a wxDateTime
wxDateTime rfc_string_to_datetime( wxString input_rfc_string )
{
    wxLogDebug( "RFC string to convert to datetime is" + input_rfc_string );
    
    wxDateTime parsed_datetime;
    wxDateTime output_datetime;
   
    parsed_datetime.ParseFormat( input_rfc_string, _T( "%a, %d %b %Y %H:%M:%S" ) );
    wxLogDebug( "parsed_datetime as parsed by ParseFormat=%s",  parsed_datetime.Format().c_str() );
    
#if ( USE_CYGWIN_GMT_ADJUSTMENT ) 
    wxTimeSpan timezone_offset = get_timezone_offset( parsed_datetime );
    output_datetime = parsed_datetime;
    output_datetime += ( timezone_offset );  
    wxLogDebug( "output_datetime after timezone_offset=%s", output_datetime.Format().c_str() );
#else
    output_datetime = parsed_datetime;
#endif

    return output_datetime;
}


// Takes a wxDateTime string and converts it to a an RFC datetime
wxString datetime_to_rfc_string( wxDateTime input_datetime )
{
    wxLogDebug( "DateTime to convert to RFC string=%s", input_datetime.Format().c_str() );
    
#if ( USE_CYGWIN_GMT_ADJUSTMENT ) 
    wxTimeSpan timezone_offset = get_timezone_offset( input_datetime );
    input_datetime += ( timezone_offset );  
    wxLogDebug( "input_datetime after timezone_offset=%s", input_datetime.Format().c_str() );
#endif

    wxString output_datetime_string;
    output_datetime_string = input_datetime.Format( _T( "%a, %d %b %Y %H:%M:%S" ) ).c_str();
    wxLogDebug( "output_datetime_string as formatted by Format=%s",  input_datetime.Format().c_str() );

    return output_datetime_string;
}


// Converts a RFC "10:30:00" time string to a wxDateTime
wxDateTime rfc_string_to_time( wxString input_rfc_string )
{
    wxLogDebug( "RFC string to convert to datetime is" + input_rfc_string );
    
    wxDateTime parsed_time;
    wxDateTime output_time;
   
    //parsed_time.ParseFormat( input_rfc_string, _T( "%H:%M:%S" ) );
    parsed_time.ParseTime( input_rfc_string );
    wxLogDebug( "parsed_time as parsed by ParseFormat=%s",  parsed_time.Format().c_str() );
    
#if ( USE_CYGWIN_GMT_ADJUSTMENT ) 
    wxTimeSpan timezone_offset = get_timezone_offset( parsed_time );
    output_time = parsed_time;
    output_time += ( timezone_offset );  
    wxLogDebug( "output_time after timezone_offset=%s", output_time.Format().c_str() );
#else
    output_time = parsed_time;
#endif

    return output_time;
}


// Takes a wxDateTime string and converts it to a an RFC time
wxString time_to_rfc_string( wxDateTime input_time )
{
    wxLogDebug( "Time to convert to RFC string=%s", input_time.Format().c_str() );
    
#if ( USE_CYGWIN_GMT_ADJUSTMENT ) 
    wxTimeSpan timezone_offset = get_timezone_offset( input_time );
    input_time += ( timezone_offset );  
    wxLogDebug( "input_time after timezone_offset=%s", input_time.Format().c_str() );
#endif

    wxString output_time_string;
    output_time_string = input_time.Format( _T( "%H:%M:%S" ) ).c_str();
    wxLogDebug( "output_time_string as formatted by Format=%s",  input_time.Format().c_str() );

    return output_time_string;
}


// Converts a datetime to the timezone converted datetime.
wxDateTime to_timezone_corrected( wxDateTime input_datetime )
{
 wxDateTime output_datetime = input_datetime;

#if ( USE_CYGWIN_GMT_ADJUSTMENT ) 
    wxTimeSpan timezone_offset;
    timezone_offset = get_timezone_offset( input_datetime );
    output_datetime += ( timezone_offset );  
#endif

    return output_datetime;
}


// Returns the hours/minutes offset from GMT of the local computer.
wxTimeSpan get_timezone_offset( wxDateTime input_datetime )
{
#if ( USE_CYGWIN_GMT_ADJUSTMENT ) 
    wxLogDebug( "Parsed datetime is=%s", input_datetime.Format().c_str() );
    
    wxString parsed_string;
    wxDateTime second_input_datetime;
    wxTimeSpan timezone_offset; 

    parsed_string = input_datetime.Format( _T( "%a, %d %b %Y %H:%M:%S" ) ).c_str();
    wxLogDebug( "parsed_string=" + parsed_string );
    second_input_datetime.ParseFormat( parsed_string, _T( "%a, %d %b %Y %H:%M:%S" ) );
    wxLogDebug( "second_input_datetime=%s", second_input_datetime.Format().c_str() );
    
    timezone_offset = ( input_datetime - second_input_datetime );

    if ( input_datetime.IsDST() ) {
        wxTimeSpan dst_hour_offset( 1,0,0,0 );
        // TODO: only apply if in DST and today isnt DST?
        timezone_offset -= dst_hour_offset;
    }
    
    return timezone_offset;
#else
    return ( 0,0,0,0 );
#endif    
}


// If don't care about what day it was time, can use this
wxTimeSpan get_timezone_offset()
{
#if ( USE_CYGWIN_GMT_ADJUSTMENT ) 
    wxDateTime current_datetime;
    current_datetime = wxDateTime::Now();
    wxTimeSpan timezone_offset;
    timezone_offset = get_timezone_offset( current_datetime );
    return timezone_offset;
#else
    return ( 0,0,0,0 );
#endif    
}


wxDateTime get_current_datetime()
{ 
    wxDateTime current_datetime = wxDateTime::Now();
    wxLogDebug( "Now is=%s", current_datetime.Format().c_str() );

#if ( USE_CYGWIN_GMT_ADJUSTMENT ) 
    wxTimeSpan timezone_offset = get_timezone_offset();
    current_datetime += ( timezone_offset );
#endif
    return current_datetime;
}


