/*
 * $Id: categorydata.c,v 1.18 2001/11/26 20:09:01 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "categorydata.h"
#include "const.h"
#include "db.h"
#include "debug.h"
#include "documentdata.h"
#include "resourceids.h"
#include "util.h"
#include "prefsdata.h"


Boolean GetCategoryName( UInt8 index, Char* name ) PLKRDB_SECTION;
Boolean CategoryExists( UInt8 index ) PLKRDB_SECTION;
UInt16 GetNumOfDatabases( void ) PLKRDB_SECTION;
UInt8 SetCategoryName( UInt8 index, Char* name ) PLKRDB_SECTION;
UInt8 GetCategoryIndex( Char* name ) PLKRDB_SECTION;
UInt8 AddCategoryToFreePosition ( Char* name ) PLKRDB_SECTION;
Err OpenDocList( void ) PLKRDB_SECTION;
Err RemoveDocInfoRecord( UInt16 index ) PLKRDB_SECTION;
Err ReturnDocInfoHandle( const UInt16 index, MemHandle* hand ) PLKRDB_SECTION;
Err UpdateDocumentName( UInt16 index, const char* name ) PLKRDB_SECTION;
void CloseDocList( void ) PLKRDB_SECTION;
void RemoveCategoryName( UInt8 index ) PLKRDB_SECTION;
void MergeCategories( UInt8 dst, UInt8 src ) PLKRDB_SECTION;

static Err InitDocList( void ) PLKRDB_SECTION;
static Err CreateDocList( void ) PLKRDB_SECTION;
static Err InitPlkrAppInfo( DmOpenRef docRef ) PLKRDB_SECTION;
static Int16 FindRecord( Char* name, UInt16 numRecords,
                MemHandle* handle, UInt16* index ) PLKRDB_SECTION;
static void UpdateCategoryData( UInt8 dst, UInt8 src,
                Boolean merge ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
static const Char PlkrDocListName[] = "PlkrDocList";


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
typedef struct {
    UInt16  renamedCategories;
    Char    categoryLabels[ dmRecNumCategories ][ dmCategoryLength ];
    UInt8   categoryUniqIDs[ dmRecNumCategories ];
    UInt8   lastUniqID;
    UInt8   padding;
} PlkrAppInfoType;


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static DmOpenRef PlkrDocList;



/* Create an app info chunk if missing, return result from the database call */
static Err InitPlkrAppInfo
    (
    DmOpenRef docRef    /* reference to document */
    )
{
    UInt16              cardNo;
    MemHandle           handle;
    LocalID             dbID;
    LocalID             appInfoID;
    PlkrAppInfoType*    appInfoP;

    if ( DmOpenDatabaseInfo( docRef, &dbID, NULL, NULL, &cardNo, NULL ) )
        return dmErrInvalidParam;

    if ( DmDatabaseInfo( cardNo, dbID, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, &appInfoID, NULL, NULL, NULL ) )
        return dmErrInvalidParam;

    if ( appInfoID == NULL ) {
        handle = DmNewHandle( docRef, sizeof( PlkrAppInfoType ) );
        if ( handle == NULL )
            return dmErrMemError;

        appInfoID = MemHandleToLocalID( handle );
        DmSetDatabaseInfo( cardNo, dbID, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, &appInfoID, NULL, NULL, NULL );
    }
    appInfoP = (PlkrAppInfoType*) MemLocalIDToLockedPtr( appInfoID, cardNo );
    DmSet( appInfoP, 0, sizeof( PlkrAppInfoType ), 0 );
    CategoryInitialize( ( AppInfoPtr ) appInfoP, strCatDefault );
    MemPtrUnlock( appInfoP );

    return 0;
}



/* Return a handle to record containing document info */
Err ReturnDocInfoHandle
    (
    const UInt16    index,  /* record index */
    MemHandle*      hand    /* upon successful return, the handle to the
                               record, otherwise undefined */
    )
{
    if ( PlkrDocList == NULL )
        return 1;

    ErrFatalDisplayIf( hand == NULL, "ReturnDocInfoHandle: NULL argument" );

    *hand = DmQueryRecord( PlkrDocList, index );
    if ( *hand == NULL )
        return 1;
    else
        return 0;
}



/* Get category name */
Boolean GetCategoryName
    (
    UInt8   index,  /* category index */
    Char*   name    /* upon successful return, the name of
                       the category, otherwise NULL */
    )
{
    CategoryGetName( PlkrDocList, index, name );
    if ( name[ 0 ] == '\0' )
        return false;
    else
        return true;
}



/* Check if category already exists */
Boolean CategoryExists
    (
    UInt8 index /* category index */
    )
{
    Char name[ dmRecNumCategories ];

    return GetCategoryName( index, name );
}



/* Update the category data for the records */
static void UpdateCategoryData
    (
    UInt8   dst,    /* destination category */
    UInt8   src,    /* source category */
    Boolean merge   /* true if the given categories should be merged */
    )
{
    UInt16 i;
    UInt16 category;

    category = 1 << src;

    for ( i = 0; i < DmNumRecords( PlkrDocList ); i++ ) {
        MemHandle       handle;
        DocumentInfo*   handlePtr;
        UInt16          categories;

        handle = DmQueryRecord( PlkrDocList, i );
        if ( handle == NULL ) {
            break;
        }
        handlePtr = (DocumentInfo*) MemHandleLock( handle );
        ErrFatalDisplayIf( handlePtr == NULL, "UpdateCategoryData: MemHandleLock failed" );
        categories = handlePtr->categories;
        if ( ( categories & category ) != 0 ) {
            categories &= ~category;
            if ( merge )
                categories |= ( 1 << dst );
            else if ( categories == 0 )
                categories = 1;
                DmWrite( handlePtr, OFFSETOF( DocumentInfo, categories ), &categories, 
                    sizeof( UInt16 ) );
        }
        MemHandleUnlock( handle );
    }
    CategorySetName( PlkrDocList, src, NULL );
}



/* Remove category */
void RemoveCategoryName
    (
    UInt8 index /* category index */
    )
{
    UpdateCategoryData( 0, index, false );
}



/* Merge categories */
void MergeCategories
    (
    UInt8 dst,  /* destination category */
    UInt8 src   /* source category */
    )
{
    UpdateCategoryData( dst, src, true );
}



/* Set category name, return dmAllCategories if successful, otherwise
   the index for the already existing category */
UInt8 SetCategoryName
    (
    UInt8   index,  /* category index */
    Char*   name    /* category name */
    )
{
    UInt8 i;

    i = CategoryFind( PlkrDocList, name );

    if ( i == dmAllCategories )
        CategorySetName( PlkrDocList, index, name );

    return i;
}



/* Add a new category to a free position, returns the index of the
   added category or dmAllCategories if no free position was found */
UInt8 AddCategoryToFreePosition
    (
    Char* name
    )
{
    UInt8 i;

    for ( i = 0; i < dmRecNumCategories; i++ ) {
        Char tempName[ dmCategoryLength ];

        MemSet( (void*) tempName, dmCategoryLength, 0 );
        CategoryGetName( PlkrDocList, i, tempName );
        if ( tempName[ 0 ] == '\0' ) {
            CategorySetName( PlkrDocList, i, name );
            Prefs()->categories |= ( 1 << i );
            return i;
        }
    }
    return dmAllCategories;
}



/* Rename document name in document list, return result from database call */
Err UpdateDocumentName
    (
    UInt16      index,  /* record index */
    const Char* name    /* new document name */
    )
{
    MemHandle       handle;
    DocumentInfo*   handlePtr;

    handle = DmQueryRecord( PlkrDocList, index );
    if ( handle == NULL ) {
        return DmGetLastErr();
    }
    handlePtr = (DocumentInfo*) MemHandleLock( handle );
    ErrFatalDisplayIf( handlePtr == NULL, "UpdateDocumentName: MemHandleLock failed" );

    DmWrite( (void*) handlePtr, OFFSETOF( DocumentInfo, name ),
        (void*) name, StrLen( name ) + 1 );
    MemHandleUnlock( handle );

    return 0;
}



/* Remove record with document info, return result from database call */
Err RemoveDocInfoRecord
    (
    UInt16 index    /* record index */
    )
{
    return DmRemoveRecord( PlkrDocList, index );
}



/* Compare names ( ascending ), return:   

   0 if rec1.name = rec2.name
 < 0 if rec1.name < rec2.name
 > 0 if rec1.name > rec2.name 

 */
Int16 CompareNames
    (
    void*               rec1,           /* Pointer to the record to sort */
    void*               rec2,           /* Pointer to the record to sort */
    Int16               other,          /* Any other custom information you want 
                                           passed to the comparison function */
    SortRecordInfoPtr   rec1SortInfo,   /* Pointer to SortRecordInfoType structure 
                                           that specify unique sorting information 
                                           for the record */
    SortRecordInfoPtr   rec2SortInfo,   /* Pointer to SortRecordInfoType structure 
                                           that specify unique sorting information 
                                           for the record */
    MemHandle           appInfoH        /* A handle to the document's application 
                                           info block */
    )
{
    Int16 result;

    SET_A4_FROM_A5 

    result = StrCompare( ( (DocumentInfo*) rec1 )->name,
        ( (DocumentInfo*) rec2 )->name );

    RESTORE_A4 

    return result;
}



/* Find record in document, return 0 if successful */
static Int16 FindRecord
    (
    Char*       name,       /* document name to search for */
    UInt16      numRecords, /* range of records to search */
    MemHandle*  handle,     /* upon successful return, the handle to the record,
                               otherwise undefined */
    UInt16*     index       /* upon successful return, the index of the record,
                               otherwise undefined. Pass NULL for this parameter
                               if you don't want to retrieve this value */
    )
{
    UInt16  l;
    UInt16  r;

    if ( numRecords <= 0 )
            return 1;

    l = 0;
    r = numRecords - 1;

    while ( l <= r ) {
        DocumentInfo*   dbInfo;
        UInt16          x;
        Int16           result;

        x = ( r + l ) / 2;

        *handle = DmQueryRecord( PlkrDocList, x );
        if ( *handle == NULL )
            break;

        dbInfo = (DocumentInfo*) MemHandleLock( *handle );
        ErrFatalDisplayIf( dbInfo == NULL, "FindRecord: MemHandleLock failed" );

        result = StrCompare( name, dbInfo->name );
        if ( result == 0 ) {
            MemHandleUnlock( *handle );
            if ( index != NULL )
                *index = x;
            return 0;
        }
        else if ( result < 0 ) {
            r = x - 1;
        }
        else {
            l = x + 1;
        }

        MemHandleUnlock( *handle );
    }
    return 1;
}



/* Populate document list with records, return 0 if successful, otherwise error
   code indicating the reason for the failure */
static Err InitDocList( void )
{
    SysDBListItemType*  dbIDs;
    MemHandle           dbHandle;
    UInt16              i;
    UInt16              dbCount;
    UInt16              numRecords;
    UInt16              index;
    Boolean*            removeDocument;

    /* find all Plucker documents -- not finding any documents is not
       an error */
    SysCreateDataBaseList( (UInt32) ViewerDocumentType,
        (UInt32) ViewerAppID, &dbCount, &dbHandle, false );

    /* get handle to IDs if we found at least one document */
    if ( dbHandle != NULL ) {
        dbIDs = (SysDBListItemType*) MemHandleLock( dbHandle );
        ErrFatalDisplayIf( dbIDs == NULL, "PlkrDocListInit: MemHandleLock failed" );
    }
    else {
        dbIDs = NULL;
    }

    /* create list for documents that have been deleted since last session */
    removeDocument = NULL;
    numRecords = DmNumRecords( PlkrDocList );
    if ( numRecords != 0 ) {
        removeDocument = (Boolean*) MemPtrNew( numRecords * sizeof( Boolean ) );
        for ( i = 0; i < numRecords; i++)
            removeDocument[ i ] = true;
    }

    for ( i = 0; i < dbCount; i++ ) {
        MemHandle       handle;
        DocumentInfo*   handlePtr;
        DocumentInfo    dbInfo;
        Err             err;

        DmDatabaseInfo( dbIDs[ i ].cardNo, dbIDs[ i ].dbID, dbInfo.name,
                   &dbInfo.attributes, NULL, &dbInfo.created,
                   NULL, NULL, NULL, NULL, NULL, NULL, NULL );

        handlePtr = NULL;

        /* check if the documents already exist in the document list,
           if not add a new entry */ 
        err = FindRecord( dbInfo.name, numRecords, &handle, &index );
        if ( err == errNone ) {
            removeDocument[ index ] = false;

            handlePtr = (DocumentInfo*) MemHandleLock( handle );
            ErrFatalDisplayIf( handlePtr == NULL, "PlkrDocListInit: MemHandleLock failed" );

            /* if the document is newer than the stored one or is on a
               different card, updated info */
            if ( handlePtr->created < dbInfo.created || handlePtr->cardNo != dbIDs[ i ].cardNo ) {
                DmDatabaseSize( dbIDs[ i ].cardNo, dbIDs[ i ].dbID, NULL, &dbInfo.size, NULL );

                dbInfo.categories   = handlePtr->categories;
                dbInfo.cardNo       = dbIDs[ i ].cardNo;

                DmWrite( (void*) handlePtr, 0, (void*) &dbInfo, sizeof( DocumentInfo ) );
            }
            MemHandleUnlock( handle );
        }
        else {
            UInt16 dbIndex = dmMaxRecordIndex;

            handle = DmNewRecord( PlkrDocList, &dbIndex, sizeof( DocumentInfo ) );
            if ( handle == NULL ) {
                if ( dbHandle != NULL ) {
                    MemHandleUnlock( dbHandle );
                    MemHandleFree( dbHandle );
                    dbHandle = NULL;
                }
                if ( removeDocument != NULL )
                    MemPtrFree( removeDocument );
                return DmGetLastErr();
            }
            handlePtr = (DocumentInfo*) MemHandleLock( handle );
            ErrFatalDisplayIf( handlePtr == NULL, "PlkrDocListInit: MemHandleLock failed" );

            DmDatabaseSize( dbIDs[ i ].cardNo, dbIDs[ i ].dbID, NULL, &dbInfo.size, NULL );
            dbInfo.categories   = GetDefaultCategories( dbIDs[ i ].dbID, dbIDs[ i ].cardNo );
            dbInfo.cardNo       = dbIDs[ i ].cardNo;
            DmWrite( (void*) handlePtr, 0, (void*) &dbInfo, sizeof( DocumentInfo ) );

            MemHandleUnlock( handle );
            DmReleaseRecord( PlkrDocList, dbIndex, true );
        }
    }

    if ( dbHandle != NULL ) {
        MemHandleUnlock( dbHandle );
        MemHandleFree( dbHandle );
        dbHandle = NULL;
    }

    /* remove records from the end of the list to the top or 
       the list will be corrupt */
    if ( removeDocument != NULL ) {
        UInt16 listIndex;

        listIndex = numRecords;
        while ( listIndex-- ) {
            if ( removeDocument[ listIndex ] )
                DmRemoveRecord( PlkrDocList, listIndex );
        }
        MemPtrFree( removeDocument );
    }

    DmInsertionSort( PlkrDocList, (DmComparF*) CompareNames, 0 );

    return errNone;
}



/* Open document list */
Err OpenDocList( void )
{
    Err err;

    err = errNone;

    if ( PlkrDocList != NULL )
        return err;

    PlkrDocList = DmOpenDatabaseByTypeCreator( PlkrDocListType, ViewerAppID,
                    dmModeReadWrite );
    if ( PlkrDocList == NULL ) {
        err = CreateDocList();
        if ( err != errNone ) {
            return err;
        }
    }
    err = InitDocList();
    if ( err != errNone ) {
        MSG( "Couldn't initialize Plkr document list\n" );
        FrmAlert( warnInsufficientMemory );
    }
    return err;
}



/* Create list of documents ( also includes info about categories ) */
static Err CreateDocList( void )
{
    UInt16  cardNo;
    Err     err;

    cardNo  = 0;
    err     = errNone;

    err = DmCreateDatabase( cardNo, PlkrDocListName, ViewerAppID, PlkrDocListType, false );
    if ( err != errNone ) {
        MSG( "Couldn't create Plkr document list\n" );
        FrmAlert( warnInsufficientMemory );
        return err;
    }
    PlkrDocList = DmOpenDatabaseByTypeCreator( PlkrDocListType, ViewerAppID, dmModeReadWrite );

    err = InitPlkrAppInfo( PlkrDocList );
    if ( err != errNone ) {
        LocalID dbID;

        DmOpenDatabaseInfo( PlkrDocList, &dbID, NULL, NULL, &cardNo, NULL );
        DmCloseDatabase( PlkrDocList );
        DmDeleteDatabase( cardNo, dbID );
        MSG( "Couldn't initialize Plkr document list [ appInfo ]\n" );
        FrmAlert( warnInsufficientMemory );
    }
    return err;
}



/* Close document list */
void CloseDocList( void )
{
    if ( PlkrDocList != NULL ) {
        CloseDatabase( PlkrDocList );
        PlkrDocList = NULL;
    }
}



/* Return number of Plucker documents */
UInt16 GetNumOfDocuments( void )
{
    return DmNumRecords( PlkrDocList );
}



/* Return category index if given category is included in list of categories, 
   otherwise dmAllCategories */
UInt8 GetCategoryIndex
    (
    Char* name  /* category name */
    )
{
    return CategoryFind( PlkrDocList, name );
}
