/*
 * $Id: categoryform.c,v 1.19 2001/12/01 19:16:15 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "categorydata.h"
#include "dbmgrform.h"
#include "debug.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "util.h"


static void StoreCategories( UInt16 index, UInt16 categories ) PLKRDB_SECTION;
static void UpdateListCategoryData( UInt8 dst, UInt8 src, Boolean merge ) PLKRDB_SECTION;
static void RemoveListCategoryName( UInt8 index ) PLKRDB_SECTION;
static void MergeListCategories( UInt8 dst, UInt8 src ) PLKRDB_SECTION;
static void CategoryNameFormInit( void ) PLKRDB_SECTION;
static void CategoryFormInit( void ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define CATEGORY_DELETE 0
#define CATEGORY_RENAME 1

#define SELECT_OK       0


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static UInt8        category            = 0;
static UInt16       categories          = 0;
static UInt16       origCategories      = 0;
static Boolean      savedPrefsData      = false;
static Boolean      origMultipleSelect  = false;
static Boolean      newCategory         = false;
static FilterType   origFilterMode      = FILTER_OR;
static FieldType*   fldPtr              = NULL;

static Char         docName[ dmDBNameLength ];



/* Store category info for document */
static void StoreCategories
    (
    UInt16 index,       /* index for document that should be updated */
    UInt16 categories   /* category data */
    )
{
    MemHandle handle;

    if ( ReturnDocInfoHandle( index, &handle ) == errNone ) {
        DocumentInfo* recordPtr;

        recordPtr = (DocumentInfo*) MemHandleLock( handle );
        ErrFatalDisplayIf( recordPtr == NULL, "StoreCategories: MemHandleLock failed" );

        DmWrite( recordPtr, OFFSETOF( DocumentInfo, categories ), &categories, sizeof( UInt16 ) );
        DocInfo( index )->categories = categories;
        MemHandleUnlock( handle );
    }
}



/* Update the category data for the DB info list */
static void UpdateListCategoryData
    (
    UInt8   dst,    /* destination category */
    UInt8   src,    /* source category */
    Boolean merge   /* true if the given categories should be merged */
    )
{
    UInt16 i;
    UInt16 category;
    UInt16 numberOfDatabases;

    category            = 1 << src;
    numberOfDatabases   = GetNumOfDocuments();

    for ( i = 0; i < numberOfDatabases; i++ ) {
        if ( ( DocInfo( i )->categories & category ) != 0 ) {
            DocInfo( i )->categories &= ~category;
            if ( merge )
                DocInfo( i )->categories |= ( 1 << dst );
            else if ( DocInfo( i )->categories == 0 )
                DocInfo( i )->categories = 1;
        }
    }
}



/* Remove category */
static void RemoveListCategoryName
    (
    UInt8 index /* category index */
    )
{
    UpdateListCategoryData( 0, index, false );
}



/* Merge categories */
static void MergeListCategories
    (
    UInt8 dst,  /* destination category */
    UInt8 src   /* source category */
    )
{
    UpdateListCategoryData( dst, src, true );
}



/* Initialize the add category form */
static void CategoryNameFormInit( void )
{
    Char        name[ dmCategoryLength ];
    FormType*   nameForm;

    nameForm = FrmGetFormPtr( frmNewCategory );

    GetCategoryName( category, name );
    fldPtr = (FieldType*) GetObjectPtr( frmNewCatName );
    FldSetSelection( fldPtr, 0, 0 );
    FldInsert( fldPtr, name, StrLen( name ) );
    FldSetSelection( fldPtr, 0, StrLen( name ) );
    FldSetDirty( fldPtr, false );

    FrmDrawForm( nameForm );
    FrmSetFocus( nameForm, FrmGetObjectIndex( nameForm, frmNewCatName ) );
}



/* Event handler for the category name form */
Boolean CategoryNameFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean handled;

    SET_A4_FROM_A5 

    handled = false;

    switch ( event->eType ) {
        case ctlSelectEvent:
            if ( event->data.ctlEnter.controlID == frmNewCatOK ) {
                if ( FldDirty( fldPtr ) ) {
                    Char* name = FldGetTextPtr( fldPtr );
                    if ( *name != '\0' ) {
                        UInt8 currentCategory;

                        currentCategory = SetCategoryName( category, name );
                        if ( currentCategory != dmAllCategories ) {
                            if ( newCategory ) {
                                FrmCustomAlert( errCategoryExists, name, NULL, NULL );
                                FldSetSelection( fldPtr, 0, StrLen( name ) );
                                break;
                            }
                            else {
                                if ( FrmCustomAlert( confirmMergeCategory, name, NULL, NULL ) == SELECT_OK ) {
                                    MergeCategories( currentCategory, category );
                                    MergeListCategories( currentCategory, category );
                                }
                            }
                        }
                        else if ( newCategory ) {
                            categories |= 1 << category;
                        }
                    }
                }
            }
            else if ( event->data.ctlEnter.controlID != frmNewCatCancel )
                break;

            FrmReturnToForm( frmCategory );
            if ( event->data.ctlEnter.controlID != frmNewCatCancel )
                FrmUpdateForm( frmCategory, frmUpdateList );
            handled = true;
            break;

        case frmOpenEvent:
            CategoryNameFormInit();
            handled = true;
            break;

        case frmCloseEvent:
            handled = false;
            break;

        default:
            handled = false;
    }

    RESTORE_A4 

    return handled;
}



/* Initialize the category form */
static void CategoryFormInit( void )
{
    FormType* categoryForm;
    UInt8     i;

    categoryForm = FrmGetFormPtr( frmCategory );
    FrmDrawForm( categoryForm );

    if ( ! savedPrefsData ) {
        if ( SelectCategoryMode() ) {
            CtlSetValue( (ControlType*) GetObjectPtr( frmCatMultiple ), Prefs()->multipleSelect );
            CtlSetValue( (ControlType*) GetObjectPtr( frmCatOR ), Prefs()->filterMode == FILTER_OR );
            CtlSetValue( (ControlType*) GetObjectPtr( frmCatAND ), Prefs()->filterMode == FILTER_AND );
            categories = Prefs()->categories;
            FrmShowObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatMultiple ) );
            if ( Prefs()->multipleSelect ) {
                FrmShowObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatOR ) );
                FrmShowObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatAND ) );
            }
            else {
                FrmHideObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatOR ) );
                FrmHideObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatAND ) );
            }
        }
        else {
            MemHandle     handle;
            DocumentInfo* recordPtr;

            /* TODO: error check? */
            ReturnDocInfoHandle( ReturnLastIndex(), &handle );
            recordPtr = (DocumentInfo*) MemHandleLock( handle );
            ErrFatalDisplayIf( recordPtr == NULL, "CategoryFormInit: MemHandleLock failed" );
            categories = recordPtr->categories;
            StrNCopy( docName, recordPtr->name, dmDBNameLength );
            MemHandleUnlock( handle );

            FrmHideObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatMultiple ) );
            FrmHideObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatOR ) );
            FrmHideObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatAND ) );
        }
    }
    CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 ), categories & 0x01 );
    for ( i = 1; i <= dmRecNumCategories - 1; i++ ) {
        Char    name[ dmCategoryLength ];
        Char*   labelPtr;

        labelPtr = (Char*) CtlGetLabel( (ControlType*) GetObjectPtr( frmCat1 + i ) );
        FrmHideObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCat1 + i ) );
        if ( GetCategoryName( i, name ) ) {
            StrNCopy( labelPtr, name, dmCategoryLength );
            FrmShowObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatOpen1 + i ) );
        }
        else {
            StrCopy( labelPtr, "" );
            FrmHideObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatOpen1 + i ) );
            categories &= ~( 1 << i );
        }
        CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 + i ), categories & ( 1 << i ) );
        FrmShowObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCat1 + i ) );
    }
    if ( SelectCategoryMode() )
        Prefs()->categories = categories;
    else {
        StoreCategories( ReturnLastIndex(), categories );

        WinDrawChars( docName, StrLen( docName ), 
            ( 160 - FntCharsWidth( docName, StrLen( docName ) ) ) / 2, 120 );
    }

    if ( ! savedPrefsData ) {
        origMultipleSelect  = Prefs()->multipleSelect;
        origFilterMode      = Prefs()->filterMode;
        origCategories      = categories;
        savedPrefsData      = true;
    }
}



/* Event handler for the category form */
Boolean CategoryFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    UInt8       i;
    Char        name[ dmCategoryLength ];
    FormType*   categoryForm;
    Boolean     handled;

    SET_A4_FROM_A5 

    handled         = false;
    categoryForm    = FrmGetFormPtr( frmCategory );

    switch ( event->eType ) {
        case ctlSelectEvent:
            switch ( event->data.ctlEnter.controlID ) {
                case frmCatOK:
                    savedPrefsData = false;

                    if ( SelectCategoryMode() )
                        Prefs()->categories = categories;
                    else if ( categories == 0 ) {
                        FrmAlert( errNoCategory );
                        handled = true;
                        break;
                    }
                    else
                        StoreCategories( ReturnLastIndex(), categories );

                    FrmGotoForm( frmLibrary );
                    FrmUpdateForm( frmLibrary, frmUpdateTable );

                    handled = true;
                    break;

                case frmCatCancel:
                    savedPrefsData = false;

                    Prefs()->multipleSelect = origMultipleSelect;
                    Prefs()->filterMode = origFilterMode;

                    if ( SelectCategoryMode() )
                        Prefs()->categories = origCategories;
                    else
                        StoreCategories( ReturnLastIndex(), origCategories );

                    FrmGotoForm( frmLibrary );

                    handled = true;
                    break;

                case frmCatAll:
                    if ( SelectCategoryMode() ) {
                        Prefs()->multipleSelect = true;
                        CtlSetValue( (ControlType*) GetObjectPtr( frmCatMultiple ), Prefs()->multipleSelect );

                        FrmShowObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatOR ) );
                        FrmShowObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatAND ) );
                    }

                    categories = 1;
                    CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 ), true );
                    for ( i = 1; i <= dmRecNumCategories - 1; i++ ) {
                        if ( GetCategoryName( i, name ) ) {
                            CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 + i ), true );
                            categories |= ( 1 << i );
                        }
                    }
                    handled = true;
                    break;

                case frmCatNone:
                    categories = 0;
                    CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 ), false );
                    for ( i = 1; i <= dmRecNumCategories - 1; i++ )
                        if ( GetCategoryName( i, name ) )
                            CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 + i ), false );

                    handled = true;
                    break;

                case frmCatMultiple:
                    Prefs()->multipleSelect = CtlGetValue( (ControlType*) GetObjectPtr( frmCatMultiple ) );
                    if ( ! Prefs()->multipleSelect ) {
                        categories = 1;
                        CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 ), true );
                        for ( i = 1; i <= dmRecNumCategories - 1; i++ )
                            if ( GetCategoryName( i, name ) )
                                CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 + i ), false );

                        FrmHideObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatOR ) );
                        FrmHideObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatAND ) );
                    }
                    else {
                        FrmShowObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatOR ) );
                        FrmShowObject( categoryForm, FrmGetObjectIndex( categoryForm, frmCatAND ) );
                    }
                    break;

                case frmCatOR:
                    Prefs()->filterMode = FILTER_OR;
                    break;

                case frmCatAND:
                    Prefs()->filterMode = FILTER_AND;
                    break;

                case frmCat1:
                case frmCat2:
                case frmCat3:
                case frmCat4:
                case frmCat5:
                case frmCat6:
                case frmCat7:
                case frmCat8:
                case frmCat9:
                case frmCat10:
                case frmCat11:
                case frmCat12:
                case frmCat13:
                case frmCat14:
                case frmCat15:
                case frmCat16:
                    if ( ! SelectCategoryMode() && categories == 1 && 
                         event->data.ctlEnter.controlID != frmCat1 ) {
                        categories = 0;
                        CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 ), false );
                    }

                    if ( SelectCategoryMode() && ! Prefs()->multipleSelect ) {
                        CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 ), false );
                        for ( i = 1; i <= dmRecNumCategories - 1; i++ )
                            if ( GetCategoryName( i, name ) )
                                CtlSetValue( (ControlType*) GetObjectPtr( frmCat1 + i ), false );
                        CtlSetValue( (ControlType*) GetObjectPtr( event->data.ctlEnter.controlID ), true );
                    }

                    category = event->data.ctlEnter.controlID - frmCat1;
                    if ( CategoryExists( category ) ) {
                        if ( ! SelectCategoryMode() || Prefs()->multipleSelect ) {
                            categories ^= ( 1 << category );
                            if ( categories == 0 )
                                categories = ( 1 << category );
                        }
                        else {
                            categories = Prefs()->categories = ( 1 << category );

                            savedPrefsData = false;

                            FrmGotoForm( frmLibrary );
                            FrmUpdateForm( frmLibrary, frmUpdateTable );

                            handled = true;
                        }
                        CtlSetValue( (ControlType*) GetObjectPtr( event->data.ctlEnter.controlID ), 
                            categories & ( 1 << category ) );
                        break;
                    }
                    CtlSetValue( (ControlType*) GetObjectPtr( event->data.ctlEnter.controlID ), false );
                    if ( SelectCategoryMode() && ! Prefs()->multipleSelect )
                        categories = 0;

                    newCategory = true;
                    FrmPopupForm( frmNewCategory );

                    handled = true;
                    break;

                case frmCatOpen1:
                case frmCatOpen2:
                case frmCatOpen3:
                case frmCatOpen4:
                case frmCatOpen5:
                case frmCatOpen6:
                case frmCatOpen7:
                case frmCatOpen8:
                case frmCatOpen9:
                case frmCatOpen10:
                case frmCatOpen11:
                case frmCatOpen12:
                case frmCatOpen13:
                case frmCatOpen14:
                case frmCatOpen15:
                case frmCatOpen16:
                {
                    ListType*   list;
                    Int16       selection;

                    list = (ListType*) GetObjectPtr( frmCatList );
                    LstSetPosition( list, event->screenX, event->screenY );

                    selection = LstPopupList( list );
                    if ( selection == noListSelection )
                        break;
                    else if ( selection == CATEGORY_DELETE ) {
                        category = event->data.ctlEnter.controlID - frmCatOpen1;
                        GetCategoryName( category, name );
                        if ( FrmCustomAlert( confirmDelete, name, NULL, NULL ) == SELECT_OK ) {
                            RemoveCategoryName( category );
                            RemoveListCategoryName( category );
                            categories &= ~( 1 << category );

                            /* if no other category is selected set it to Unfiled */
                            if ( ! SelectCategoryMode() && categories == 0 ) {
                                categories      = 1;
                                origCategories  = 1;
                            }
                            FrmEraseForm( FrmGetFormPtr( frmCategory ) );
                            CategoryFormInit();
                        }
                    }
                    else if ( selection == CATEGORY_RENAME ) {
                        category    = event->data.ctlEnter.controlID - frmCatOpen1;
                        newCategory = false;
                        FrmPopupForm( frmNewCategory );
                    }
                    handled = true;
                    break;
                }

                default:
                    break;
            }
            break;

        case frmCloseEvent:
            handled = false;
            break;

        case frmOpenEvent:
            CategoryFormInit();
            handled = true;
            break;

        case frmUpdateEvent:
            if ( event->data.frmUpdate.updateCode == frmUpdateList ) {
                FrmEraseForm( FrmGetFormPtr(frmCategory) );
                CategoryFormInit();
                handled = true;
            }
            break;

        default:
            handled = false;
    }

    RESTORE_A4 

    return handled;
}
