/*
 * $Id: dbmgrform.c,v 1.51 2001/12/07 20:38:37 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "dbmgrform.h"
#include "debug.h"
#include "documentdata.h"
#include "emailform.h"
#include "image.h"
#include "mainform.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "session.h"
#include "util.h"
#include "beam.h"
#include "os.h"

Int16 ReturnLastIndex( void ) PLKRDB_SECTION;
Boolean SelectCategoryMode( void ) PLKRDB_SECTION;
DocumentInfo* DocInfo( Int16 index ) PLKRDB_SECTION;

static Err OpenNewDocument( Int16 index ) PLKRDB_SECTION;
static Boolean InitDBInfo( Int16 index ) PLKRDB_SECTION;
static Boolean ReadOnlyDocument( Int16 index ) PLKRDB_SECTION;
static Boolean DeleteDocument( Int16 selection ) PLKRDB_SECTION;
static Boolean CreateTable( void ) PLKRDB_SECTION;
static Char* GetDocumentDate( Int16 index ) PLKRDB_SECTION;
static LocalID GetDocumentID( Int16 index ) PLKRDB_SECTION;
static Int16 FindRow( Int16 y ) PLKRDB_SECTION;
static Int16 CompareNameAsc( DocumentInfo* d1, DocumentInfo* d2 ) PLKRDB_SECTION;
static Int16 CompareNameDesc( DocumentInfo* d1, DocumentInfo* d2 ) PLKRDB_SECTION;
static Int16 CompareSizeAsc( DocumentInfo* d1, DocumentInfo* d2 ) PLKRDB_SECTION;
static Int16 CompareSizeDesc( DocumentInfo* d1, DocumentInfo* d2 ) PLKRDB_SECTION;
static Int16 CompareDateAsc( DocumentInfo* d1, DocumentInfo* d2 ) PLKRDB_SECTION;
static Int16 CompareDateDesc( DocumentInfo* d1, DocumentInfo* d2 ) PLKRDB_SECTION;
static void InitFieldValue( RectangleType* bounds, Char* text, 
                UInt16 alignment ) PLKRDB_SECTION;
static void UpdateScrollbar( void ) PLKRDB_SECTION;
static void ScrollTo( Int16 row ) PLKRDB_SECTION;
static void ScrollUp( Int16 amount ) PLKRDB_SECTION;
static void ScrollDown( Int16 amount ) PLKRDB_SECTION;
static void DeleteOneDocument( Int16 index ) PLKRDB_SECTION;
static void DeleteAllDocuments( void ) PLKRDB_SECTION;
static void InvertRow( Int16 row ) PLKRDB_SECTION;
static void RenameDocFormInit( void ) PLKRDB_SECTION;
static void UpdateColumns( void ) PLKRDB_SECTION;
static void UpdateIndexList( void ) PLKRDB_SECTION;
static void ReleaseTable( void ) PLKRDB_SECTION;
static void LibraryFormInit( void ) PLKRDB_SECTION;
static void ShowSortMessage( void ) PLKRDB_SECTION;
static void HideSortMessage( void ) PLKRDB_SECTION;
static void ShowSyncMessage( void ) PLKRDB_SECTION;
static void HideSyncMessage( void ) PLKRDB_SECTION;
static void UpdateSortMethod( void ) PLKRDB_SECTION;
static void DrawHelpButton( FormType* form ) PLKRDB_SECTION;
static void InitiateTable( void ) PLKRDB_SECTION;
static void InitiateScrollbar( FormType* form ) PLKRDB_SECTION;
static void SortIndexList( void ) PLKRDB_SECTION;

static Int16( *compareItems ) ( DocumentInfo*, DocumentInfo* );

#define CompareItems( d1, d2 ) ( *compareItems )( ( d1 ),( d2 ) )


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define REMOVE_COLUMN       0
#define ONE_ROW             1

#define DOC_OPEN            0
#define DOC_CATEGORIZE      1
#define DOC_DELETE          2
#define DOC_RENAME          3
#define DOC_BEAM            4

#define VISIBLE_ROWS        11
#define MAX_COLUMNS         4

#define SELECTDOC_COLUMN    0
#define DOCNAME_COLUMN      1
#define DATE_COLUMN         2
#define SIZE_COLUMN         3

#define TABLE_WIDTH         153
#define SELECTDOC_WIDTH     9
#define DOCNAME_WIDTH       73
#define DATE_WIDTH          40
#define SIZE_WIDTH          26

#define SELECT_OK           0


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static Int16*           indexList           = NULL;
static DocumentInfo     docInfo;

static UInt16           entries             = 0;
static Int16            firstVisibleRow     = 0;
static Int16            numberOfRows        = 0;

static Int16            startY              = 0;
static TableType*       table               = NULL;
static RectangleType    lastBounds;

static FieldType*       fldPtr              = NULL;
static Int16            lastIndex           = -1;
static Boolean          selectCategories    = false;

static MemHandle        docBitmapHandle     = NULL;
static BitmapType*      docBitmapPtr        = NULL;

static const Char*      Ascending           = "\007";
static const Char*      Descending          = "\010";



/* Display Sorting message */
static void ShowSortMessage( void )
{
    FormType* frm;

    frm = FrmInitForm( frmSortDoc );

    FrmSetActiveForm( frm );
    FrmDrawForm( frm );
}



/* Hide sorting message */
static void HideSortMessage( void )
{
    FormType* libraryForm;
    FormType* frm;

    libraryForm = FrmGetFormPtr( frmLibrary );
    frm         = FrmGetFormPtr( frmSortDoc );

    FrmEraseForm( frm );
    FrmDeleteForm( frm );
    FrmSetActiveForm( libraryForm );
}



/* Display Sync message */
static void ShowSyncMessage( void )
{
    FormType* frm;

    frm = FrmInitForm( frmInitCategory );

    FrmSetActiveForm( frm );
    FrmDrawForm( frm );
}



/* Hide Sync message */
static void HideSyncMessage( void )
{
    FormType* libraryForm;
    FormType* frm;
    
    libraryForm = FrmGetFormPtr( frmLibrary );
    frm         = FrmGetFormPtr( frmInitCategory );

    FrmEraseForm( frm );
    FrmDeleteForm( frm );
    FrmSetActiveForm( libraryForm );
}



/* Initialize docinfo structure */
static Boolean InitDBInfo
    (
    Int16 index /* item number in list */
    )
{
    MemHandle       handle;
    DocumentInfo*   recordPtr;

    if ( lastIndex == index )
        return true;

    if ( ReturnDocInfoHandle( index, &handle ) == errNone ) {
        recordPtr = (DocumentInfo*) MemHandleLock( handle );
        ErrFatalDisplayIf( recordPtr == NULL, "InitDBInfo: MemHandleLock failed" );

        StrNCopy( docInfo.name, recordPtr->name, dmDBNameLength );

        docInfo.cardNo      = recordPtr->cardNo;
        docInfo.created     = recordPtr->created;
        docInfo.attributes  = recordPtr->attributes;
        docInfo.size        = recordPtr->size;
        docInfo.categories  = recordPtr->categories;

        MemHandleUnlock( handle );

        lastIndex = index;

        return true;
    }
    return false;
}



/* Return pointer to docinfo structure */
DocumentInfo* DocInfo
    (
    Int16 index /* item number in list */
    )
{
    if ( lastIndex != index )
        InitDBInfo( index );

    return &docInfo;
}



/* Return index for selected document */
Int16 ReturnLastIndex( void )
{
    return lastIndex;
}



/* Check if in select categories mode */
Boolean SelectCategoryMode( void )
{
    return selectCategories;
}



/* Check if document is read only */
static Boolean ReadOnlyDocument
    (
    Int16 index /* item number in list */
    )
{
    if ( lastIndex != index )
        InitDBInfo( index );

    return ( ( docInfo.attributes & dmHdrAttrReadOnly ) == dmHdrAttrReadOnly );
}



/* Check if document is copy protected */
static Boolean CopyProtectedDocument
    (
    Int16 index /* item number in list */
    )
{
    if ( lastIndex != index )
        InitDBInfo( index );

    return ( ( docInfo.attributes & dmHdrAttrCopyPrevention ) == dmHdrAttrCopyPrevention );
}



/* Return ID for document */
LocalID GetDocumentID
    (
    Int16 index /* item number in list */
    )
{
    if ( lastIndex != index )
        InitDBInfo( index );

    return DmFindDatabase( docInfo.cardNo, docInfo.name );
}



/* Return creation date of document */
static Char* GetDocumentDate
    (
    Int16 index /* item number in list */
    )
{
    static Char     dateStr[ dateStringLength ];
    DateTimeType    dateTime;

    if ( lastIndex != index )
        InitDBInfo( index );

    TimSecondsToDateTime( docInfo.created, &dateTime );

    DateToAscii( dateTime.month, dateTime.day, dateTime.year,
        (DateFormatType) PrefGetPreference( prefDateFormat ),
        (Char*) &dateStr );

    return (Char*) &dateStr;
}



/* Initialize field values */
static void InitFieldValue
    (
    RectangleType*  bounds,     /* boundaries for the field */
    Char*           text,       /* text value */
    UInt16          alignment   /* alignment type ( rightAlign or leftAlign ) */
    )
{
    FieldType       fld;
    FieldAttrType   attr;

    MemSet( &fld, sizeof( FieldType ), 0 );
    FldSetBounds( &fld, bounds );

    MemSet( &attr, sizeof( FieldAttrType ), 0 );
    attr.usable         = true;
    attr.singleLine     = true;
    attr.justification  = alignment;
    FldSetAttributes( &fld, &attr );

    FldSetTextPtr( &fld, text );
    FldDrawField( &fld );
}



/* Update scrollbar */
static void UpdateScrollbar( void )
{
    UInt16 maxValue;

    if ( numberOfRows < VISIBLE_ROWS )
        maxValue = 0;
    else
        maxValue = numberOfRows - VISIBLE_ROWS;

    SclSetScrollBar( (ScrollBarType*) GetObjectPtr( frmLibraryScrollBar ),
        firstVisibleRow, 0, maxValue, VISIBLE_ROWS );
}



/* Scroll to given row */
static void ScrollTo
    (
    Int16 row   /* row to scroll to */
    )
{
    Int16 maxScroll;

    maxScroll       = numberOfRows - VISIBLE_ROWS;
    firstVisibleRow = row;

    if ( row < 0 || maxScroll < 0 )
        firstVisibleRow = 0;
    else if ( maxScroll < row )
        firstVisibleRow = maxScroll;

    TblMarkTableInvalid( table );
    TblRedrawTable( table );
    UpdateScrollbar();
}



/* Scroll up */
static void ScrollUp
    (
    Int16 amount    /* number of rows to scroll */
    )
{
    ScrollTo( firstVisibleRow - amount );
}



/* Scroll down */
static void ScrollDown
    (
    Int16 amount    /* number of rows to scroll */
    )
{
    ScrollTo( firstVisibleRow + amount );
}



/* Draw item in table */
void DrawItem
    (
    void*           tbl,    /* pointer to table */
    Int16           row,    /* item's row ( zero-based ) */
    Int16           column, /* item's column ( zero-based ) */
    RectangleType*  bounds  /* boundaries for the item */
    )
{
    SET_A4_FROM_A5 

    WinEraseRectangle( bounds, 0 );
    row += firstVisibleRow;

    if ( row < numberOfRows ) {
        FontID  oldFont;
        Char    text[ dmDBNameLength ];
        Int16   index;
        UInt32  size;

        index = indexList[ row ];
        InitDBInfo( index );

        text[ 0 ] = '\0';

        switch ( column ) {
            case SELECTDOC_COLUMN:
                bounds->extent.x = TABLE_WIDTH;
                WinEraseRectangle( bounds, 0 );
                WinDrawBitmap( docBitmapPtr, bounds->topLeft.x + 1, bounds->topLeft.y + 1 );
                if ( ( StrCompare( Prefs()->docName, docInfo.name ) == 0 ) && 
                     Prefs()->cardNo == docInfo.cardNo ) {
                    bounds->extent.x = SELECTDOC_WIDTH;
                    WinInvertRectangle( bounds, 0 );
                }
                break;

            case DOCNAME_COLUMN:
                oldFont = FntSetFont( stdFont );
                StrNCopy( text, docInfo.name, dmDBNameLength );
                TrimText( text, bounds->extent.x - 2 );
                InitFieldValue( bounds, text, leftAlign );
                FntSetFont( oldFont );
                break;

            case DATE_COLUMN:
                oldFont = FntSetFont( stdFont );
                if ( Prefs()->showDate && Prefs()->showSize ) {
                    StrNCopy( text, GetDocumentDate( index ), dmDBNameLength );
                }
                InitFieldValue( bounds, text, rightAlign );
                FntSetFont( oldFont );
                break;

            case SIZE_COLUMN:
                oldFont = FntSetFont( stdFont );
                if ( Prefs()->showSize ) {
                    size = ( docInfo.size + 512 ) / 1024;
                    if ( size != 0 )
                        StrPrintF( text, "%ldk", size );
                    else
                        StrPrintF( text, "%ld", docInfo.size );
                }
                else if ( Prefs()->showDate ) {
                    StrNCopy( text, GetDocumentDate( index ), dmDBNameLength );
                }
                InitFieldValue( bounds, text, rightAlign );
                FntSetFont( oldFont );
                break;

            default:
                break;
        }
    }
    else if ( row == 0 && column == SELECTDOC_COLUMN && numberOfRows == 0 ) {
        FontID  oldFont;
        Char    message[ dmDBNameLength ];
        UInt16  messageWidth;

        /* change column widths in table so the message fits */
        messageWidth = DOCNAME_WIDTH + DATE_WIDTH + SIZE_WIDTH;
        TblSetColumnWidth( table, SIZE_COLUMN, REMOVE_COLUMN );
        TblSetColumnWidth( table, DATE_COLUMN, REMOVE_COLUMN );
        TblSetColumnWidth( table, DOCNAME_COLUMN, REMOVE_COLUMN );
        TblSetColumnWidth( table, SELECTDOC_COLUMN, messageWidth );

        /* change current position */
        bounds->topLeft.x   = 3;
        bounds->extent.x    = messageWidth;

        /* include message using bold font */
        oldFont = FntSetFont( stdFont );
        SysCopyStringResource( message, strNoDocumentsFound );
        InitFieldValue( bounds, message, leftAlign );
        FntSetFont( oldFont );
    }

    RESTORE_A4 
}



/* Populate table with documents */
static Boolean CreateTable( void )
{
    /* Keep the doc bitmap locked since it is frequently used */
    if ( docBitmapHandle == NULL ) {
        docBitmapHandle = DmGetResource( bitmapRsc, bmpDoc );
        docBitmapPtr    = (BitmapType*) MemHandleLock( docBitmapHandle );
    }

    entries = GetNumOfDocuments();
    if ( 0 < entries ) {
        /* Allocate arrays for document info table */
        indexList = (Int16*) MemPtrNew( entries * sizeof( Int16 ) );
        if ( indexList == NULL ) {
            MSG( "Couldn't allocate memory for index list\n" );
            FrmAlert( warnInsufficientMemory );
            return false;
        }
    }
    return true;
}



/* Release allocated memory */
static void ReleaseTable( void )
{
    if ( docBitmapHandle != NULL ) {
        MemHandleUnlock( docBitmapHandle );
        DmReleaseResource( docBitmapHandle );
        docBitmapHandle = NULL;
    }
    if ( indexList != NULL ) {
        MemPtrFree( indexList );
        indexList = NULL;
    }
}



/* Compare names ( ascending ) */
static Int16 CompareNameAsc
    (
    DocumentInfo* d1,   /* pointer to the record to sort */
    DocumentInfo* d2    /* pointer to the record to sort */
    )
{
    /*     0 if rec1.name = rec2.name */
    /*   < 0 if rec1.name < rec2.name */
    /*   > 0 if rec1.name > rec2.name */
    return StrCompare( d1->name, d2->name );
}



/* Compare names ( descending ) */
static Int16 CompareNameDesc
    (
    DocumentInfo* d1,   /* pointer to the record to sort */
    DocumentInfo* d2    /* pointer to the record to sort */
    )
{
    /*     0 if rec1.name = rec2.name */
    /*   < 0 if rec1.name > rec2.name */
    /*   > 0 if rec1.name < rec2.name */
    return StrCompare( d2->name, d1->name );
}



/* Compare size ( ascending ) */
static Int16 CompareSizeAsc
    (
    DocumentInfo* d1,   /* pointer to the record to sort */
    DocumentInfo* d2    /* pointer to the record to sort */
    )
{
    /*   0 if rec1.size = rec2.size */
    /* < 0 if rec1.size < rec2.size */
    /* > 0 if rec1.size > rec2.size */
    if ( d2->size < d1->size )
        return 1;
    else if ( d1->size < d2->size )
        return -1;
    else
        return 0;
}



/* Compare size ( descending ) */
static Int16 CompareSizeDesc
    (
    DocumentInfo* d1,   /* pointer to the record to sort */
    DocumentInfo* d2    /* pointer to the record to sort */
    )
{
    /*   0 if rec1.size = rec2.size */
    /* < 0 if rec1.size > rec2.size */
    /* > 0 if rec1.size < rec2.size */
    if ( d2->size < d1->size )
        return -1;
    else if ( d1->size < d2->size )
        return 1;
    else
        return 0;
}



/* Compare date ( ascending ) */
static Int16 CompareDateAsc
    (
    DocumentInfo* d1,   /* pointer to the record to sort */
    DocumentInfo* d2    /* pointer to the record to sort */
    )
{
    /*   0 if rec1.created = rec2.created */
    /* < 0 if rec1.created < rec2.created */
    /* > 0 if rec1.created > rec2.created */
    if ( d2->created < d1->created )
        return 1;
    else if ( d1->created < d2->created )
        return -1;
    else
        return 0;
}



/* Compare date ( descending ) */
static Int16 CompareDateDesc
    (
    DocumentInfo* d1,   /* pointer to the record to sort */
    DocumentInfo* d2    /* pointer to the record to sort */
    )
{
    /*   0 if rec1.created = rec2.created */
    /* < 0 if rec1.created > rec2.created */
    /* > 0 if rec1.created < rec2.created */
    if ( d2->created < d1->created )
        return -1;
    else if ( d1->created < d2->created )
        return 1;
    else
        return 0;
}



/* Update settings for selected sort method */
static void UpdateSortMethod( void )
{
    switch ( Prefs()->sortType ) {
        case SORT_NAME:
            if ( Prefs()->sortOrder == SORTORDER_ASC )
                compareItems = ( Int16( * )( DocumentInfo*, DocumentInfo* ) ) &CompareNameAsc;
            else
                compareItems = ( Int16( * )( DocumentInfo*, DocumentInfo* ) ) &CompareNameDesc;
            break;

        case SORT_DATE:
            if ( Prefs()->sortOrder == SORTORDER_ASC )
                compareItems = ( Int16( * )( DocumentInfo*, DocumentInfo* ) ) &CompareDateAsc;
            else
                compareItems = ( Int16( * )( DocumentInfo*, DocumentInfo* ) ) &CompareDateDesc;
            break;

        case SORT_SIZE:
            if ( Prefs()->sortOrder == SORTORDER_ASC )
                compareItems = ( Int16( * )( DocumentInfo*, DocumentInfo* ) ) &CompareSizeAsc;
            else
                compareItems = ( Int16( * )( DocumentInfo*, DocumentInfo* ) ) &CompareSizeDesc;
            break;
    }
}



/* Update list of documents sorting them according to the selected sort method */
static void UpdateIndexList( void )
{
    UInt16 i;

    lastIndex       = -1;
    numberOfRows    =  0;

    for ( i = 0; i < entries; i++ ) {
        Int16   j;
        UInt16  category;

        InitDBInfo( i );

        category = docInfo.categories & Prefs()->categories;
        if ( ( Prefs()->filterMode == FILTER_OR && category ) ||
             ( Prefs()->filterMode == FILTER_AND && 
               category == Prefs()->categories ) ) {

            /* Put items in sorted order */
            j = numberOfRows;
            if ( j != 0 ) {
                DocumentInfo currentItem;

                MemMove( &currentItem, &docInfo, sizeof( DocumentInfo ) );

                while ( j != 0 && InitDBInfo( indexList[ j - 1 ] ) && 
                        0 < CompareItems( &docInfo, &currentItem ) ) {
                    indexList[ j ] = indexList[ j - 1 ];
                    j--;
                }
            }
            indexList[ j ] = i;
            numberOfRows++;
        }
    }
}



/* Open the specified document */
static Err OpenNewDocument
    (
    Int16 index /* item number in list */
    )
{
    Err     err;
    LocalID dbID;

    if ( lastIndex != index )
        InitDBInfo( index );

    dbID    = GetDocumentID( index );
    err     = OpenDocument( dbID, docInfo.cardNo );
    if ( err != errNone )
        return err;

    StrCopy( Prefs()->docName, docInfo.name );
    Prefs()->dbID   = dbID;
    Prefs()->cardNo = docInfo.cardNo;

    InitSessionData();

    return 0;
}



/* Delete document and remove it from the list of documents */
static Boolean DeleteDocument
    (
    Int16 index /* item number in list */
    )
{
    if ( lastIndex != index )
        InitDBInfo( index );

    if ( DmDeleteDatabase( docInfo.cardNo, GetDocumentID( index ) ) == errNone ) {
        Char* label;

        label = docInfo.name;
        DeleteMetaDocument( label );

        RemoveDocInfoRecord( index );

        if ( ( StrCompare( Prefs()->docName, docInfo.name ) == 0 ) && 
             Prefs()->cardNo == docInfo.cardNo ) {
            Prefs()->docName[ 0 ]   = '\0';
            Prefs()->dbID           = NULL;
            Prefs()->cardNo         = 0;
        }
        entries--;

        return true;
    }
    else
        return false;
}



/* Delete the document specified by the user */
static void DeleteOneDocument
    (
    Int16 index /* item number in list */
    )
{
    if ( lastIndex != index )
        InitDBInfo( index );

    if ( FrmCustomAlert( confirmDelete, docInfo.name, NULL, NULL ) == SELECT_OK ) {
        if ( DeleteDocument( index ) ) {
            UpdateIndexList();
            ScrollUp( ONE_ROW );
            TblDrawTable( table );
        }
        else
            FrmAlert( errNoDeleteDoc );
    }
}



static void SortIndexList( void )
{
    int i;
    int j;
    int v;

    for ( i = 1; i < numberOfRows; i++ ) {
        v = indexList[ i ];
        j = i;
        while ( j != 0 && v < indexList[ j - 1 ] ) {
            indexList[ j ] = indexList[ j - 1 ];
            j--;
        }
        indexList[ j ] = v;
    }
}



/* Delete all displayed documents */
static void DeleteAllDocuments( void )
{
    if ( FrmAlert( confirmDeleteAllDoc ) == SELECT_OK ) {
        Boolean failed;

        SortIndexList();

        failed = false;
        while ( numberOfRows-- ) {
            Int16 index;

            index = indexList[ numberOfRows ];
            InitDBInfo( index );

            if ( ! DeleteDocument( index ) )
                failed = true;
        }
        UpdateIndexList();
        ScrollTo( 0 );
        TblDrawTable( table );
        if ( failed )
            FrmAlert( errNoDeleteAllDoc );
    }
}



/* Find row in table given y position of pen */
static Int16 FindRow
    (
    Int16 y /* y coordinate of pen tap */
    )
{
    Int16 row;

    for ( row = 0; row < VISIBLE_ROWS; row++ ) {
        RectangleType bounds;

        TblGetItemBounds( table, row, 0, &bounds );
        if ( RctPtInRectangle( bounds.topLeft.x + 1, y, &bounds ) )
            return row;
    }
    return NOT_FOUND;
}



/* Invert row in table */
static void InvertRow
    (
    Int16 row  /* row to invert */
    )
{
    RectangleType bounds;

    TblGetItemBounds( table, row, 0, &bounds );

    if ( Prefs()->scrollbar == SCROLLBAR_LEFT )
        bounds.topLeft.x = SELECTDOC_WIDTH + 7;
    else
        bounds.topLeft.x = SELECTDOC_WIDTH;

    bounds.extent.x = TABLE_WIDTH - SELECTDOC_WIDTH;
    WinInvertRectangle( &bounds, 0 );
    MemMove( &lastBounds, &bounds, sizeof( bounds ) );
}



/* Initialize the rename document form */
static void RenameDocFormInit( void )
{
    FormType* nameForm;

    fldPtr = (FieldType*) GetObjectPtr( frmRenameDocField );
    FldSetSelection( fldPtr, 0, 0 );
    FldInsert( fldPtr, docInfo.name, StrLen( docInfo.name ) );
    FldSetSelection( fldPtr, 0, StrLen( docInfo.name ) );
    FldSetDirty( fldPtr, false );

    nameForm = FrmGetFormPtr( frmRenameDoc );
    FrmDrawForm( nameForm );
    FrmSetFocus( nameForm, FrmGetObjectIndex( nameForm, frmRenameDocField ) );
}



/* Event handler for the rename document form */
Boolean RenameDocFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean handled;

    SET_A4_FROM_A5 

    handled = false;

    switch ( event->eType ) {
        case ctlSelectEvent:
            if ( event->data.ctlEnter.controlID == frmRenameDocOK ) {
                if ( FldDirty( fldPtr ) ) {
                    Char* text;

                    text = FldGetTextPtr( fldPtr );
                    if ( *text != '\0' ) {
                        Err err;

                        err = RenameDocument( text, docInfo.name, docInfo.cardNo );
                        if ( err != errNone ) {
                            FrmCustomAlert( errAlreadyExists, text, NULL, NULL );
                            FldSetSelection( fldPtr, 0, StrLen( text ) );
                            break;
                        }
                        else {
                            UpdateDocumentName( lastIndex, text );
                        }
                    }
                    else
                        break;
                }
            }
            else if ( event->data.ctlEnter.controlID != frmRenameDocCancel )
                break;

            FrmReturnToForm( frmLibrary );
            if ( event->data.ctlEnter.controlID != frmRenameDocCancel )
                FrmUpdateForm( frmLibrary, frmUpdateTable );
            handled = true;
            break;

        case frmOpenEvent:
            RenameDocFormInit();
            handled = true;
            break;

        case frmCloseEvent:
            handled = false;
            break;

        default:
            handled = false;
    }

    RESTORE_A4 

    return handled;
}



/* Update the width of the columns depending on filter */
static void UpdateColumns( void )
{
    UInt16 DocNameWidth;

    DocNameWidth = DOCNAME_WIDTH;

    if ( Prefs()->showDate ) {
        TblSetColumnWidth( table, DATE_COLUMN, DATE_WIDTH );
    }
    else {
        DocNameWidth += DATE_WIDTH;
        TblSetColumnWidth( table, DATE_COLUMN, REMOVE_COLUMN );
    }

    if ( Prefs()->showSize ) {
        TblSetColumnWidth( table, SIZE_COLUMN, SIZE_WIDTH );
    }
    else {
        DocNameWidth += SIZE_WIDTH;
        if ( Prefs()->showDate ) {
            TblSetColumnWidth( table, DATE_COLUMN, REMOVE_COLUMN );
            TblSetColumnWidth( table, SIZE_COLUMN, DATE_WIDTH );
        }
        else {
            TblSetColumnWidth( table, SIZE_COLUMN, REMOVE_COLUMN );
        }
    }
    TblSetColumnWidth( table, DOCNAME_COLUMN, DocNameWidth );
}



/* Initiate table */
static void InitiateTable( void )
{
    Int16 column;

    table = (TableType*) GetObjectPtr( frmLibraryTable );
    for ( column = 0; column < MAX_COLUMNS; column++ ) {
        Int16 row;

        TblSetCustomDrawProcedure( table, column, DrawItem );
        TblSetColumnUsable( table, column, true );
        for ( row = 0; row < VISIBLE_ROWS; row++ ) {
            TblSetRowUsable( table, row, true );
            TblSetItemStyle( table, row, column, customTableItem );
        }
    }
}



/* Initiate settings for scrollbar */
static void InitiateScrollbar
    (
    FormType* form  /* pointer to active form */
    )
{
    RectangleType r;

    TblGetBounds( table, &r );
    if ( Prefs()->scrollbar == SCROLLBAR_LEFT ) {
        r.topLeft.x = 7;
        FrmSetObjectPosition( form, FrmGetObjectIndex( form, frmLibraryScrollBar ), 
            0, 16 );
    }
    else {
        r.topLeft.x = 0;
        FrmSetObjectPosition( form, FrmGetObjectIndex( form, frmLibraryScrollBar ), 
            TABLE_WIDTH, 16 );
    }
    TblSetBounds( table, &r );
}



/* Display help icon in header */
static void DrawHelpButton
    (
    FormType* form  /* pointer to active form */
    )
{
    RectangleType   r;
    FontID          font;
    Char            help;

    font = FntSetFont( symbolFont );
    help = symbolHelp;

    FrmGetObjectBounds( form, FrmGetObjectIndex( form, frmLibraryHelp ), &r );
    WinEraseRectangle( &r, 4 );
    WinDrawChars( &help, 1, r.topLeft.x + 3, 2 );

    FntSetFont( font );
}



/* Initialize the Document Library form */
static void LibraryFormInit( void )
{
    FormType*       libraryForm;
    ListType*       list;
    ControlType*    ctl;
    Err             err;

    libraryForm = FrmGetFormPtr( frmLibrary );

    InitiateTable();
    InitiateScrollbar( libraryForm );

    ShowSyncMessage();

    DeleteOffScreenWindow();
    SaveSessionData();
    CloseDocument();

    err = OpenDocList();
    if ( err != errNone ) {
        SendAppStopEvent();
        return;
    }
    if ( ! CreateTable() ) {
        SendAppStopEvent();
        return;
    }
    UpdateSortMethod();
    UpdateIndexList();
    UpdateColumns();

    HideSyncMessage();
    FrmDrawForm( libraryForm );

    DrawHelpButton( libraryForm );

    if ( Prefs()->sortOrder == SORTORDER_ASC )
        CtlSetLabel( (ControlType*) GetObjectPtr( frmLibrarySortOrder ),
            Ascending );
    else
        CtlSetLabel( (ControlType*) GetObjectPtr( frmLibrarySortOrder ),
            Descending );

    list = (ListType*) GetObjectPtr( frmLibrarySortList );
    LstSetSelection( list, Prefs()->sortType );

    ctl = (ControlType*) GetObjectPtr( frmLibrarySortPopup );
    CtlSetLabel( ctl, LstGetSelectionText( list, Prefs()->sortType ) );

    UpdateScrollbar();

    CtlSetValue( (ControlType*) GetObjectPtr( frmLibraryDate ),
        Prefs()->showDate );
    CtlSetValue( (ControlType*) GetObjectPtr( frmLibrarySize ),
        Prefs()->showSize );

    Prefs()->lastForm = frmLibrary;
}



/* Event handler for the document manager form */
Boolean LibraryFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean         handled;
    RectangleType   bounds;
    Int16           currentRow;

    SET_A4_FROM_A5 

    handled = false;

    switch ( event->eType ) {
        case ctlSelectEvent:
            if ( event->data.ctlEnter.controlID == frmLibraryCategory ) {
                selectCategories = true;
                FrmGotoForm( frmCategory );
            }
            else if ( event->data.ctlEnter.controlID == frmLibraryDate ) {
                Prefs()->showDate = ! Prefs()->showDate;
                UpdateColumns();
                TblDrawTable( table );
            }
            else if ( event->data.ctlEnter.controlID == frmLibrarySize ) {
                Prefs()->showSize = ! Prefs()->showSize;
                UpdateColumns();
                TblDrawTable( table );
            }
            else if ( event->data.ctlEnter.controlID == frmLibraryHelp ) {
                FrmHelp( strLibraryHelp );
            }
            else if ( event->data.ctlEnter.controlID == frmLibrarySortOrder ) {
                if ( Prefs()->sortOrder == SORTORDER_ASC ) {
                    Prefs()->sortOrder = SORTORDER_DESC;
                    CtlSetLabel( (ControlType*) GetObjectPtr( frmLibrarySortOrder ), Descending );
                }
                else {
                    Prefs()->sortOrder = SORTORDER_ASC;
                    CtlSetLabel( (ControlType*) GetObjectPtr( frmLibrarySortOrder ), Ascending );
                }
                UpdateSortMethod();
                FrmUpdateForm( frmLibrary, frmUpdateTable );
            }
            else
                break;

            handled = true;
            break;

        case penDownEvent:
            TblGetBounds( table, &bounds );
            if ( RctPtInRectangle( event->screenX, event->screenY, &bounds ) ) {
                startY = event->screenY;
                currentRow = FindRow( startY );

                if ( currentRow != NOT_FOUND && currentRow < numberOfRows ) {
                    if ( ( event->screenX - bounds.topLeft.x ) < 10 ) {
                        ListType*   list;
                        Int16       selection;

                        startY = 0;

                        list = (ListType*) GetObjectPtr( frmLibraryList );
                        LstSetPosition( list, event->screenX, event->screenY );

                        selection = LstPopupList( list );
                        if ( selection != noListSelection ) {
                            Int16 index;

                            index = indexList[ currentRow + firstVisibleRow ];
                            InitDBInfo( index );

                            if ( selection == DOC_DELETE ) {
                                DeleteOneDocument( index );
                            }
                            else if ( selection == DOC_OPEN ) {
                                Err err;

                                err = OpenNewDocument( index );
                                if ( err == errNone )
                                    FrmGotoForm( Prefs()->toolbar );
                            }
                            else if ( selection == DOC_RENAME ) {
                                if ( ReadOnlyDocument( index ) )
                                    FrmAlert( errReadOnly );
                                else
                                    FrmPopupForm( frmRenameDoc );
                            }
                            else if ( selection == DOC_CATEGORIZE ) {
                                selectCategories = false;
                                FrmGotoForm( frmCategory );
                            }
                            else if ( selection == DOC_BEAM ) {
                                if ( BeamSupported() ) {
                                    if ( CopyProtectedDocument( index ) )
                                        FrmAlert( infoCopyProtected );
                                    else
                                        BeamDocument( docInfo.name,
                                            GetDocumentID( index ),
                                            docInfo.cardNo );
                                }
                                else
                                    FrmAlert( infoNoBeamSupport );
                            }
                        }
                    }
                    else {
                        InvertRow( currentRow );
                    }
                }
                if ( currentRow < VISIBLE_ROWS )
                    handled = true;
            }
            break;

        case penUpEvent:
            TblGetBounds( table, &bounds );
            if ( startY != 0 &&
                 RctPtInRectangle( event->screenX, event->screenY, &bounds ) ) {
                currentRow = FindRow( event->screenY );
                if ( currentRow != NOT_FOUND && currentRow < numberOfRows ) {
                    Err err;
                    Int16 index;

                    index = indexList[ currentRow + firstVisibleRow ];
                    InitDBInfo( index );

                    SndPlaySystemSound( sndClick );
                    WinInvertRectangle( &lastBounds, 0 );
                    MemSet( &lastBounds, sizeof( lastBounds ), 0 );

                    err = OpenNewDocument( index );
                    if ( err == errNone )
                        FrmGotoForm( Prefs()->toolbar );
                }
                if ( currentRow < VISIBLE_ROWS )
                    handled = true;
            }
            break;

        case penMoveEvent:
            if ( startY != 0 && 0 < numberOfRows ) {
                TblGetBounds( table, &bounds );
                if ( ! RctPtInRectangle( event->screenX, event->screenY,
                        &bounds ) ) {
                    Int16   x;
                    Int16   y;
                    Boolean penDown;

                    WinInvertRectangle( &lastBounds, 0 );
                    MemSet( &lastBounds, sizeof( lastBounds ), 0 );

                    if ( event->screenY < bounds.topLeft.y )
                        do {
                            ScrollUp( ONE_ROW );
                            EvtGetPen( &x, &y, &penDown );
                        } while ( penDown && ( y < bounds.topLeft.y ) );

                    if ( bounds.topLeft.y + bounds.extent.y < event->screenY )
                        do {
                            ScrollDown( ONE_ROW );
                            EvtGetPen( &x, &y, &penDown );
                        } while ( penDown && ( bounds.topLeft.y + bounds.extent.y < y ) );
                }
                else {
                    currentRow = FindRow( event->screenY );

                    if ( numberOfRows <= currentRow ) {
                        WinInvertRectangle( &lastBounds, 0 );
                        MemSet( &lastBounds, sizeof( lastBounds ), 0 );
                    }
                    else if ( 5 <= abs( startY - event->screenY ) ) {
                        WinInvertRectangle( &lastBounds, 0 );
                        InvertRow( currentRow );
                    }
                }
            }
            handled = true;
            break;

        case sclRepeatEvent:
            ScrollTo( event->data.sclRepeat.newValue );
            handled = false;
            break;

        case popSelectEvent:
        {
            Int16 selection;

            selection   = event->data.popSelect.selection;
            if ( selection != noListSelection ) {
                ListType*       list;
                Char*           label;
                ControlType*    ctl;

                list    = event->data.popSelect.listP;
                label   = LstGetSelectionText( list, selection );
                ctl     = (ControlType*) GetObjectPtr( event->data.popSelect.controlID );

                CtlSetLabel( ctl, label );
                LstSetSelection( list, selection );

                if ( Prefs()->sortType != selection ) {
                    Prefs()->sortType = (SortType) selection;
                    UpdateSortMethod();
                    FrmUpdateForm( frmLibrary, frmUpdateTable );
                }
                handled = true;
            }
            break;
        }

        case keyDownEvent:
            switch ( event->data.keyDown.chr ) {
                case pageUpChr:
                    ScrollUp( VISIBLE_ROWS );
                    handled = true;
                    break;

                case pageDownChr:
                    ScrollDown( VISIBLE_ROWS );
                    handled = true;
                    break;
            }
            break;

        case frmUpdateEvent:
            if ( event->data.frmUpdate.updateCode == frmUpdateTable ) {
                ShowSortMessage();
                UpdateIndexList();
                UpdateColumns();
                HideSortMessage();

                /* if list has "shrunk" or has too few visible rows then 
                   scroll to top of list */
                if ( numberOfRows < firstVisibleRow || 
                     numberOfRows < VISIBLE_ROWS ) {
                    ScrollTo( 0 );
                }
                else {
                    ScrollTo( firstVisibleRow );
                }
                handled = true;
            }
            else if ( event->data.frmUpdate.updateCode == frmRedrawUpdateCode ) {
                TblMarkTableInvalid( table );
                TblRedrawTable( table );
                UpdateScrollbar();
                handled = true;
            }
            break;

        case menuCmdBarOpenEvent:
            MenuCmdBarAddButton( menuCmdBarOnLeft, BarDeleteBitmap, menuCmdBarResultMenuItem, 
                mOptionsDeleteAll, NULL );
            MenuCmdBarAddButton( menuCmdBarOnLeft, bmpBtnAction, menuCmdBarResultMenuItem, 
                mOptionsButton, NULL );
            MenuCmdBarAddButton( menuCmdBarOnLeft, bmpTapAction, menuCmdBarResultMenuItem, 
                mOptionsControl, NULL );
            MenuCmdBarAddButton( menuCmdBarOnLeft, bmpSettings, menuCmdBarResultMenuItem, 
                mOptionsPref, NULL );

            handled = false;
            break;

        case menuEvent:
            /* Refresh display to get rid of command text in bottom left corner */
            MenuEraseStatus( NULL );
            if ( HandleCommonMenuItems( event->data.menu.itemID ) )
                handled = true;
            else if ( event->data.menu.itemID == mOptionsDeleteAll ) {
                DeleteAllDocuments();
                handled = true;
            }
            break;

        case frmOpenEvent:
            LibraryFormInit();
            handled = true;
            break;

        case frmCloseEvent:
            ReleaseTable();
            startY  = 0;
            handled = false;
            break;

        default:
            handled = false;
    }

    RESTORE_A4 

    return handled;
}
