/*
 * $Id: document.c,v 1.51 2001/12/14 19:02:31 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "anchor.h"
#include "const.h"
#include "control.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "emailform.h"
#include "externalform.h"
#include "history.h"
#include "image.h"
#include "link.h"
#include "mainform.h"
#include "os.h"
#include "paragraph.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "screen.h"
#include "search.h"
#include "uncompress.h"
#include "util.h"


Boolean ViewRecord( const Int16 recordId, const Boolean newPage,
            Int16 pOffset ) PLKRDB_SECTION;
Int16 GetSearchAdjustment( void ) PLKRDB_SECTION;
MemHandle GetCurrentRecord( void ) PLKRDB_SECTION;
MemHandle GetRecordHandle( const Int16 recordId ) PLKRDB_SECTION;
MemHandle GetMetaRecord( void );
MemHandle GetMetaHandle( const Int16 recordId, Boolean update ) PLKRDB_SECTION;
RectangleType* ViewportBounds( void ) PLKRDB_SECTION;
void DoPageMove( const Int16 pixels ) PLKRDB_SECTION;
void ResetHeight( void ) PLKRDB_SECTION;
void ResetRecordReferences( void ) PLKRDB_SECTION;

static void DrawRecord( Header* record ) PLKRDB_SECTION;

/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define NO_SCROLLBAR    0

/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static Int16            minVerticalOffset   = 0;
static MemHandle        currentRecord       = NULL;
static MemHandle        metaRecord          = NULL;
static RectangleType    viewportBounds;
static RGBColorType     deviceForeColors; 


/* Return pointer to viewport */
RectangleType* ViewportBounds( void )
{
    return &viewportBounds;
}



/* Initialize boundaries for viewport */
static void InitViewportBounds( void )
{
    viewportBounds.topLeft.x    = TopLeftX();
    viewportBounds.topLeft.y    = TopLeftY();
    viewportBounds.extent.x     = ExtentX();
    viewportBounds.extent.y     = ExtentY();
}


/* Save the device's colors before drawing, so can later restore 
   to original when done drawing record */
static void SaveDrawState (void)
{
    RGBColorType rgb;       
        
    /* OS 3.5 or greater. Use the good API's */
    if ( sysMakeROMVersion( 3, 5, 0, sysROMStageRelease, 0 ) <= GetOSVersion() ) 
        WinPushDrawState();
        
    /* OS 3.0-3.3. Store the draw state as a private deviceForeColors variable 
       in document.c before rendering page */    
    else {
        WinSetColors( 0, &rgb, 0, 0 );  
        deviceForeColors = rgb;
    }
}


/* Restore the device's colors after done drawing record */
static void RestoreDrawState (void)
{
    /* OS 3.5 or greater. Use the good API's */
    if ( sysMakeROMVersion( 3, 5, 0, sysROMStageRelease, 0 ) <= GetOSVersion() ) 
        WinPopDrawState(); 
        
    /* OS 3.0-3.3. Restore the draw state from deviceForeColors variable */    
    else 
        WinSetColors( &deviceForeColors, 0, 0, 0 );
}


/* Update the scrollbar */
static void UpdateScrollbar
    (
    MetaRecord* meta  /* pointer to ( meta ) record */
    )
{
    Int16 maxValue;

    if ( Prefs()->scrollbar == SCROLLBAR_NONE ||
         meta->height < viewportBounds.extent.y )
        maxValue = NO_SCROLLBAR;
    else
        maxValue = meta->height - viewportBounds.extent.y;

    SclSetScrollBar( (ScrollBarType*) GetObjectPtr( frmMainScrollBar ),
        -meta->verticalOffset, 0, maxValue, GetScrollValue() );

    SclDrawScrollBar( (ScrollBarType*) GetObjectPtr( frmMainScrollBar ) );
}



/* Set the height of the paragraph */
static void SetParagraphHeight
    (
    Int16 paragraph,    /* paragraph's index */
    Int16 height        /* paragraph's height */
    )
{
    MetaRecord* meta;

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "SetParagraphHeight: MemHandleLock failed" );

    DmWrite( (void*) meta, sizeof( MetaRecord ) + 
        paragraph * sizeof( MetaParagraph ), (void*) &height, sizeof( Int16 ) );

    MemHandleUnlock( metaRecord );
}



/* Indicate that the application is working */
static void SetWorking( void )
{
    FormType* mainForm;

    if ( Prefs()->toolbar == frmMainNone )
        return;

    mainForm = FrmGetFormPtr( Prefs()->toolbar );

    FrmHideObject( mainForm, FrmGetObjectIndex( mainForm, bmpLeft ) );
    FrmHideObject( mainForm, FrmGetObjectIndex( mainForm, bmpRight ) );
    FrmHideObject( mainForm, FrmGetObjectIndex( mainForm, bmpHome ) );

    FrmShowObject( mainForm, FrmGetObjectIndex( mainForm, bmpWait ) );
}



/* Indicate that the application is idle */
static void SetIdle( void )
{
    FormType* mainForm;

    if ( Prefs()->toolbar == frmMainNone )
        return;

    mainForm = FrmGetFormPtr( Prefs()->toolbar );

    FrmHideObject( mainForm, FrmGetObjectIndex( mainForm, bmpWait ) );

    FrmShowObject( mainForm, FrmGetObjectIndex( mainForm, bmpLeft ) );
    FrmShowObject( mainForm, FrmGetObjectIndex( mainForm, bmpRight ) );
    FrmShowObject( mainForm, FrmGetObjectIndex( mainForm, bmpHome ) );
}



/* Show vertical offset for record */
static void ShowVerticalOffset
    (
    Header* record  /* current record */
    )
{
    Char        offText[ 5 ];
    Int32       percent;
    MetaRecord* meta;

    if ( Prefs()->toolbar == frmMainNone )
        return;

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "ShowVerticalOffset: MemHandleLock failed" );

    percent = 0;
    if ( 0 < meta->height ) {
        percent  = -meta->verticalOffset + ExtentY();
        percent *= 100;
        percent /= meta->height;

        if ( 100 < percent )
            percent = 100;
    }
    MemHandleUnlock( metaRecord );

    StrPrintF( offText, "%ld%%", percent );
    CtlSetLabel( (ControlType*) GetObjectPtr( frmMainPercentPopup ), offText );
}



/* Store information about the last visible paragraph */
static void SetLastVisibleParagraph
    (
    const Int16 paragraph,  /* paragraph's index */
    const Int16 y           /* paragraph's y coordinate within record */
    )
{
    MetaRecord* meta;

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "SetLastVisibleParagraph: MemHandleLock failed" );

    DmWrite( (void*) meta, OFFSETOF( MetaRecord, lastVisibleParagraph ), 
        (void*) &paragraph, sizeof( Int16 ) );
    DmWrite( (void*) meta, OFFSETOF( MetaRecord, lastParagraphY ), 
        (void*) &y, sizeof( Int16 ) );

    MemHandleUnlock( metaRecord );
    SetHistoryLastParagraph( paragraph, y );
}



/* Store information about the first visible paragraph */
static void SetFirstVisibleParagraph
    (
    const Int16 paragraph,  /* paragraph's index */
    const Int16 y           /* paragraph's y coordinate within record */
    )
{
    MetaRecord* meta;

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "SetFirstVisibleParagraph: MemHandleLock failed" );

    DmWrite( (void*) meta, OFFSETOF( MetaRecord, firstVisibleParagraph ),
        (void*) &paragraph, sizeof( Int16 ) );

    DmWrite( (void*) meta, OFFSETOF( MetaRecord, firstParagraphY ), 
        (void*) &y, sizeof( Int16 ) );

    MemHandleUnlock( metaRecord );
    SetHistoryFirstParagraph( paragraph, y );
}



/* Initialize text context */
static void InitializeTextContext
    (
    TextContext* tContext   /* pointer to text context */
    )
{
    tContext->cursorX       = viewportBounds.topLeft.x;
    tContext->cursorY       = viewportBounds.topLeft.y;
    tContext->writeMode     = true;
    tContext->activeAnchor  = false;
    tContext->patternOffset = -1;
}



/* Draw the record at most until the end of the visible screen */
static void DrawRecord
    (
    Header* record  /* pointer to record header */
    )
{
    Paragraph*      paragraph;
    MetaParagraph*  metaParagraph;
    UInt16          numOfParagraphs;
    Boolean         needNewParagraph;
    Int16           firstVisiblePixel;
    Int16           lastVisiblePixel;
    Int16           cursor;
    Int16           lastCursor;
    Int16           paragraph_index;
    TextContext     thisContext;
    MetaRecord*     meta;

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "DrawRecord: MemHandleLock failed" );

    InitializeTextContext( &thisContext );

    paragraph_index     = meta->firstVisibleParagraph;
    numOfParagraphs     = record->paragraphs - paragraph_index;

    paragraph           = GET_PARAGRAPH( record, paragraph_index );
    metaParagraph       = GET_METAPARAGRAPH( meta, paragraph_index );

    firstVisiblePixel   = -meta->verticalOffset;
    lastVisiblePixel    = viewportBounds.extent.y - meta->verticalOffset;

    cursor              = meta->firstParagraphY;
    thisContext.cursorY = cursor + meta->verticalOffset + 
                          viewportBounds.topLeft.y;

    needNewParagraph    = false;


    /* If PalmOS 3+, and depth > 1, then save colors' state for a later restore */    
    if ( sysGetROMVerMajor( GetOSVersion() ) != 0x02 && Prefs()->screenDepth != 1 ) 
        SaveDrawState();        
        
    WinSetClip( &viewportBounds );

    while ( numOfParagraphs-- ) {
        lastCursor = cursor;

        DrawParagraph( &thisContext, paragraph, record );

        cursor += metaParagraph->height;

        /*
            Save pointers to the first and last visible paragraphs. It is
            possible that this is the same paragraph, so be careful.
     
            Also note that the very first paragraph in the record is
            the initial "firstParagraph", so we don't need code to
            deal with that special case.
        */
        if ( cursor < firstVisiblePixel )
            needNewParagraph = true;
        else if ( needNewParagraph ) {
            SetFirstVisibleParagraph( paragraph_index, lastCursor );
            needNewParagraph = false;
        }
        if ( lastVisiblePixel < cursor ) {
            SetLastVisibleParagraph( paragraph_index, lastCursor );
            break;
        }
        paragraph++;
        metaParagraph++;
        paragraph_index++;
    }
    MemHandleUnlock( metaRecord );
    WinResetClip();
    
    /* Finished drawing page, restore device's colors */    
    if ( sysGetROMVerMajor( GetOSVersion() ) != 0x02 && Prefs()->screenDepth != 1 )
        RestoreDrawState();        
}



/* Scroll record up */
static void ScrollUp
    (
    Header* record  /* pointer to record header */
    )
{
    Int16           firstVisiblePixel;
    Int16           visibleParagraphPixel;
    Int16           paragraph_index;
    MetaParagraph*  metaParagraph;
    MetaRecord*     meta;

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "ScrollUp: MemHandleLock failed" );

    firstVisiblePixel   = -meta->verticalOffset;
    paragraph_index     = meta->firstVisibleParagraph;

    /* Work out the new "first visible" paragraph by moving down */
    metaParagraph           = GET_METAPARAGRAPH( meta, paragraph_index );
    visibleParagraphPixel   = meta->firstParagraphY + metaParagraph->height;

    while ( visibleParagraphPixel < firstVisiblePixel ) {
        paragraph_index++;
        metaParagraph++;
        visibleParagraphPixel += metaParagraph->height;
    }
    visibleParagraphPixel -= metaParagraph->height;

    SetFirstVisibleParagraph( paragraph_index, visibleParagraphPixel );

    MemHandleUnlock( metaRecord );
    DrawRecord( record );
}



/* Scroll record down */
static void ScrollDown
    (
    Header*     record, /* pointer to record header */
    const Int16 height  /* height of scrolled area */
    )
{
    Int16           firstVisiblePixel;
    Int16           lastVisiblePixel;
    Int16           visibleParagraphPixel;
    Int16           paragraph_index;
    MetaParagraph*  metaParagraph;
    MetaRecord*     meta;

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "ScrollDown: MemHandleLock failed" );

    firstVisiblePixel   = -meta->verticalOffset;
    paragraph_index     = meta->firstVisibleParagraph;

    /* Work out the new "first visible" paragraph by searching up */
    metaParagraph           = GET_METAPARAGRAPH( meta, paragraph_index );
    visibleParagraphPixel   = meta->firstParagraphY;

    while ( firstVisiblePixel < visibleParagraphPixel ) {
        paragraph_index--;
        metaParagraph--;
        visibleParagraphPixel -= metaParagraph->height;
    }
    SetFirstVisibleParagraph( paragraph_index, visibleParagraphPixel );

    lastVisiblePixel    = viewportBounds.extent.y - meta->verticalOffset;
    paragraph_index     = meta->lastVisibleParagraph;

    /* Work out the new "last visible" paragraph by searching up */
    metaParagraph           = GET_METAPARAGRAPH( meta, paragraph_index );
    visibleParagraphPixel   = meta->lastParagraphY;

    while ( lastVisiblePixel < visibleParagraphPixel ) {
        paragraph_index--;
        metaParagraph--;
        visibleParagraphPixel -= metaParagraph->height;
    }
    SetLastVisibleParagraph( paragraph_index, visibleParagraphPixel );

    MemHandleUnlock( metaRecord );
    DrawRecord( record );
}



/* Adjust vertical offset of record */
static void AdjustVerticalOffset
    (
    Header* record,     /* pointer to record header */
    Int16   adjustment  /* adjustment in pixels */
    )
{
    RectangleType   vacatedRectangle;
    MetaRecord      newRecord;
    Int16           adjustAmount;
    MetaRecord*     meta;

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "AdjustVerticalOffset: MemHandleLock failed" );

    /* Scrolling of Tbmp records is handled by another function... */
    if ( meta->height == 0 ) {
        MemHandleUnlock( metaRecord );
        return;
    }

    /* Adjust the record */
    newRecord = *meta;
    if ( 0 <= newRecord.verticalOffset + adjustment ) {
        if ( 0 <= newRecord.verticalOffset ) {
            MemHandleUnlock( metaRecord );
            return;
        }
        else
            adjustment = -newRecord.verticalOffset;
    }
    else if ( newRecord.verticalOffset + adjustment < minVerticalOffset ) {
        if ( newRecord.verticalOffset <= minVerticalOffset ) {
            adjustment = 0;
        }
        else
            adjustment = minVerticalOffset - newRecord.verticalOffset;
    }
    adjustAmount                = abs( adjustment );
    newRecord.verticalOffset   += adjustment;
    SetHistoryVerticalOffset( newRecord.verticalOffset );

    DmWrite( (void*) meta, OFFSETOF( MetaRecord, verticalOffset ), 
        (void*) &newRecord.verticalOffset, sizeof( Int16 ) );

    /* Scroll the window */
    vacatedRectangle.topLeft.x  = 0;
    vacatedRectangle.topLeft.y  = 0;
    vacatedRectangle.extent.x   = 0;
    vacatedRectangle.extent.y   = 0;

    WinSetClip( &viewportBounds );
    WinScrollRectangle( &viewportBounds, ( adjustment < 0 ) ? winUp : winDown, adjustAmount, &vacatedRectangle );
    WinEraseRectangle( &vacatedRectangle, 0 );

    /* Adjust the on-screen anchors. Be careful to do this BEFORE drawing
        the new anchors! */
    AdjustVisibleAnchors( adjustment );

    /* This code also draws the new anchors */
    if ( adjustment < 0 )
        ScrollUp( record );
    else
        ScrollDown( record, adjustment );

    UpdateScrollbar( &newRecord );

    MemHandleUnlock( metaRecord );
    ShowVerticalOffset( record );

    WinResetClip();
}


/* Find the necessary adjustment for the search pattern to be visible in the record */
Int16 GetSearchAdjustment( void )
{
    Header*     record;
    MetaRecord* meta;
    Paragraph*  paragraph;
    TextContext thisContext;
    UInt16      numOfParagraphs;
    Int16       adjustment;

    record = (Header*) MemHandleLock( currentRecord );
    ErrFatalDisplayIf( record == NULL, "GetSearchAdjustment: MemHandleLock failed" );

    numOfParagraphs = record->paragraphs;

    InitializeTextContext( &thisContext );
    thisContext.writeMode       = false;
    thisContext.patternOffset   = 0;

    paragraph = GET_PARAGRAPH( record, 0 );

    while ( thisContext.patternOffset == 0 && numOfParagraphs-- ) {
        DrawParagraph( &thisContext, paragraph, record );
        paragraph++;
    }

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "GetSearchAdjustment: MemHandleLock failed" );

    adjustment = -( meta->verticalOffset + thisContext.patternOffset );

    MemHandleUnlock( metaRecord );
    MemHandleUnlock( currentRecord );

    return adjustment;
}



/* Calculate the total height of the record and store it in the record header */
static Int16 CalculateHeight
    (
    Header* record,     /* pointer to record header */
    Int16   pOffset     /* offset to first paragraph */
    )
{
    Paragraph*  paragraph;
    TextContext thisContext;
    UInt16      numOfParagraphs;
    Int16       totalHeight;
    Int16       paragraph_index;

    InitializeTextContext( &thisContext );

    if ( pOffset != NO_OFFSET )
        thisContext.writeMode = false;

    paragraph = GET_PARAGRAPH( record, 0 );

    WinSetClip( &viewportBounds );

    paragraph_index = 0;
    totalHeight     = 0;
    numOfParagraphs = record->paragraphs;
    while ( numOfParagraphs-- ) {
        Int16 paragraphHeight;
        Int16 lastCursor;

        if ( pOffset == NO_OFFSET )
            InitializeTextContext( &thisContext );

        pOffset--;

        lastCursor = thisContext.cursorY;

        DrawParagraph( &thisContext, paragraph, record );

        paragraphHeight = thisContext.cursorY - lastCursor;
        totalHeight    += paragraphHeight;

        SetParagraphHeight( paragraph_index, paragraphHeight );

        paragraph++;
        paragraph_index++;
    }
    WinResetClip();

    return totalHeight;
}



/* Find the named anchor paragraph */
static void HandleNamedAnchor
    (
    Header* record, /* pointer to record header */
    Int16   pOffset /* offset to first paragraph */
    )
{
    Int16           visibleParagraphPixel;
    Int16           heightDiff;
    Int16           verticalOffset;
    Int16           i;
    Int16           paragraph_index;
    MetaParagraph*  metaParagraph;
    MetaRecord*     meta;

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "HandleNamedAnchor: MemHandleLock failed" );

    WinEraseRectangle( &viewportBounds, 0 );

    heightDiff = meta->height - viewportBounds.extent.y;
    if ( heightDiff <= 0 ) {
        MemHandleUnlock( metaRecord );
        return;
    }

    paragraph_index         = meta->firstVisibleParagraph;
    visibleParagraphPixel   = meta->firstParagraphY;
    metaParagraph           = GET_METAPARAGRAPH( meta, paragraph_index );

    /* Find named anchor paragraph */
    for ( i = 0; i < pOffset; i++ ) {
        visibleParagraphPixel  += metaParagraph->height;
        metaParagraph++;
        paragraph_index++;
    }

    verticalOffset = -visibleParagraphPixel;

    if ( heightDiff < visibleParagraphPixel ) {
        Int16 firstVisiblePixel;

        firstVisiblePixel = meta->height - viewportBounds.extent.y;

        metaParagraph--;

        visibleParagraphPixel -= metaParagraph->height;

        paragraph_index--;

        while ( firstVisiblePixel < visibleParagraphPixel ) {
            metaParagraph--;

            visibleParagraphPixel -= metaParagraph->height;

            paragraph_index--;
        }
        verticalOffset = viewportBounds.extent.y - meta->height;
    }
    SetFirstVisibleParagraph( paragraph_index, visibleParagraphPixel );
    SetHistoryVerticalOffset( verticalOffset );
    DmWrite( (void*) meta, OFFSETOF( MetaRecord, verticalOffset ), 
        (void*) &verticalOffset, sizeof( Int16 ) );

    MemHandleUnlock( metaRecord );
}



/* View Plucker HTML record */
static Boolean ViewPHTML
    (
    Header* record,     /* pointer to record header */
    Boolean newPage,    /* true if the page is not from the history */
    Int16   pOffset     /* offset to first paragraph */
    )
{
    MetaRecord* meta;

    if ( record->type == DATATYPE_PHTML_COMPRESSED ) {
        if ( Uncompress( record ) == NULL )
            return false;
    }

    meta = (MetaRecord*) MemHandleLock( metaRecord );
    ErrFatalDisplayIf( meta == NULL, "ViewPHTML: MemHandleLock failed" );

    if ( newPage || meta->height == 0 ) {
        MetaRecord recordData;

        recordData                = *meta;
        recordData.verticalOffset = 0;

        DmWrite( (void*) meta, OFFSETOF( MetaRecord, verticalOffset ), 
            (void*) &recordData.verticalOffset, sizeof( Int16 ) );

        SetHistoryVerticalOffset( recordData.verticalOffset );
        SetFirstVisibleParagraph( 0, 0 );
    }
    if ( meta->height == 0 ) {
        MetaRecord recordData;

        recordData = *meta;

        SetWorking();
        recordData.height = CalculateHeight( record, pOffset );
        SetIdle();

        DmWrite( (void*) meta, OFFSETOF( MetaRecord, height ), 
            (void*) &recordData.height, sizeof( Int16 ) );
    }
    if ( pOffset != NO_OFFSET )
        HandleNamedAnchor( record, pOffset );

    UpdateScrollbar( meta );
    ShowVerticalOffset( record );

    minVerticalOffset = viewportBounds.extent.y - meta->height;

    MemHandleUnlock( metaRecord );

    DrawRecord( record );

    return true;
}



/* Move page up/down */
void DoPageMove
    (
    const Int16 pixels  /* number of pixels to move ( positive values move up, negative down ) */
    )
{
    Header* record;

    record = (Header*) MemHandleLock( currentRecord );
    ErrFatalDisplayIf( record == NULL, "DoPageMove: MemHandleLock failed" );

    InitViewportBounds();

    /* if image record we return to the previous record when the user press up/down */
    if ( record->type == DATATYPE_TBMP || record->type == DATATYPE_TBMP_COMPRESSED ) {
        Int16 previousRecord;

        previousRecord = GetHistoryPrev();
        if ( previousRecord != NOT_FOUND ) {
            MemHandleUnlock( currentRecord );
            ViewRecord( previousRecord, false, NO_OFFSET );
            SetVisitedLink( previousRecord );
            return;
        }
    }

    AdjustVerticalOffset( record, pixels );

    MemHandleUnlock( currentRecord );
}



/* Return a handle to the meta data */
MemHandle GetMetaHandle
    (
    const Int16 recordId,   /* record ID */
    Boolean     update      /* if true, update meta data */
    )
{
    MemHandle   meta;
    Header*     record;
    UInt16      numOfParagraphs;

    record          = (Header*) MemHandleLock( GetRecordHandle( recordId ) );
    numOfParagraphs = record->paragraphs;
    MemPtrUnlock( record );

    if ( ReturnMetaHandle( recordId, numOfParagraphs, update, &meta ) != errNone )
        ErrDisplay( "GetMetaHandle: fatal error" );

    return meta;
}



/* Return a handle to a record */
MemHandle GetRecordHandle
    (
    const Int16 recordId  /* record ID */
    )
{
    MemHandle record;

    if ( ReturnRecordHandle( recordId, &record ) != errNone )
        if ( ReturnRecordHandle( PLUCKER_LINKS_ID, &record ) != errNone )
            record = NULL;

    return record;
}



/* View record ( or image ) */
Boolean ViewRecord
    (
    const Int16     recordId,   /* record ID */
    const Boolean   newPage,    /* indicates if the page is from the history or not */
    Int16           pOffset     /* offset to first paragraph */
    )
{
    Boolean status;
    Header* record;

    MSG( _( "record #%d\n", recordId ) );

    ClearVisibleAnchors();
    InitViewportBounds();
    DeleteOffScreenWindow();
    WinEraseRectangle( &viewportBounds, 0 );

    currentRecord = GetRecordHandle( recordId );
    if ( currentRecord == NULL ) {
        SetLinkIndex( recordId );
        FrmGotoForm( frmExternalLinks );

        return true;
    }

    status = false;

    record = (Header*) MemHandleLock( currentRecord );
    ErrFatalDisplayIf( record == NULL, "ViewRecord: MemHandleLock failed" );

    switch ( record->type ) {
        case DATATYPE_PHTML:
        case DATATYPE_PHTML_COMPRESSED:
            if ( newPage )
                AddToHistory( recordId );

            metaRecord = GetMetaHandle( recordId, true );
            ErrFatalDisplayIf( metaRecord == NULL, "ViewRecord: NULL record" );

            status = ViewPHTML( record, newPage, pOffset );
            break;

        case DATATYPE_TBMP:
        case DATATYPE_TBMP_COMPRESSED:
            if ( newPage )
                AddToHistory( recordId );

            metaRecord = GetMetaHandle( recordId, true );
            ErrFatalDisplayIf( metaRecord == NULL, "ViewRecord: NULL record" );

            status = ViewTbmp( record, newPage );
            break;

        case DATATYPE_MAILTO:
            SetMailto( recordId );
            FrmGotoForm( frmEmail );

            status = true;
            break;

        case DATATYPE_LINK_INDEX:
            SetLinkIndex( recordId );
            FrmGotoForm( frmExternalLinks );

            status = true;
            break;

        default:
            ErrDisplay( "ViewRecord: unknown record type" );
            break;
    }
    MemHandleUnlock( currentRecord );

    return status;
}



/* Set height of record to zero */
void ResetHeight( void )
{
    if ( metaRecord != NULL ) {
        Int16       height;
        MetaRecord* meta;

        meta = (MetaRecord*) MemHandleLock( metaRecord );
        ErrFatalDisplayIf( meta == NULL, "ResetHeight: MemHandleLock failed" );

        height = 0;
        DmWrite( (void*) meta, OFFSETOF( MetaRecord, height ), 
            (void*) &height, sizeof( Int16 ) );

        MemHandleUnlock( metaRecord );
    }
}



/* reset the record references (e.g. when the document is closed) */
void ResetRecordReferences( void )
{
    currentRecord   = NULL;
    metaRecord      = NULL;
}



/* Return handle to current record */
MemHandle GetCurrentRecord( void )
{
    return currentRecord;
}



/* Return handle to meta record */
MemHandle GetMetaRecord( void )
{
    return metaRecord;
}

