/*
 * $Id: prefsdata.c,v 1.39 2001/11/26 20:09:02 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "debug.h"
#include "prefsdata.h"
#include "screen.h"


void ReadPrefs( void ) PLKRDB_SECTION;
void WritePrefs( void ) PLKRDB_SECTION;
Preferences *Prefs( void ) PLKRDB_SECTION;
void GetSearchString( Char* string ) PLKRDB_SECTION;
void AddSearchString( Char* string ) PLKRDB_SECTION;
Char *GetSearchPatterns( void ) PLKRDB_SECTION;
void ReleaseSearchPatterns( void ) PLKRDB_SECTION;

static LocalID CheckDocument( LocalID* db ) PLKRDB_SECTION;
static void RemovePref( UInt16 prefID ) PLKRDB_SECTION;
static void ReadSearchPatterns( void ) PLKRDB_SECTION;
static void WriteSearchPatterns( void ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define ViewerOldPrefID             0
#define ViewerPrefID                2
#define ViewerOldPrefSearchStringID 1
#define ViewerPrefSearchStringID    3


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static Preferences  pref;
static Char*        patterns        = NULL;
static UInt16       patternsSize    = 0;



/* Check ID for document */
static LocalID CheckDocument
    (
    LocalID* db /* pointer to the document ID */
    )
{
    DmSearchStateType   stateInfo;
    UInt16              card;
    UInt16              firstCard;
    LocalID             id;
    LocalID             firstId;
    Err                 err;

    err = DmGetNextDatabaseByTypeCreator( true, &stateInfo,
            (UInt32) ViewerDocumentType, (UInt32) ViewerAppID, false, &card, &id );

    firstId     = id;
    firstCard   = card;

    if ( *db != NULL ) {
        while ( err == errNone ) {
            if ( *db == id ) {
                pref.cardNo = card;
                return *db;
            }
            err = DmGetNextDatabaseByTypeCreator( false, &stateInfo,
                    (UInt32) ViewerDocumentType, (UInt32) ViewerAppID, false, &card, &id );
        }
    }
    pref.cardNo = firstCard;
    return ( *db = firstId );
}



/* Remove preferences from the Preferences database */
static void RemovePref
    (
    UInt16 prefID   /* preference to remove */
    )
{
    DmOpenRef           ref;
    DmSearchStateType   state;
    LocalID             dbID;
    UInt16              cardNo;
    Int16               index;

    DmGetNextDatabaseByTypeCreator( true, &state, 'sprf', 'psys', false, &cardNo, &dbID );
    ref = DmOpenDatabase( cardNo, dbID, dmModeReadWrite );
    if ( ref == NULL )
        return;

    index = DmFindResource( ref, ViewerAppID, prefID, NULL );
    if ( index != -1 )
        DmRemoveResource( ref, index );

    DmCloseDatabase( ref );
}



/* Retrieve preferences from the Preferences database */
void ReadPrefs( void )
{
    UInt16 prefSize = sizeof( Preferences );
    UInt16 tempSize = 0;

    MemSet( &pref, sizeof( Preferences ), 0 );

    if ( PrefGetAppPreferences( (UInt32) ViewerAppID, (UInt16) ViewerPrefID,
            &pref, &prefSize, true) == noPreferenceFound ) {
        if ( PrefGetAppPreferences ((UInt32) ViewerAppID,
                (UInt16) ViewerOldPrefID, NULL, &tempSize, true ) != noPreferenceFound )
            RemovePref( ViewerOldPrefID );

        RemovePref( ViewerPrefSearchStringID );

        pref.dbID                       = NULL;
        pref.strikethrough              = false;
        pref.showDate                   = true;
        pref.showSize                   = true;
        pref.multipleSelect             = true;
        pref.linkClick                  = true;
        pref.hardKeys                   = false;
        pref.gestures                   = false;
        pref.searchInAllPages           = false;
        pref.caseSensitive              = false;
        pref.toolbar                    = frmMainTop;
        pref.lastForm                   = frmLibrary;
        pref.scrollbar                  = SCROLLBAR_RIGHT;
        pref.controlMode                = MODE1;
        pref.screenDepth                = 0;
        pref.searchEntries              = 0;
        pref.categories                 = 1;
        pref.cardNo                     = 0;
        pref.toolbarButton              = 0;
        pref.filterMode                 = FILTER_OR;
        pref.fontMode                   = FONT_DEFAULT;
        pref.underlineMode              = false;
        pref.sortType                   = SORT_NAME;
        pref.sortOrder                  = SORTORDER_ASC;
        pref.autoscrollEnabled          = false;
        pref.autoscrollInterval         = 0;
        pref.autoscrollLastScrollTime   = 0;
        pref.autoscrollJump             = 1;
        pref.autoscrollMode             = AUTOSCROLL_PIXELS;
        pref.autoscrollDir              = AUTOSCROLL_DOWN;
        pref.autoscrollStayOn           = false;   

        StrCopy( pref.docName, "" );

        MemSet( &pref.select, sizeof( pref.select ), SELECT_NONE );
        MemSet( &pref.hwMode, sizeof( pref.hwMode ), SELECT_NONE );
        MemSet( &pref.gestMode, sizeof( pref.gestMode ), SELECT_NONE );
    }
    pref.dbID = DmFindDatabase( pref.cardNo, pref.docName );
    CheckDocument( &pref.dbID );
}



/* Store preferences in the Preferences database */
void WritePrefs( void )
{
    PrefSetAppPreferences( (UInt32) ViewerAppID, (UInt16) ViewerPrefID,
        (Int16) ViewerVersion, &pref, sizeof( Preferences ), true );
}



/* Retrieve search patterns */
static void ReadSearchPatterns( void )
{
    UInt16 tempSize;

    tempSize        = 0;
    patternsSize    = 0;

    if ( PrefGetAppPreferences( (UInt32) ViewerAppID,
            (UInt16) ViewerPrefSearchStringID, NULL, &patternsSize,
            true ) == noPreferenceFound ) {
        if ( PrefGetAppPreferences( (UInt32) ViewerAppID,
                (UInt16) ViewerOldPrefSearchStringID, NULL, &tempSize,
                true ) != noPreferenceFound )
            RemovePref( ViewerOldPrefSearchStringID );
    }
    else {
        if ( patterns != NULL )
            MemPtrFree( patterns );
        patterns = (Char*) MemPtrNew( patternsSize );
        PrefGetAppPreferences( (UInt32) ViewerAppID,
            (UInt16) ViewerPrefSearchStringID, patterns, &patternsSize, true );
    }
}



/* Store search patterns */
static void WriteSearchPatterns( void )
{
    if ( patterns != NULL ) {
        PrefSetAppPreferences( (UInt32) ViewerAppID,
            (UInt16) ViewerPrefSearchStringID, (Int16) ViewerVersion, patterns,
            patternsSize, true );

        MemPtrFree( patterns );
        patterns = NULL;
    }
}



/* Retrieve search string */
void GetSearchString
    (
    Char* string    /* string pointer to store result */
    )
{
    MemSet( string, MAX_PATTERN_LEN + 1, 0 );

    ReadSearchPatterns();

    if ( patterns != NULL ) {
        Int8   i;
        Char*   temp;

        temp = patterns;
        for ( i = 0; i < pref.searchEntries - 1; i++ )
            temp += StrLen( temp ) + 1;
        StrNCopy( string, temp, MAX_PATTERN_LEN );

        MemPtrFree( patterns );
        patterns = NULL;
    }
    else {
        pref.searchEntries = 0;
        StrCopy( string, "" );
    }
}



/* Add new search string to patterns list */
void AddSearchString
    (
    Char* string    /* pattern string */
    )
{
    UInt8 item;
    UInt8 count;
    Char* temp1;
    Char* temp2;
    Char* newPatterns;

    ReadSearchPatterns();

    temp1       = NULL;
    temp2       = NULL;
    newPatterns = NULL;

    /* Add string to patterns list */
    if ( pref.searchEntries < 10 ) {
        patternsSize   += StrLen( string ) + 1;
        newPatterns     = (Char*) MemPtrNew( patternsSize );
        temp1           = patterns;
        temp2           = newPatterns;
        item            = 0;
    }
    /* Replace first string in patterns list with the new string */
    else {
        patternsSize   -= StrLen( patterns ) - StrLen( string );
        newPatterns     = (Char*) MemPtrNew( patternsSize );
        temp1           = patterns + StrLen( patterns ) + 1;
        temp2           = newPatterns;
        item            = 1;
    }

    count = 0;
    while ( item++ < pref.searchEntries ) {
        if ( StrCompare( temp1, string ) ) {
            StrCopy( temp2, temp1 );
            temp2 += StrLen( temp2 ) + 1;
            count++;
        }
        else
            patternsSize -= StrLen( string ) + 1;

        temp1 += StrLen( temp1 ) + 1;
    }
    StrCopy( temp2, string );
    pref.searchEntries = count + 1;
    if ( patterns != NULL )
        MemPtrFree( patterns );
    patterns = newPatterns;

    WriteSearchPatterns();
}



/* Retrieve search patterns, call ReleaseSearchPatterns to release the memory
   allocated by this function */
Char* GetSearchPatterns( void )
{
    ReadSearchPatterns();
    return patterns;
}



/* Release memory allocated by GetSearchPatterns */
void ReleaseSearchPatterns( void )
{
    if ( patterns != NULL ) {
        MemPtrFree( patterns );
        patterns = NULL;
    }
}



/* Retrieve the value of a preference ( through the pointer ) */
Preferences* Prefs( void )
{
    return &pref;
}
