//----------------------------------------------------------------------------------------
// Name:        channel_dialog.cpp
// Purpose:     Configure channel dialog
// Author:      Robert O'Connor
// Modified by:
// Created:     2001/10/20
// Copyright:   (c) Robert O'Connor ( rob@medicalmnemonics.com )
// Licence:     GPL
// RCS-ID:      $Id: channel_dialog.cpp,v 1.11 2002/01/03 06:02:27 robertoconnor Exp $
//----------------------------------------------------------------------------------------

// ---------------------------------------------------------------------------------------
// Headers
// ---------------------------------------------------------------------------------------
#ifdef __GNUG__
    #pragma implementation "channel_dialog.cpp"
    #pragma interface "channel_dialog.cpp"
#endif
// ---------------------------------------------------------------------------------------
// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// For all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWindows headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif
// ---------------------------------------------------------------------------------------
#include "wx/radiobut.h"                // wxRadioButton
#include "wx/spinctrl.h"                // wxSpinCtrl
#include "wx/calctrl.h"                 // wxCalendarCtrl
// ---------------------------------------------------------------------------------------
#include "wx/xrc/xmlres.h"              // XRC XML resouces
// ---------------------------------------------------------------------------------------
#include "channel_dialog.h"
#include "showcase_dialog.h"
#include "editor_dialog.h"
#include "exclusion_dialog.h"
#include "configuration.h"
#include "utils_datetime.h"

// ---------------------------------------------------------------------------------------
// Private variables
// ---------------------------------------------------------------------------------------

channel_dialog* the_channel_dialog = NULL;
const long ONLY_THUMBNAIL   = 0;            // For alt_maxwidth/alt_maxheight
const long ALWAYS_LINK      = 1000000;      // For alt_maxwidth/alt_maxheight

// ---------------------------------------------------------------------------------------
// Event table: connect the events to the handler functions to process them
// ---------------------------------------------------------------------------------------

BEGIN_EVENT_TABLE(channel_dialog, wxDialog)
    EVT_UPDATE_UI( XMLID( "channel_dialog_start_tab_url_radiobutton" ), channel_dialog::on_start_tab_url_radiobutton )
    EVT_UPDATE_UI( XMLID( "channel_dialog_start_tab_file_radiobutton" ), channel_dialog::on_start_tab_file_radiobutton )
    EVT_UPDATE_UI( XMLID( "channel_dialog_start_tab_links_radiobutton" ), channel_dialog::on_start_tab_links_radiobutton )
    EVT_BUTTON( XMLID( "channel_dialog_start_tab_url_button" ), channel_dialog::on_start_tab_url_button )
    EVT_BUTTON( XMLID( "channel_dialog_start_tab_file_button" ), channel_dialog::on_start_tab_file_button )
    EVT_BUTTON( XMLID( "channel_dialog_start_tab_links_button" ), channel_dialog::on_start_tab_links_button )

    EVT_BUTTON( XMLID( "channel_dialog_limits_tab_exclusion_button" ), channel_dialog::on_limits_tab_exclusion_button )
    EVT_UPDATE_UI( XMLID( "channel_dialog_limits_tab_staybelow_checkbox" ), channel_dialog::on_limits_tab_staybelow_checkbox )

    EVT_UPDATE_UI( XMLID( "channel_dialog_images_tab_include_images_checkbox" ), channel_dialog::on_images_tab_include_images_checkbox )
    EVT_UPDATE_UI( XMLID( "channel_dialog_images_tab_only_link_radiobutton" ), channel_dialog::on_images_tab_only_link_radiobutton )
    
    EVT_UPDATE_UI( XMLID( "channel_dialog_output_tab_only_compress_checkbox" ), channel_dialog::on_output_tab_only_compress_checkbox )
    EVT_UPDATE_UI( XMLID( "channel_dialog_output_tab_category_checkbox" ), channel_dialog::on_output_tab_category_checkbox )
    EVT_UPDATE_UI( XMLID( "channel_dialog_output_tab_launchable_checkbox" ), channel_dialog::on_output_tab_launchable_checkbox )
    EVT_UPDATE_UI( XMLID( "channel_dialog_output_tab_custom_large_icon_checkbox" ), channel_dialog::on_output_tab_custom_large_icon_checkbox )
    EVT_UPDATE_UI( XMLID( "channel_dialog_output_tab_custom_small_icon_checkbox" ), channel_dialog::on_output_tab_custom_small_icon_checkbox )
    EVT_BUTTON( XMLID( "channel_dialog_output_tab_custom_small_icon_button" ), channel_dialog::on_output_tab_custom_small_icon_button )
    EVT_BUTTON( XMLID( "channel_dialog_output_tab_custom_large_icon_button" ), channel_dialog::on_output_tab_custom_large_icon_button )
    
    EVT_BUTTON( XMLID( "channel_dialog_sync_tab_sync_directory_button" ), channel_dialog::on_sync_tab_sync_directory_button )
    EVT_UPDATE_UI( XMLID( "channel_dialog_sync_tab_specify_filename_radiobutton" ), channel_dialog::on_sync_tab_specify_filename_radiobutton )
    EVT_UPDATE_UI( XMLID( "channel_dialog_sync_tab_command_before_checkbox" ), channel_dialog::on_sync_tab_command_before_checkbox)
    EVT_UPDATE_UI( XMLID( "channel_dialog_sync_tab_command_after_checkbox" ), channel_dialog::on_sync_tab_command_after_checkbox)
   
    EVT_BUTTON( XMLID( "channel_dialog_scheduling_tab_now_button" ), channel_dialog::on_scheduling_tab_now_button )

    EVT_BUTTON( wxID_OK, channel_dialog::OnOK )
END_EVENT_TABLE()

//----------------------------------------------------------------------------------------
// Non-event handler functions
//----------------------------------------------------------------------------------------

void init_a_channel_dialog( wxWindow* parent, wxString configuration_section )
{
    wxLogDebug( "Starting to create dialog" );
    wxString string;
    
    the_channel_dialog = new channel_dialog; 
    wxTheXmlResource->LoadDialog( the_channel_dialog, parent, "channel_dialog" );
   
    // Set the path (section) to look for the values to the passed section name. For example
    // if passed "news", a call to the_configuration->Read("bpp"...) will now get the 
    // value of /news/bpp. It also creates this section if it doesn't exist yet in the
    // config file.    
    the_configuration->SetPath( configuration_section );
    wxLogDebug( "Set configuration section" );
    
    // Store the configuration section as a class member to use for things like 
    // which exclusion list to load.
    the_channel_dialog->m_configuration_section = configuration_section;
    
    // Read and set the saved values for the controls from the configuration file
    the_channel_dialog->read_configuration();
    
    the_channel_dialog->ShowModal(); 
    
    // ShowModal() halts program flow. So when dialog is done, set the default config path
    // (section) back to '/' just to clean up.
    the_configuration->SetPath( '/' );
}

// Read and set the saved values for the controls from the channel-specific section of
// the configuration file
void channel_dialog::read_configuration()
{
    wxLogDebug( "Starting to read configuration file" ); 
    
    wxString key;
    wxString doc_name;
    
    key = "doc_name";

    // See comments in main_dialog::listctrl_load_rows() about doc_name
    if ( ! the_configuration->Exists( key ) )
        key = "db_name";

    if ( ! the_configuration->Exists( key ) )
        key = "doc_file";

    if ( ! the_configuration->Exists( key ) )
        key = "db_file";

    doc_name = the_configuration->Read( key, _T( "Unnamed Channel" ) );    
    // If a file key, strip off the path and extension, leaving only the basename
    if ( key == "doc_file" || key == "db_file" ) 
        wxSplitPath( doc_name.c_str(), NULL, &doc_name, NULL );  

    XMLCTRL( *the_channel_dialog, "channel_dialog_textctrl", wxTextCtrl )
        ->SetValue( doc_name );
    wxLogDebug( "Initialized doc_name" ); 
    
    //------Start Tab-------------------------------------------------------------------
    
    wxString home_url;

    home_url = the_configuration->Read( "home_url", _T( "" ) );    
    // If no url specified: must be the default plucker home.html HTML page
    if ( home_url == "" || home_url == "plucker:/home.html" ||
         ( home_url.Contains( "plucker:" ) &&
           home_url.Contains ( "channels" ) &&
           home_url.Contains( "home.html" ) ) ) {    
        XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_links_radiobutton", wxRadioButton )
            ->SetValue( TRUE );    
    // Else a file: or plucker: protocol
    } else if ( home_url.StartsWith( "file:" ) || home_url.StartsWith( "plucker:" ) ) {    
        XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_file_radiobutton", wxRadioButton )
            ->SetValue( TRUE );   
        XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_file_textctrl", wxTextCtrl )
            ->SetValue( home_url );    
    // Everything else into a website address
    } else {
        XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_url_radiobutton", wxRadioButton )
            ->SetValue( TRUE );   
        XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_url_textctrl", wxTextCtrl )
            ->SetValue( home_url );  
    }    
    wxLogDebug( "Initialized home_url" );
      
    //------Limits Tab------------------------------------------------------------------

    XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_maxdepth_spinctrl", wxSpinCtrl )
        ->SetValue( (int) the_configuration->Read( "home_maxdepth", 2l ) ); 
    wxLogDebug( "Initialized home_maxdepth" ); 

    XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_stayonhost_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "home_stayonhost", 0L ) );
    wxLogDebug( "Initialized home_stayonhost" ); 

    wxString staybelow;
    
    staybelow = the_configuration->Read( "home_staybelow", _T("") );
    XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_staybelow_textctrl", wxTextCtrl )
        ->SetValue( staybelow );
    if ( staybelow != _T("") ) {
        XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_staybelow_checkbox", wxCheckBox )
            ->SetValue ( TRUE );
    }
    wxLogDebug( "Initialized home_staybelow" );
        
    // Exclusion list handled a bit differently. No key means exclusion list is on, ,
    // and will look in /<OS-SPECIFIC>/exclusion_lists to get the exclusion list files.
    // An existing but blank key in the channel section turns off exclusion lists for
    // this channel.
    if ( ! the_configuration->Exists( "exclusion_lists" ) == TRUE ) {
        XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_exclusion_checkbox", wxCheckBox )
            ->SetValue( TRUE );
        wxLogDebug( "Aan exclusion_lists key not in channel, so checkbox filled" );
    }
    wxLogDebug( "Initialized exclusion_lists" );

    //------Images Tab------------------------------------------------------------------
  
    unsigned long bpp;
    int choice;

    bpp = the_configuration->Read( "bpp", 1L );
    if ( 0 < bpp )
        XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_include_images_checkbox", wxCheckBox )
            ->SetValue( TRUE );

    switch ( bpp ) { 
    case 1:
        choice = 0;
        break;

    case 2:
        choice = 1;
        break;

    case 4:
        choice = 2;
        break;

    case 8:
        choice = 3;
        break;    

    case 16:
        choice = 4;
        break;

    default:
        choice = 0;
        break;
    }
    XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_depth_choice", wxChoice )
        ->SetSelection( choice );

    wxLogDebug( "Initialized bpp" ); 

        
    XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_thumbnail_if_bigger_width_spinctrl", wxSpinCtrl )
        ->SetValue( (int) the_configuration->Read( "maxwidth", 150L ) ); 
    wxLogDebug( "Initialized maxwidth" ); 

    XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_thumbnail_if_bigger_height_spinctrl", wxSpinCtrl )
        ->SetValue( (int) the_configuration->Read( "maxheight", 250L ) );  
    wxLogDebug( "Initialized maxheight" ); 


    // Some GUI tricks here. Always link by setting a very high number (ONLY_THUMBNAIL)
    // Never link by setting to zero (ALWAYS_LINK) [Both defined above].
    // Else use the specified value.

    long alt_maxwidth;

    alt_maxwidth = the_configuration->Read( "alt_maxwidth", ALWAYS_LINK );    
    switch ( alt_maxwidth ) { 
        case ALWAYS_LINK:            
            XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_always_link_radiobutton", wxRadioButton )
                ->SetValue( TRUE );
            wxLogDebug( "Initialized alt_maxwidth-height as 1000000 (ALWAYS_LINK)" );     
            break;            

        case ONLY_THUMBNAIL:
            XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_thumbnail_radiobutton", wxRadioButton )
                ->SetValue( TRUE );
            wxLogDebug( "Initialized alt_maxwidth-height as 0 (ONLY_THUMBNAIL)" ); 
            break;

        default:
            XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_link_radiobutton", wxRadioButton )
                ->SetValue( TRUE );
            XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_link_width_spinctrl", wxSpinCtrl )
                ->SetValue( (int) the_configuration->Read( "alt_maxwidth", 800L ) );     
            XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_link_height_spinctrl", wxSpinCtrl )
                ->SetValue( (int) the_configuration->Read( "alt_maxheight", 600L ) ); 
            wxLogDebug( "Initialized alt_maxwidth/height as a specified value" ); 
            break;
    }
    wxLogDebug( "Initializion of alt_maxwidth complete" ); 
    
    //------Output Tab------------------------------------------------------------------

    int compression;

    key = "compression";
    if ( ! the_configuration->Exists( key ) )
        key = "zlib_compression";

    // DOC is choice index 0, ZLIB is choice index 1
    compression = the_configuration->Read( key, 0L ); 
    
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_compression_choice", wxChoice )
        ->SetSelection( compression ); 
    wxLogDebug( "Initialized compression" ); 
  
    long compression_limit;

    compression_limit = the_configuration->Read( "image_compression_limit", 50L); 
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_only_compress_spinctrl", wxSpinCtrl )
        ->SetValue( compression_limit ); 
    if ( 0 < compression_limit ) {
        XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_only_compress_checkbox", wxCheckBox )
             ->SetValue ( TRUE ); 
    }
    wxLogDebug( "Initialized image_compression" ); 
  
    wxString category;

    category = the_configuration->Read( "category", _T("") );
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_category_textctrl", wxTextCtrl )
         ->SetValue( category );
    if ( category != _T("") ) {
        XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_category_checkbox", wxCheckBox )
            ->SetValue ( TRUE ); 
    }
    wxLogDebug( "Initialized category" ); 

    // Note ! since asking in the positive
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_include_url_checkbox", wxCheckBox )
        ->SetValue( ! (bool) the_configuration->Read( "no_url_info", 1L ) );
        
    // Note ! since asking in the positive
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_beamable_checkbox", wxCheckBox )
         ->SetValue( ! (bool) the_configuration->Read( "copyprevention_bit", 0L ) );
    wxLogDebug( "Initialized copyprevention_bit" ); 

    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_backup_checkbox", wxCheckBox )
         ->SetValue( (bool) the_configuration->Read( "backup_bit", 0L ) );        
    wxLogDebug( "Initialized backup_bit" ); 
    
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_launchable_checkbox", wxCheckBox )
         ->SetValue( (bool) the_configuration->Read( "launchable_bit", 0L ) );
    wxLogDebug( "Initialized launchable_bit" ); 
    
    wxString big_icon;

    big_icon = the_configuration->Read( "big_icon", _T("") );
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_large_icon_textctrl", wxTextCtrl )
        ->SetValue( big_icon );
    if ( big_icon != _T("") ) {
        XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_large_icon_checkbox", wxCheckBox )
            ->SetValue ( TRUE );
    }
    wxLogDebug( "Initialized big_icon" ); 
    
    wxString small_icon;

    small_icon = the_configuration->Read( "small_icon", _T("") );
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_small_icon_textctrl", wxTextCtrl )
        ->SetValue( small_icon );
    if ( small_icon != _T("") ) {
        XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_small_icon_checkbox", wxCheckBox )
            ->SetValue ( TRUE ); 
    }                
    wxLogDebug( "Initialized small_icon" ); 
    
    //------Sync Tab--------------------------------------------------------------------

    wxString doc_file;
    wxString path;

    key = "doc_file";
    if ( ! the_configuration->Exists( key ) )
        key = "db_file";

    doc_file = the_configuration->Read( key, _T("") );    

    // If doc_file is empty, or already is the section name, select the autogenerate button
    path = the_configuration->GetPath();
    path = path.AfterFirst( '/' ); 
    if ( doc_file == _T("") || doc_file == path ) {
        XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_autogenerate_filename_radiobutton", wxRadioButton )
            ->SetValue( TRUE );   
    // Else fill in the custom filename textctrl and set the radiobutton             
    } else {
        XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_specify_filename_textctrl", wxTextCtrl )
            ->SetValue( doc_file );
        XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_specify_filename_radiobutton", wxRadioButton )
            ->SetValue( TRUE );
    }
    wxLogDebug( "Initialized doc_file" ); 

    wxString before_command;

    before_command = the_configuration->Read( "before_command", _T("") );
    XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_before_textctrl", wxTextCtrl )
        ->SetValue( before_command );
    if ( before_command != _T("") ) {
        XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_before_checkbox", wxCheckBox )
            ->SetValue ( TRUE );
    }
    wxLogDebug( "Initialized before_command" ); 

    wxString after_command;

    after_command = the_configuration->Read( "after_command", _T("") );
    XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_after_textctrl", wxTextCtrl )
        ->SetValue( after_command );
    if ( after_command != _T("") ) {
        XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_after_checkbox", wxCheckBox )
            ->SetValue ( TRUE );
    }
    wxLogDebug( "Initialized after_command" );

    //------Scheduling Tab--------------------------------------------------------------   
    
    long maxage;
    long maxage_hours;
    long number_of_age_units;
    long age_units_divider;
    maxage = the_configuration->Read( "maxage", 86400L );
    // Get how many hours are in maxage (seconds), discarding the remainder. No Pascal 'div' in C++
    // so just using a long int which will shave off the remainder after the decimal.
    maxage_hours = maxage / 3600;
    // Test if the hours divides evenly into 24 hours, and if so, select the days choice
    if ( maxage_hours % 24 == 0 ) {
        // Hours to seconds divider
        XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_update_choice", wxChoice )->SetSelection( 1 );
        age_units_divider = 24;
    } else {
        XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_update_choice", wxChoice )->SetSelection( 0 );
        age_units_divider = 1;
    }
    number_of_age_units = maxage_hours / age_units_divider;
    XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_update_spinctrl", wxSpinCtrl )->SetValue( number_of_age_units );
    
    wxString last_updated_string;    
    wxDateTime last_updated;
    last_updated_string = the_configuration->Read( "last_updated", "" );
    // In utils_datetime.h   
    last_updated = rfc_string_to_datetime( last_updated_string );
    // Set to now if an invalid date.
    if ( ! last_updated.IsValid() ) {
        last_updated = wxDateTime::Now();
    }    
    XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_calendarctrl", wxCalendarCtrl )->SetDate( last_updated ); 
    wxDateTime::wxDateTime_t last_updated_hour;
    last_updated_hour = last_updated.GetHour(); 
    XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_start_hour_choice", wxChoice )->SetSelection( (int) last_updated_hour );   
}

// Write out the values of the dialog's controls to the channel-specific section of
// the configuration file
void channel_dialog::write_configuration()
{
    //bool b;
    long lng    = 0;
    long lngx   = 0;
    long lngy   = 0;
    wxString string;

    the_configuration->Write( "doc_name", 
        XMLCTRL( *the_channel_dialog, "channel_dialog_textctrl", wxTextCtrl )->GetValue() );
                
    //------Start Tab-------------------------------------------------------------------
    
    // A website address
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_url_radiobutton", wxRadioButton )
                    ->GetValue() == TRUE ) {         
        string = XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_url_textctrl", wxTextCtrl )
                    ->GetValue();
    // A local file
    } else if (  XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_file_radiobutton", wxRadioButton )
                    ->GetValue() == TRUE ) {         
        string = XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_file_textctrl", wxTextCtrl )
                    ->GetValue();
    // List of links
    } else {
        if ( the_configuration->GetPath( ) != "/DEFAULT" ) {
            // Make the home_url a string to the channel's home.html file.
            // TODO: change to proper final location of channels directory
            string = the_configuration->GetPath( );
            string = string.AfterFirst('/');
            string = "plucker:/channels/" + string + "/home.html";
        // If DEFAULT section, then not a channel, just the template. Leave it the way
        // it was or blank if no key.
        } else {
            string == the_configuration->Read( "home_url", _T("") );
        }
    } 
    the_configuration->Write( "home_url", string );                
    
    //------Limits Tab------------------------------------------------------------------
  
    the_configuration->Write( "home_maxdepth", 
        (long) XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_maxdepth_spinctrl", wxSpinCtrl )->GetValue() );

    the_configuration->Write( "home_stayonhost",
        (bool) XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_stayonhost_checkbox", wxCheckBox )->GetValue() );

    the_configuration->Write( "home_staybelow", 
        XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_staybelow_textctrl", wxTextCtrl )->GetValue() );
  
    // TODO: FIXME: this doesn't work, it won't save an exclusion key properly
    // Possible: Writing of defaults is off, and this is a default??.
    // Exclusion list handled a bit differently. To turn on exclusion lists, delete the key,
    // and will look in main plucker.ini to get the exclusion list files.To turn off
    // exclusion lists add an empty key in the channel's plucker.ini. 
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_exclusion_checkbox", wxCheckBox )
            ->GetValue() == TRUE ) {
        // Get an assert failure if try to delete a non-existing key, so have to check that
        // it exists first.
        if ( the_configuration->Exists( "exclusion_lists" ) ) {
            // Delete it. FALSE (second parameter) is just a boolean to not delete
            // group if group is now empty after that key is removed.
            the_configuration->DeleteEntry( "exclusion_lists", FALSE );
            wxLogDebug( "Checkbox filled, so empty exclusion_lists key deleted from channel section" );
        } else {
            wxLogDebug( "Checkbox filled but no delete, since exclusion_lists key already gone" );
        }    
    } else {
        the_configuration->Write( "exclusion_lists", _T("") );
        wxLogDebug( "Checkbox empty, so empty exclusion_lists key written to channel section" );
    }
      
    //------Images Tab------------------------------------------------------------------
    
    switch ( XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_depth_choice", wxChoice )->GetSelection() ) { 
        case 0:
            lng = 1;
            break;
        case 1:
            lng = 2;
            break;
        case 2:
            lng = 4;
            break;
        case 3:
            lng = 8;
            break;
        case 4:
            lng = 16;
            break;
    }
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_include_images_checkbox", wxCheckBox )->GetValue() == FALSE ) {
        lng = 0;
    }
    the_configuration->Write( "bpp", lng);  
    
    the_configuration->Write( "maxwidth", 
        (long) XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_thumbnail_if_bigger_width_spinctrl", wxSpinCtrl )->GetValue() );

    the_configuration->Write( "maxheight", 
        (long) XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_thumbnail_if_bigger_height_spinctrl", wxSpinCtrl )->GetValue() );
    
    // Some GUI tricks here. Always link by setting a very high number, Never link by setting
    // to zero. Else use the specified value.
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_thumbnail_radiobutton", wxRadioButton )
                ->GetValue() == TRUE ) {
        lngx = 0;
        lngy = 0;
    } else if ( XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_link_radiobutton", wxRadioButton )
                ->GetValue() == TRUE ) {
        lngx = XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_link_width_spinctrl", wxSpinCtrl )
                ->GetValue();
        lngy = XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_link_height_spinctrl", wxSpinCtrl )
                ->GetValue();
    } else {
        lngx = ALWAYS_LINK;
        lngy = ONLY_THUMBNAIL;
    }    
    the_configuration->Write( "alt_maxwidth", lngx );
    the_configuration->Write( "alt_maxheight", lngy );

    //------Output Tab------------------------------------------------------------------
  
    // DOC is choice index 0, ZLIB is choice index 1, so just cast to boolean directly.
    the_configuration->Write( "zlib_compression", 
        (bool) XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_compression_choice", wxChoice )->GetSelection() );
  
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_only_compress_checkbox", wxCheckBox )
            ->GetValue() == TRUE ) { 
        the_configuration->Write( "no_image_compression_limit", 
            (long) XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_only_compress_spinctrl", wxSpinCtrl )->GetValue() );
    } else {
        the_configuration->Write( "no_image_compression_limit", 0L );
    }
    
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_category_checkbox", wxCheckBox )
            ->GetValue() == TRUE ) { 
        the_configuration->Write( "category", 
            XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_category_textctrl", wxTextCtrl )->GetValue() );
    } else {
        the_configuration->Write( "category", _T("") );
    }
    
    // Note the ! in there, as the GUI (as GUIs should) asks in the positive, not negative
    the_configuration->Write( "no_url_info",
        ! (bool) XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_include_url_checkbox", wxCheckBox )->GetValue() );

    // Note the ! in there, as the GUI (as GUIs should) asks in the positive, not negative
    the_configuration->Write( "copyprevention_bit", 
        ! (bool) XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_beamable_checkbox", wxCheckBox )->GetValue() );
    
    the_configuration->Write( "backup_bit",
        (bool) XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_backup_checkbox", wxCheckBox )->GetValue() );

    the_configuration->Write( "launchable_bit",
        (bool) XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_launchable_checkbox", wxCheckBox )->GetValue() );
  
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_large_icon_checkbox", wxCheckBox )
            ->GetValue() == TRUE ) { 
        the_configuration->Write( "big_icon", 
            XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_large_icon_textctrl", wxTextCtrl )->GetValue() );
    } else {
        the_configuration->Write( "big_icon", _T("") );
    }  
     
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_small_icon_checkbox", wxCheckBox )
            ->GetValue() == TRUE ) { 
        the_configuration->Write( "small_icon", 
            XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_small_icon_textctrl", wxTextCtrl )->GetValue() );
    } else {
        the_configuration->Write( "small_icon", _T("") );
    }
    
    //------Sync Tab--------------------------------------------------------------------
  
    // TODO: Anything wrong here? Something is awry.
    string = XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_specify_filename_textctrl", wxTextCtrl )->GetValue();                
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_specify_filename_radiobutton", wxRadioButton )
            ->GetValue() == FALSE || string ==  _T("") ) {
        // Autogenerate a unique filename, based on the section name
        string = the_configuration->GetPath();
        string = string.AfterFirst('/');       
    } 
    the_configuration->Write( "doc_file", string );
    
    string = XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_before_textctrl", wxTextCtrl )->GetValue(); 
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_before_checkbox", wxCheckBox )
            ->GetValue() == FALSE ) {
        string = _T("");
    }
    the_configuration->Write( "before_command", string );
    
    string = XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_after_textctrl", wxTextCtrl )->GetValue(); 
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_after_checkbox", wxCheckBox )
            ->GetValue() == FALSE ) {
        string = _T("");
    }     
    the_configuration->Write( "after_command", string );
    
    //------Scheduling Tab--------------------------------------------------------------

    long maxage;
    long number_of_age_units;
    long age_units_multiplier;    
    number_of_age_units = (long) XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_update_spinctrl", wxSpinCtrl )->GetValue();
    if ( XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_update_choice", wxChoice )->GetSelection() == 0 ) {
        // Hours to seconds multiplier
        age_units_multiplier = 3600;
    } else {
        // Days to seconds multiplier
        age_units_multiplier = 86400;
    }
    maxage = number_of_age_units * age_units_multiplier;
    the_configuration->Write( "maxage", maxage );
    
    wxDateTime last_updated;
    wxString last_updated_string;
    wxDateTime::wxDateTime_t last_updated_hour;         
    last_updated = XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_calendarctrl", wxCalendarCtrl )->GetDate(); 
       
    last_updated_hour = (wxDateTime::wxDateTime_t) XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_start_hour_choice", wxChoice )->GetSelection();
    last_updated.SetHour( last_updated_hour ); 
    // In utils_datetime.h
    last_updated_string = datetime_to_rfc_string( last_updated );
    the_configuration->Write( "last_updated", last_updated_string );
    
    //----------------------------------------------------------------------------------
    // Flush() writes the things to the file immediately, instead of waiting for program 
    // to exit to write them.
    the_configuration->Flush();
}

//----------------------------------------------------------------------------------------
// Event handlers
//----------------------------------------------------------------------------------------

//------Start Tab-------------------------------------------------------------------------

void channel_dialog::on_start_tab_url_radiobutton( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_url_radiobutton", wxRadioButton )
            ->GetValue();
    XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_url_textctrl", wxTextCtrl )
            ->Enable( enabled );
    XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_url_button", wxButton )
            ->Enable( enabled );
}


void channel_dialog::on_start_tab_file_radiobutton( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_file_radiobutton", wxRadioButton )
            ->GetValue();
    XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_file_textctrl", wxTextCtrl )
            ->Enable( enabled );
    XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_file_button", wxButton )
            ->Enable( enabled );
}


void channel_dialog::on_start_tab_links_radiobutton( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_links_radiobutton", wxRadioButton )
            ->GetValue();
    XMLCTRL( *the_channel_dialog, "channel_dialog_start_tab_links_button", wxButton )
            ->Enable( enabled );
}


void channel_dialog::on_start_tab_url_button( wxCommandEvent &event )
{
    init_a_showcase_dialog( this );
}


void channel_dialog::on_start_tab_file_button( wxCommandEvent &event )
{
}

void channel_dialog::on_start_tab_links_button( wxCommandEvent &event )
{
    wxString string;
    
    // The channel section is the currently active one, as an absolute path
    string = the_configuration->GetPath();
    // Strip off the leading '/' since it is an absolute.
    string = string.AfterFirst('/');
    // TODO: error handle if we somehow lost the path.
    // Show an editor dialog with this.
    init_a_editor_dialog( this, string );
}

//------Limits Tab-------------------------------------------------------------------------

void channel_dialog::on_limits_tab_exclusion_button( wxCommandEvent &event )
{
    init_a_exclusion_dialog( this, m_configuration_section );
}

void channel_dialog::on_limits_tab_staybelow_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_staybelow_checkbox", wxCheckBox )
            ->GetValue();
    XMLCTRL( *the_channel_dialog, "channel_dialog_limits_tab_staybelow_textctrl", wxTextCtrl )
            ->Enable( enabled );      
}

//------Images Tab-------------------------------------------------------------------------

void channel_dialog::on_images_tab_include_images_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_include_images_checkbox", wxCheckBox )
            ->GetValue();
    XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_depth_choice", wxChoice )
            ->Enable( enabled );      
}

void channel_dialog::on_images_tab_only_link_radiobutton( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_link_radiobutton", wxRadioButton )
            ->GetValue();
    XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_link_width_spinctrl", wxSpinCtrl )
            ->Enable( enabled );
    XMLCTRL( *the_channel_dialog, "channel_dialog_images_tab_only_link_height_spinctrl", wxSpinCtrl )
            ->Enable( enabled );
}

//------Output Tab-------------------------------------------------------------------------

void channel_dialog::on_output_tab_only_compress_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_only_compress_checkbox", wxCheckBox )
            ->GetValue();             
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_only_compress_spinctrl", wxSpinCtrl )
            ->Enable( enabled );
}

void channel_dialog::on_output_tab_category_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_category_checkbox", wxCheckBox )
                ->GetValue();             
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_category_textctrl", wxTextCtrl )
        ->Enable( enabled );
}

void channel_dialog::on_output_tab_launchable_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_launchable_checkbox", wxCheckBox )
                ->GetValue();             
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_large_icon_checkbox", wxCheckBox )
        ->Enable( enabled );
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_small_icon_checkbox", wxCheckBox )
        ->Enable( enabled );
}

void channel_dialog::on_output_tab_custom_large_icon_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
    
    // Only enable children if this checkbox is both checked and enabled    
    enabled = ( XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_large_icon_checkbox", wxCheckBox )
                ->GetValue() &&
               XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_large_icon_checkbox", wxCheckBox )
                ->IsEnabled() );
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_large_icon_textctrl", wxTextCtrl )
        ->Enable( enabled );
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_large_icon_button", wxButton )
        ->Enable( enabled );
}

void channel_dialog::on_output_tab_custom_small_icon_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
    
    // Only enable children if this checkbox is both checked and enabled  
    enabled = ( XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_small_icon_checkbox", wxCheckBox )
                ->GetValue() &&
               XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_small_icon_checkbox", wxCheckBox )
                ->IsEnabled() );      
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_small_icon_textctrl", wxTextCtrl )
        ->Enable( enabled );
    XMLCTRL( *the_channel_dialog, "channel_dialog_output_tab_custom_small_icon_button", wxButton )
        ->Enable( enabled );
}

void channel_dialog::on_output_tab_custom_small_icon_button( wxCommandEvent &event )
{
}

void channel_dialog::on_output_tab_custom_large_icon_button( wxCommandEvent &event )
{
}

//------Sync Tab---------------------------------------------------------------------------

void channel_dialog::on_sync_tab_sync_directory_button( wxCommandEvent &event )
{
}

void channel_dialog::on_sync_tab_specify_filename_radiobutton( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_specify_filename_radiobutton", wxRadioButton )
            ->GetValue();
    XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_specify_filename_textctrl", wxTextCtrl )
            ->Enable( enabled );
}

void channel_dialog::on_sync_tab_command_before_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_before_checkbox", wxCheckBox )
                ->GetValue();             
    XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_before_textctrl", wxTextCtrl )
        ->Enable( enabled );
}

void channel_dialog::on_sync_tab_command_after_checkbox( wxUpdateUIEvent &event )
{
    bool enabled;
        
    enabled = XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_after_checkbox", wxCheckBox )
                ->GetValue();             
    XMLCTRL( *the_channel_dialog, "channel_dialog_sync_tab_command_after_textctrl", wxTextCtrl )
        ->Enable( enabled );
}


//------Scheduling Tab---------------------------------------------------------------------
    
void channel_dialog::on_scheduling_tab_now_button( wxCommandEvent &event )
{
    wxDateTime current_datetime;
    wxDateTime::wxDateTime_t current_datetime_hour;
    // In utils_datetime.h  
    current_datetime = get_current_datetime();
    current_datetime_hour = current_datetime.GetHour();
    XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_calendarctrl", wxCalendarCtrl )->SetDate( current_datetime ); 
    XMLCTRL( *the_channel_dialog, "channel_dialog_scheduling_tab_start_hour_choice", wxChoice )->SetSelection( (int) current_datetime_hour );
}

// Override wxDialog's default behavior for clicking an OK button.
void channel_dialog::OnOK( wxCommandEvent& event )
{
    // Write out the values of the dialog's controls to the configuration file.
    the_channel_dialog->write_configuration();
    // Get rid of the modal dialog. Not transferring any info from this modal's control
    // to a parent dialog, so don't have to bother with wxWindow::Validate or 
    // wxWindow::TransferDataFromWindow. TODO: could return whether or not the list control
    // needs to be redrawn (since changed channel name or update time), or use a global.
    EndModal( wxID_OK );
}

