//----------------------------------------------------------------------------------------
// Name:        showcase_dialog.cpp
// Purpose:     Plucker channel showcase dialog (renders plucker_showcase.htm)
// Author:      Robert O'Connor
// Modified by:
// Created:     2001/10/26
// Copyright:   (c) Robert O'Connor ( rob@medicalmnemonics.com )
// Licence:     GPL
// RCS-ID:      $Id: showcase_dialog.cpp,v 1.8 2002/01/02 01:20:39 robertoconnor Exp $
//----------------------------------------------------------------------------------------

// ---------------------------------------------------------------------------------------
// Headers
// ---------------------------------------------------------------------------------------

#ifdef __GNUG__
    #pragma implementation "showcase_dialog.cpp"
    #pragma interface "showcase_dialog.cpp"
#endif
// ---------------------------------------------------------------------------------------
// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// For all others, include the necessary headers (this file is usually all you
// need because it includes almost all "standard" wxWindows headers)
#ifndef WX_PRECOMP
    #include "wx/wx.h"
#endif
// ---------------------------------------------------------------------------------------
#include "wx/html/m_templ.h"        // Custom HTML tag handler macros
#include "wx/html/htmlwin.h"        // wxHtmlWindow
#include "wx/listctrl.h"            // wxListCtrl
#include "wx/splitter.h"            // wxSplitterWindow
#include "wx/fs_inet.h"             // wxInternetFSHandler [wxHtmlWindow->LoadPage(http:)]
#include "wx/file.h"                // wxFile
// ---------------------------------------------------------------------------------------
#include "wx/xrc/xmlres.h"          // XRC XML resouces
// ---------------------------------------------------------------------------------------
#include "showcase_dialog.h"
#include "configuration.h"
#include "utils_internet.h"
#include "utils_controls.h"         // splitterwindow_fix_sash_out_of_range(...) 
#include "utils_string.h"           // color_to_hex_string(...)

// ---------------------------------------------------------------------------------------
// Private variables
// ---------------------------------------------------------------------------------------

showcase_dialog* the_showcase_dialog = NULL;
long showcase_listctrl_row_number                      = 0;    // Listctrl row to insert the channel info 

// ---------------------------------------------------------------------------------------
// Internal constants
// ---------------------------------------------------------------------------------------

// Easier code maintenance/translations for listctrl's columns. Keep in the same order as the
// showcase_field_names[] declared in showcase_dialog.h, though.
enum {
    NAME_COLUMN             = 0,
    URL_COLUMN,
    DESCRIPTION_COLUMN,
    LANGUAGE_COLUMN,
    CATEGORY_COLUMN,
    MAXAGE_COLUMN,
    BPP_COLUMN,
    MAXDEPTH_COLUMN,
    STAYONHOST_COLUMN,
    STAYBELOW_COLUMN
};

// Creating some controls not from XML file, so need some IDs for the windows
// Window ids, so can track their events
enum {
    SHOWCASE_LISTCTRL                    = 4101,
    PREVIEW_HTMLWINDOW,
    DETAILS_HTMLWINDOW,
    MAIN_SPLITTERWINDOW,
    DETAILS_PREVIEW_SPLITTERWINDOW
};

// Used in automatically sizing the listctrl's column widths to use full width
// without causing a horizontal scrollbar in the listctrl object window.
#define TOTAL_SPACING_WIDTH     17   // Sum of pixel spacing between all columns

#define MINIMUM_PANE_WIDTH      20   // Minimum pane width that can be moved 

// Whether the prefs said vertical or horizontal layout
#define HORIZONTAL_LAYOUT       0
#define VERTICAL_LAYOUT         1

// ---------------------------------------------------------------------------------------
// Custom handler for 'handheldsite' tag
// ---------------------------------------------------------------------------------------

// These are macros to make writing custom tag handers for wxHtmlParser much faster.
// See wx/html/m_templ.h for details of the macro's definitions.
// Note that the parameter names of a tag need to be in uppercase. We are writing a custom
// tag handler to parse the plucker_showcase.htm file to load it into the listctrl.
TAG_HANDLER_BEGIN(HANDHELDSITE, "HANDHELDSITE")
    TAG_HANDLER_PROC(tag)
    {
    wxString name;
	wxString url;
    wxString language;            // Two letter character code of site's language.
	wxString category;            // Suggests a category for viewer's library manager too.
	int maxage          = -1;     // Time (in seconds) to next update.
	int maxdepth        = -1;
	bool stayonhost     = false;
	wxString staybelow;
    wxString description;         // Sentences which describe the channel's content    
    
    wxString buf;                 // Buffer for converting integers.
    
    // Parse out parameters of tag and assign them to variables to create a listctrl row.
    if ( tag.HasParam( "NAME" ) )
        name        = tag.GetParam( "NAME" );	
    if ( tag.HasParam( "URL" ) )
        url         = tag.GetParam( "URL" );	
    if ( tag.HasParam( "DESCRIPTION" ) )
        description = tag.GetParam( "DESCRIPTION" ); 
    if ( tag.HasParam( "LANGUAGE" ) )
        language    = tag.GetParam( "LANGUAGE" ); 
    if ( tag.HasParam( "CATEGORY" ) )
        category    = tag.GetParam( "CATEGORY" ); 
    if ( tag.HasParam( "MAXAGE" ) )
        tag.ScanParam( "MAXAGE", "%i", &maxage );
    if ( tag.HasParam( "MAXDEPTH" ) )
        tag.ScanParam( "MAXDEPTH", "%i", &maxdepth );
    if ( tag.HasParam( "STAYONHOST" ) )
        stayonhost  = true;
    if ( tag.HasParam( "STAYBELOW" ) )
        staybelow   = tag.GetParam( "STAYBELOW" );    
    
    the_showcase_dialog->showcase_listctrl->InsertItem( showcase_listctrl_row_number, name, 0 );
    the_showcase_dialog->showcase_listctrl->SetItem( showcase_listctrl_row_number, URL_COLUMN, url );
    the_showcase_dialog->showcase_listctrl->SetItem( showcase_listctrl_row_number, DESCRIPTION_COLUMN, description );
    the_showcase_dialog->showcase_listctrl->SetItem( showcase_listctrl_row_number, LANGUAGE_COLUMN, language );
    the_showcase_dialog->showcase_listctrl->SetItem( showcase_listctrl_row_number, CATEGORY_COLUMN, category );
    // TODO: if -1 (none in tag), then make the string a "-" for the column    
    buf.Printf( _T( "%d" ), maxage );
    the_showcase_dialog->showcase_listctrl->SetItem( showcase_listctrl_row_number, MAXAGE_COLUMN, buf );    
    // TODO: if -1 (none in tag), then make the string a "-" for the column    
    buf.Printf( _T( "%d" ), maxdepth );
    the_showcase_dialog->showcase_listctrl->SetItem( showcase_listctrl_row_number, MAXDEPTH_COLUMN, buf ); 
    wxString stayonhost_string;
    if ( stayonhost ) {
        stayonhost_string = _T( "Yes" );
    } else {
        stayonhost_string = _T( "No" );
    }
    the_showcase_dialog->showcase_listctrl->SetItem( showcase_listctrl_row_number, STAYONHOST_COLUMN, stayonhost_string );
    the_showcase_dialog->showcase_listctrl->SetItem( showcase_listctrl_row_number, STAYBELOW_COLUMN, staybelow );


    // Increase the row number, to know where to do next insert.
    wxLogDebug( "value of showcase_listctrl_row_number is now %ld", showcase_listctrl_row_number );
    showcase_listctrl_row_number++;
    
    // This return FALSE is part of the custom tag macro.
    return FALSE;
    }
TAG_HANDLER_END(HANDHELDSITE)


TAGS_MODULE_BEGIN(handheldsite)
    TAGS_MODULE_ADD(HANDHELDSITE)
TAGS_MODULE_END(handheldsite)

// ---------------------------------------------------------------------------------------
// Event table: connect the events to the handler functions to process them
// ---------------------------------------------------------------------------------------

BEGIN_EVENT_TABLE(showcase_dialog, wxDialog)
    // Using defined ID number, since not an XML controls.
    EVT_LIST_ITEM_SELECTED( SHOWCASE_LISTCTRL, showcase_dialog::on_listctrl_item_selected )
    EVT_UPDATE_UI( XMLID( "showcase_dialog_preview_checkbox" ), showcase_dialog::on_update_ui_preview_checkbox )
    EVT_UPDATE_UI( XMLID( "showcase_dialog_details_checkbox" ), showcase_dialog::on_update_ui_details_checkbox )
    // wxWindowsBUG(2.3.2): This event isn't being passed to parent.
    EVT_SPLITTER_SASH_POS_CHANGED( XMLID( "showcase_dialog_main_splitterwindow" ), showcase_dialog::on_main_splitterwindow_sash_pos_changed )
    // Override base class wirtuals
    EVT_BUTTON( wxID_OK, showcase_dialog::OnOK )
END_EVENT_TABLE()

//----------------------------------------------------------------------------------------
// Non-event handler functions
//----------------------------------------------------------------------------------------

void init_a_showcase_dialog( wxWindow* parent )
{
    the_showcase_dialog = new showcase_dialog; 
    wxTheXmlResource->LoadDialog( the_showcase_dialog, parent, "showcase_dialog" );

    // Initialize (create) the splitters and child windows.
    the_showcase_dialog->init_splitters();
    
    // Read and set the saved values for the controls from the configuration file
    the_showcase_dialog->read_configuration();
      
    // Initialize the listctrl
    the_showcase_dialog->listctrl_init();
    the_showcase_dialog->ShowModal();
}


void showcase_dialog::init_splitters()
{
    // Construct the main_splitterwindow and the details_preview_splitterwindow
    // wxCLIP_CHILDREN commented out since it seemed to not redraw the listctrl when
    // returning from obscuring the dialog with another window/dialog. Need a refresh?
    // wxSP_3DBORDER is just there as a placeholder so won't default to SP_3D and 
    // make the 3D sash, which I think looks like ass.
    main_splitterwindow = new wxSplitterWindow( this, MAIN_SPLITTERWINDOW,
                                                wxDefaultPosition, wxDefaultSize,
                                                wxSP_3DBORDER /*wxCLIP_CHILDREN*/ );

    details_preview_splitterwindow = new wxSplitterWindow( main_splitterwindow, 
                  DETAILS_PREVIEW_SPLITTERWINDOW, wxDefaultPosition, wxDefaultSize,
                  wxCLIP_CHILDREN );
    wxLogDebug( "Constructed splitterwindows" );

    // Construct the listctrl, details htmlwindow and preview htmlwindow.
    // Note their specified parents (the 1st argument in each of their constructors)
    showcase_listctrl  = new wxListCtrl( main_splitterwindow, SHOWCASE_LISTCTRL,
                                         wxDefaultPosition, wxDefaultSize,
                                         wxLC_REPORT|wxLC_SINGLE_SEL|wxSUNKEN_BORDER );

    preview_htmlwindow = new wxHtmlWindow( details_preview_splitterwindow, PREVIEW_HTMLWINDOW,
                                           wxDefaultPosition, wxDefaultSize,
                                           wxHW_SCROLLBAR_AUTO|wxSUNKEN_BORDER );
                                                                                   
    details_htmlwindow = new wxHtmlWindow( details_preview_splitterwindow, DETAILS_HTMLWINDOW,
                                           wxDefaultPosition, wxDefaultSize,
                                           wxHW_SCROLLBAR_AUTO|wxSUNKEN_BORDER );  

    wxLogDebug( "Constructed listctrl, details htmlwindow and preview htmlwindow." );
    
    // The interesting part: want to attach the main_splitter into its placeholder inside
    // the XRC resource file. This is done with AttachUnknownControl (see xmlres.h and
    // xmlres.cpp for details). Premise is to use the <class="unknown"> to make an empty panel
    // the correct size, then use this function to reparent the actual desired control to that
    // panel at runtime.   
    wxTheXmlResource->AttachUnknownControl( _T( "showcase_dialog_main_splitterwindow" ),
                                            main_splitterwindow );
                                            
    wxLogDebug( "Attached unknown control (main_splitterwindow)" );                                

    // Set the sash sizes and borders of splitter windows for prettiness. As of 
    // wxWindows 2.3.1, these aren't in docs yet, see /include/wx/splitter.h for details.
    XMLCTRL( *the_showcase_dialog, "showcase_dialog_main_splitterwindow", wxSplitterWindow )
        ->SetBorderSize( 0 );
    details_preview_splitterwindow->SetBorderSize( 0 );
    
    //XMLCTRL( *the_showcase_dialog, "showcase_dialog_main_splitterwindow", wxSplitterWindow )
    //    ->SetSashSize( 7 );
    //details_preview_splitterwindow->SetSashSize( 7 );
    wxLogDebug( "Set the sash sizes and borders" );    

    // Set minimum pane widths
    XMLCTRL( *the_showcase_dialog, "showcase_dialog_main_splitterwindow", wxSplitterWindow )
        ->SetMinimumPaneSize( MINIMUM_PANE_WIDTH );
    details_preview_splitterwindow->SetMinimumPaneSize( MINIMUM_PANE_WIDTH );
    
    // Note: the splitterwindows now are ready to be split and windows assigned to their
    // panes. This will be done in the read_configuration().
}


// Read and set the saved values for the controls from the configuration file
void showcase_dialog::read_configuration()
{    
    //------Checkboxes, Choices---------------------------------------------------------
 
    XMLCTRL( *the_showcase_dialog, "showcase_dialog_preview_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "/PLUCKER-DESKTOP/showcase_dialog_show_preview", 1L ) );
   
    XMLCTRL( *the_showcase_dialog, "showcase_dialog_details_checkbox", wxCheckBox )
        ->SetValue( (bool) the_configuration->Read( "/PLUCKER-DESKTOP/showcase_dialog_show_details", 1L ) );

    //------SplitterWindows-------------------------------------------------------------

    // Get the values for the sash positions of the splitter windows from the preferences    . 
    long main_splitterwindow_sash_position = 
        the_configuration->Read( "/PLUCKER-DESKTOP/showcase_dialog_main_splitterwindow_sash_position", 380L );
    
    long details_preview_splitterwindow_sash_position = 
        the_configuration->Read( "/PLUCKER-DESKTOP/showcase_dialog_details_preview_splitterwindow_sash_position", 275L );
    
    // Split the windows according to the integer value read from preferences. Using
    // a switch/case to allow for more layouts later if desired.
    int layout_type = the_configuration->Read( "/PLUCKER-DESKTOP/showcase_layout_type", 1L );     
    switch ( layout_type ) {
    
        case HORIZONTAL_LAYOUT:
          
            XMLCTRL( *the_showcase_dialog, "showcase_dialog_main_splitterwindow", wxSplitterWindow )
                ->SplitVertically( showcase_listctrl, details_preview_splitterwindow,
                                   main_splitterwindow_sash_position );
                           
            details_preview_splitterwindow
                ->SplitHorizontally( preview_htmlwindow, details_htmlwindow,
                                     details_preview_splitterwindow_sash_position );
                                     
            break;    
        
        case VERTICAL_LAYOUT:
            
            XMLCTRL( *the_showcase_dialog, "showcase_dialog_main_splitterwindow", wxSplitterWindow )
                ->SplitHorizontally( showcase_listctrl, details_preview_splitterwindow,
                                     main_splitterwindow_sash_position );
                           
            details_preview_splitterwindow
                ->SplitVertically( preview_htmlwindow, details_htmlwindow,
                                   details_preview_splitterwindow_sash_position );
        
            break;
            
        default:
        
            XMLCTRL( *the_showcase_dialog, "showcase_dialog_main_splitterwindow", wxSplitterWindow )
                ->SplitVertically( showcase_listctrl, details_preview_splitterwindow,
                                   main_splitterwindow_sash_position );
                           
            details_preview_splitterwindow
                ->SplitHorizontally( preview_htmlwindow, details_htmlwindow,
                                     details_preview_splitterwindow_sash_position );
            break;
    }    
    
    // Fix them if their sashes are set out of range (this can happen especially
    // if changed Vertical->Horizontal layout in the preferences. This is from
    // utils_controls.h
    splitterwindow_fix_sash_out_of_range( XMLCTRL( *the_showcase_dialog, 
                                          "showcase_dialog_main_splitterwindow", wxSplitterWindow ) );   
    
    splitterwindow_fix_sash_out_of_range( details_preview_splitterwindow );
        
    wxLogDebug( "Split splitterwindows" );
}


// Write out the values of the dialog's controls to the configuration file.
void showcase_dialog::write_configuration()
{
    //------Checkboxes, Choices---------------------------------------------------------
 
    the_configuration->Write( "/PLUCKER-DESKTOP/showcase_dialog_show_preview",
        (bool) XMLCTRL( *the_showcase_dialog, "showcase_dialog_preview_checkbox", wxCheckBox )->GetValue() );

    the_configuration->Write( "/PLUCKER-DESKTOP/showcase_dialog_show_details",
        (bool) XMLCTRL( *the_showcase_dialog, "showcase_dialog_details_checkbox", wxCheckBox )->GetValue() );

    //------SplitterWindows-------------------------------------------------------------

    long main_splitterwindow_sash_position =
        XMLCTRL( *the_showcase_dialog, "showcase_dialog_main_splitterwindow", wxSplitterWindow )
        ->GetSashPosition();
    the_configuration->Write( "/PLUCKER-DESKTOP/showcase_dialog_main_splitterwindow_sash_position", 
        main_splitterwindow_sash_position );
    
    // Recall that this splitterwindow isn't an XRC resource, it is run-time generated.
    the_configuration->Write( "/PLUCKER-DESKTOP/showcase_dialog_details_preview_splitterwindow_sash_position", 
        (long) details_preview_splitterwindow->GetSashPosition() );
}


//Initialize the list control
void showcase_dialog::listctrl_init()
{
    // This loop is the same as writing out many lines of the form:
    // showcase_listctrl->InsertColumn( MAXDEPTH_COLUMN, showcase_field_names[ MAXDEPTH_COLUMN ], wxLIST_FORMAT_LEFT, 0 ); 
    for ( int i = 0; i < (int) WXSIZEOF( showcase_field_names ); i++ ) {   
        showcase_listctrl->InsertColumn( i, showcase_field_names[ i ], wxLIST_FORMAT_LEFT, 0 );
    }
    
    // To speed up inserting we hide the control temporarily
    showcase_listctrl->Hide();

    // Create a dummy htmlwindow to parse out the gallery tags.
    dummy_htmlwindow = new wxHtmlWindow( this, -1, wxDefaultPosition, 
                                      wxSize( 0,0 ), wxHW_SCROLLBAR_NEVER );
    
    // Load a page to the dummy_htmlwindow. As it parses, it sticks rows into the listctrl    
    // First, get the path to the resource html file  Note htmlwindow needs htm or html
    // extension to render it [without adding filters].
    wxString showcase_resource_file;
    showcase_resource_file = get_plucker_directory( RESOURCES ) << "/plucker_showcase.htm";
    // Now load the page into the dummy htmlwindow.
    dummy_htmlwindow->LoadPage( showcase_resource_file );
    
    // Reset the row number. Required.
    showcase_listctrl_row_number = 0;
    
    // Stretch the columns, now that all the name entries are available
    listctrl_stretch_columns_to_fit();
    
    // Finished inserting. Show the control
    showcase_listctrl->Show();
} 

// Stretch the name, url columns to fit the list control
void showcase_dialog::listctrl_stretch_columns_to_fit()
{
    // Set the NAME column automatically to the width of the name column widest entry
    showcase_listctrl->SetColumnWidth( NAME_COLUMN, wxLIST_AUTOSIZE );    
    int new_name_column_width = showcase_listctrl->GetColumnWidth( NAME_COLUMN );
    
    // Set the URL column to the rest
    int ctrl_width;
    int ctrl_height;    
    showcase_listctrl->GetSize( &ctrl_width, &ctrl_height ); 
    showcase_listctrl->SetColumnWidth( URL_COLUMN, 
                                       ctrl_width - new_name_column_width - TOTAL_SPACING_WIDTH );
}

// Loads up details window form a table row ( item_id )
void showcase_dialog::load_details_htmlwindow_content( long item_id )
{
    wxLogDebug( "Details htmlwindow about to load. Item id (row) is %ld", item_id ); 

    // Make a #rrggbb color to put into the HTML page, using the system color [utils_string.cpp]
    wxString hex_color_string = color_to_hex_string( wxSystemSettings::GetSystemColour( wxSYS_COLOUR_HIGHLIGHT ) );
    
    // Read the index of fields to show, and convert the strings to an array of integers [utils_string.cpp].
    wxString showcase_details_fields_string = the_configuration->Read( "/PLUCKER-DESKTOP/showcase_dialog_details_fields", "7" ); 
    wxArrayInt columns_to_display;
    string_to_arrayint( showcase_details_fields_string, ';', &columns_to_display, true );
    
    wxString details_text;      // Output text to put into details pane.
    wxListItem info;            // Stores the properties to find cell, and store its info
    
    // Get what row was selected, by looking at the event's properties
    info.m_itemId = item_id;
    for ( int i = 0; i < (int) WXSIZEOF( showcase_field_names ); i++ ) {
        // Check to see if the current column index is in our list from config file, of columns to show.
        if ( columns_to_display.Index( i ) != wxNOT_FOUND ) {
            // Set the column of the cell to look for information
            info.m_col = i;
            // Set text mask (don't know why that is here, but seems to be always used).
            info.m_mask = wxLIST_MASK_TEXT;
            // Now, get the properties of the cell we described, and store them all in info.
            showcase_listctrl->GetItem( info );
            // We successfully pulled the value from that vell
            wxLogDebug( showcase_field_names[ i ] + " column value is %s", info.m_text.c_str() );
            // Check to see that that the next text isn't just an empty string
            if ( info.m_text != "" ) {
                details_text    += _T( "<b><font color=\"" + hex_color_string + "\">" +
                                       showcase_field_names[ i ] + "</font></b>: " );
                details_text    += info.m_text;
                details_text    += "<br>";
            }
        }
        wxLogDebug( "details_text thus far=" + details_text );
    }
  
    // Set the details pane html text to the string we generated
    details_htmlwindow->SetPage( details_text );    
    
    wxLogDebug( "Finished displaying details." );
}


void showcase_dialog::load_preview_htmlwindow_content( wxString url )
{
    wxLogDebug( "Preview htmlwindow about to load. URL is %s", url.c_str() ); 
    
#if USE_BUSY_CURSOR
    // Show a busy cursor. These _automatically_ get destroyed, as soon as
    // leave the current scope (ie finish off this function). 
    // wxWindowsBug(2.3.2) Won't show one (neither does 'controls' sample).
    wxBusyCursor busy_cursor;
    wxLogDebug( "Displaying a busy_cursor." ); 
#endif    
    
    // A delay while we will be downloading. Inform user of what's going on.
    preview_htmlwindow->SetPage( _T( "Downloading channel preview..." ) );
    
    // In plucker-desktop.cpp in the initialization of handlers, one of them was
    // a Internet file system handler. This is what allows us to tell LoadPage() to
    // use a internet file system (http://) just as if it was a local file system.
    // There there is 2 general approaches:
    // (1) Use preview_htmlwindow->LoadPage(url), in which case need to 
    // add filters to let it read a file that doesn't end in .htm or .html extension,
    // and plus do something to deal with absolute host paths of images being called
    // in there. I feel that this is a shortcoming in the FileSystem, or I am not
    // using it right.
    // (2) Download the file with a stream, and save with an '.htm' file extension.
    // so load page can load it without dealing with mimefilter stuff.
    // Then replace the <img src="/logo.gif"> strings in it with 
    // <img src="http://www.thedomain.org/logo.gif">. Same for <a hrefs="">. As a bonus
    // can strip out bad html (ie pods:// protocol) while there. Then load that page 
    // into the html window.
    // (3) Save it as a .tmp file (wxWindows tells it where it should go), replace the
    // strings, and then use SetPage(). This is the approach currently done.
    
    // Generate a temporary file name, using a plkr prefix. 
    wxString local_filename;
    wxGetTempFileName( _T( "plkr" ), local_filename );
    
    // Call function to download file from remote url [see utils_internet.cpp]
    bool downloaded_successfully;
    downloaded_successfully = download_url_to_local_file( url, local_filename );
    
    // Let user know if was a problem and then abort
    if ( ! downloaded_successfully ) {    
        preview_htmlwindow->SetPage( _T( "Channel preview unavailable." ) );
        wxLogDebug( "Unable to successfully download remote url to local file." );
        return;
    }
        
    // Parse out the parts of the path from url. This seems more clear than the wxURL
    // functions for doing some of these..
    
    // Create a file object and a string to hold the text file's content    
    wxFile downloaded_file( local_filename ); // The downloaded textfile
    wxString content_string;                  // A string to hold the contnets of the file.

    // Check to see that file exists.
    if ( ! downloaded_file.Exists( local_filename ) ) {
        preview_htmlwindow->SetPage( _T( "Channel preview unavailable." ) );
        wxLogDebug( "Couldn't make a file object of file '%s'", local_filename.c_str() ); 
        return;
    } 

    // This makes a buffer, puts textfile into a buffer, then dumps buffer into a single
    // string, then immediately sets the string back to normal using UngetWriteBuf().
    char* buff = content_string.GetWriteBuf( downloaded_file.Length() );
    downloaded_file.Read( buff, downloaded_file.Length() );
    content_string.UngetWriteBuf();
    wxLogDebug( "Transferred downloaded_file into content_string" ); 

    wxYield();
    
    bool show_images = the_configuration->Read( "/PLUCKER-DESKTOP/showcase_show_images", 1L );        
    
    if ( show_images ) {
        // This is the workhorse. Replace all the href= and src= parameters with
        // their respective absolute urls [see utils_internet.cpp]. Works except 
        // mobile.theonion.com (2.3.1, cygwinb20, 256 colors) crashes program for some reason.
        // However, 2.3,2, cygwinb20.1 15-bit color OK).  TODO: Find out why.
        preview_htmlwindow->SetPage( _T( "Formatting images..." ) );
        content_string = to_absolute_urls( content_string, "src", url );    
        //wxYield();     
        preview_htmlwindow->SetPage( _T( "Formatting hyperlinks..." ) );
        content_string = to_absolute_urls( content_string, "href", url );
        
        wxYield();  
        
        // Download images, if they so desired in the preferences.
        preview_htmlwindow->SetPage( _T( "Downloading images..." ) );   
    }
       
    // Set the html window to the final string    
    preview_htmlwindow->SetPage( content_string );
    // Close the textfile object
    downloaded_file.Close(); 
    // Delete the temp file    
    wxRemoveFile( local_filename );
    
    wxLogDebug( "Finished displaying preview. Whew!" );
}


//----------------------------------------------------------------------------------------
// Event handlers
//----------------------------------------------------------------------------------------

void showcase_dialog::on_update_ui_preview_checkbox( wxUpdateUIEvent &event )
{
 

    // Save the setting to config
    the_showcase_dialog->write_configuration();
}


void showcase_dialog::on_update_ui_details_checkbox( wxUpdateUIEvent &event )
{
   
    // Save the setting to config
    the_showcase_dialog->write_configuration();
}


void showcase_dialog::on_main_splitterwindow_sash_pos_changed( wxCommandEvent& event )
{
    listctrl_stretch_columns_to_fit();
}


// When click a row in the listctrl, show that gallery site's info in the details and
// preview panel if those panes are currently visible
void showcase_dialog::on_listctrl_item_selected( wxListEvent& event )
{
    // Get the row number (m_itemIndex) that comes in with the event
    int item_id = event.m_itemIndex;

    // Wipe the window, so an unchanging content doesn't remain, if changed checkboxes
    // between the item selected events.
    details_htmlwindow->SetPage( "<html></html>" ); 
    preview_htmlwindow->SetPage( "<html></html>" );  
    // Load up the details pane, if so requested by user
    if ( the_configuration->Read( "/PLUCKER-DESKTOP/showcase_dialog_show_details", 1L ) ) {
        load_details_htmlwindow_content( item_id );
    }
    
    // Set the preview pane to the url, if so requested. If can't, then show error message.
    // TODO: error catch if possible instead of the default warning box. Could use a
    // wxDialupManager::IsOnline, but can only have 1 object at a time, so need to 
    // coordinate that with program as a whole.

    if ( the_configuration->Read( "/PLUCKER-DESKTOP/showcase_dialog_show_preview", 1L ) ) {
        wxListItem info;        // Stores the properties to both find a cell in the
                                // listctrl, or to get the cell properties out of that cell
    
        // Get what row was selected, by looking at the event's properties
        info.m_itemId = item_id;        
        info.m_col  = URL_COLUMN;
        // Set text mask (required).
        info.m_mask = wxLIST_MASK_TEXT;     
        // Now, get the properties of the cell we described, and store them all in info.
        showcase_listctrl->GetItem( info );
        wxString location = info.m_text;
        // Call up our function to load up the preview htmlwindow from 'location'
        load_preview_htmlwindow_content( location );
    }
}


// Override wxDialog's default behavior for clicking an OK button.
void showcase_dialog::OnOK( wxCommandEvent& event )
{
    // Write out the values of the dialog's controls to the configuration file.
    the_showcase_dialog->write_configuration();
    // Get rid of the modal dialog. Not transferring any info from this modal's control
    // to a parent dialog, so don't have to bother with wxWindow::Validate or 
    // wxWindow::TransferDataFromWindow.    
    EndModal( wxID_OK );
}

