/*
 * $Id: bookmark.c,v 1.23 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "bookmark.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "history.h"
#include "resourceids.h"
#include "util.h"


UInt16 CreatePopupBookmarkList( ListType* list ) PLKRDB_SECTION;
UInt16 CreateBookmarkList( ListType* list ) PLKRDB_SECTION;
Int16 RestoreBookmarkData( UInt16 index ) PLKRDB_SECTION;
void ReleaseBookmarkList( void ) PLKRDB_SECTION;
void AddBookmark( Char* name ) PLKRDB_SECTION;
Boolean DeleteBookmark( UInt16 index ) PLKRDB_SECTION;

static void DrawListItem( Int16 itemNum, RectangleType* bounds,
                    Char** itemsText ) PLKRDB_SECTION;
static UInt16 InitBookmarkList( ListType* list ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define ADD_BOOKMARK_LEN        25
#define EDIT_BOOKMARK_LEN       25

#define ADD_BOOKMARK            0
#define EDIT_BOOKMARK           1

#define NO_PARAGRAPHS           0

#define MAX_BOOKMARK_LIST_LEN   12
#define MAX_BOOKMARK_LEN        20

#define BOOKMARK_HEADER_LEN     ( 3 * sizeof( UInt16 ) )

#define GetEntries( ptr )   ( ( ptr )[ 2 ] * 256 + ( ptr )[ 3 ] )
#define GetOffset( ptr )    ( ( ptr )[ 4 ] * 256 + ( ptr )[ 5 ] )


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
typedef struct {
    Int16 verticalOffset;
    Int16 firstVisibleParagraph;
    Int16 firstParagraphY;
    Int16 lastVisibleParagraph;
    Int16 lastParagraphY;
    Int16 recordId;
} BookmarkData;


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static MemHandle    nameListHandle  = NULL;
static Boolean      popupList       = false;

static Char         addBookmark[ ADD_BOOKMARK_LEN ];
static Char         editBookmark[ EDIT_BOOKMARK_LEN ];



/* Callback function that draws list items */
static void DrawListItem
    (
    Int16           itemNum,    /* item number */
    RectangleType*  bounds,     /* pointer to a RectangleType structure that
                                   specifies the bounds for the list item */
    Char**          itemsText   /* pointer to an array of text strings */
    )
{
    FontID oldFont;

    SET_A4_FROM_A5 

    if ( popupList && ( itemNum == ADD_BOOKMARK || itemNum == EDIT_BOOKMARK ) )
        oldFont = FntSetFont( boldFont );
    else
        oldFont = FntSetFont( stdFont );

    if ( itemsText[ itemNum ] != NULL )
        WinDrawChars( itemsText[ itemNum ], StrLen( itemsText[ itemNum ] ), 
            bounds->topLeft.x, bounds->topLeft.y );

    FntSetFont( oldFont );

    RESTORE_A4
}



/* Populate list with bookmarks, return number of bookmarks in list */
static UInt16 InitBookmarkList
    (
    ListType* list  /* pointer to list */
    )
{
    UInt16    entries;
    UInt16    i;
    MemHandle handle;
    Char**    nameList;
    Char*     bookmarkPtr;
    UInt16    extraListItems;

    handle        = NULL;
    nameList      = NULL;
    bookmarkPtr   = NULL;

    if ( ReturnMetaHandle( INTERNAL_BOOKMARKS_ID, NO_PARAGRAPHS, false, &handle ) == errNone ) {
        if ( popupList )
            extraListItems = 2;
        else
            extraListItems = 0;

        bookmarkPtr = (Char*) MemHandleLock( handle );
        ErrFatalDisplayIf( bookmarkPtr == NULL, "InitBookmarkList: MemHandleLock failed" );

        entries         = GetEntries( bookmarkPtr ) + extraListItems;
        bookmarkPtr    += BOOKMARK_HEADER_LEN;
    }
    else {
        if ( popupList ) {
            extraListItems  = 2;
            entries         = 2;
        }
        else {
            entries = 0;

            LstSetListChoices( list, nameList, entries );
            LstSetDrawFunction( list, DrawListItem );

            return entries;
        }
    }

    /* Allocate arrays for name list */
    nameListHandle = MemHandleNew( entries * sizeof( *nameList ) );
    if ( nameListHandle == NULL ) {
        if ( extraListItems < entries )
            MemHandleUnlock( handle );

        return 0;
    }
    nameList = (Char**) MemHandleLock( nameListHandle );
    ErrFatalDisplayIf( nameList == NULL, "InitBookmarkList: MemHandleLock failed" );

    if ( popupList ) {
        SysCopyStringResource( addBookmark, strMainAddBookmark );
        nameList[ ADD_BOOKMARK ] = addBookmark;
        SysCopyStringResource( editBookmark, strMainViewBookmark );
        nameList[ EDIT_BOOKMARK ] = editBookmark;
    }
    for ( i = extraListItems; i < entries; i++ ) {
        nameList[ i ]   = bookmarkPtr;
        bookmarkPtr    += StrLen( bookmarkPtr ) + 1;
    }
    LstSetListChoices( list, nameList, entries );
    LstSetDrawFunction( list, DrawListItem );

    if ( MAX_BOOKMARK_LIST_LEN < entries )
        LstSetHeight( list, MAX_BOOKMARK_LIST_LEN );
    else
        LstSetHeight( list, entries );

    if ( extraListItems < entries )
        MemHandleUnlock( handle );

    return entries;
}



/* Populate list with bookmarks ( including Add/View Bookmarks ), return number 
   of bookmarks in list */
UInt16 CreatePopupBookmarkList
    (
    ListType* list  /* pointer to list */
    )
{
    popupList = true;
    return InitBookmarkList( list );
}



/* Populate list with bookmarks ( skip Add/View Bookmarks ), return number 
   of bookmarks in list */
UInt16 CreateBookmarkList( ListType* list )
{
    popupList = false;
    return InitBookmarkList( list );
}



/* Unlock list handle and free allocated memory */
void ReleaseBookmarkList( void )
{
    if ( nameListHandle != NULL ) {
        MemHandleUnlock( nameListHandle );
        MemHandleFree( nameListHandle );
        nameListHandle = NULL;
    }
}



/* Restore data for current bookmark, return the record ID or if 
   there are no bookmarks NOT_FOUND */
Int16 RestoreBookmarkData
    (
    UInt16 index     /* index in bookmarks list */
    )
{
    MetaRecord*     meta;
    BookmarkData    bookmarkData;
    Char*           bookmarkPtr;
    MemHandle       handle;
    UInt16          offset;

    handle = NULL;

    if ( ReturnMetaHandle( INTERNAL_BOOKMARKS_ID, NO_PARAGRAPHS, false, &handle ) != errNone )
        return NOT_FOUND;

    /* Skip "Add/View Bookmarks" */
    index -= 2;

    bookmarkPtr = (Char*) MemHandleLock( handle );
    ErrFatalDisplayIf( bookmarkPtr == NULL, "RestoreBookmarkData: MemHandleLock failed" );

    offset = GetOffset( bookmarkPtr );
    bookmarkPtr = (void *) bookmarkPtr + offset + index * sizeof(BookmarkData);
    MemMove( &bookmarkData, bookmarkPtr, sizeof( BookmarkData ) );

    meta = (MetaRecord*) MemHandleLock( GetMetaHandle( bookmarkData.recordId, false ) );
    DmWrite( meta, OFFSETOF( MetaRecord, verticalOffset ), (void*) &bookmarkData, 
        sizeof( BookmarkData ) - sizeof( Int16 ) );

    /* Add to history */
    AddToHistory( bookmarkData.recordId );
    SetHistoryFirstParagraph( bookmarkData.firstVisibleParagraph, bookmarkData.firstParagraphY );
    SetHistoryLastParagraph( bookmarkData.lastVisibleParagraph, bookmarkData.lastParagraphY );
    SetHistoryVerticalOffset( bookmarkData.verticalOffset );

    MemPtrUnlock( meta );
    MemHandleUnlock( handle );

    return bookmarkData.recordId;
}



/* Add data for bookmark */
void AddBookmark
    (
    Char* name      /* name of bookmark */
    )
{
    MetaRecord*     meta;
    BookmarkData    bookmarkData;
    Char*           bookmarkPtr;
    MemHandle       handle;
    UInt32          endOfRecord;
    UInt16          entries;
    UInt16          offset;
    UInt16          newSize;
    UInt16          nameLength;
    UInt16          i;

    handle = NULL;

    if ( ReturnMetaHandle( INTERNAL_BOOKMARKS_ID, NO_PARAGRAPHS, false, &handle ) != errNone )
        if ( AddBookmarkRecord( &handle ) != errNone )
            return;

    endOfRecord = MemHandleSize( handle );
    nameLength  = StrLen( name ) + 1;
    newSize     = endOfRecord + sizeof( BookmarkData ) + nameLength;

    ResizeRecord( INTERNAL_BOOKMARKS_ID, newSize, &handle );
    if ( handle == NULL )
        return;

    bookmarkPtr = (Char*) MemHandleLock( handle );
    ErrFatalDisplayIf( bookmarkPtr == NULL, "AddBookmark: MemHandleLock failed" );

    entries = GetEntries( bookmarkPtr ) + 1;
    offset  = GetOffset( bookmarkPtr ) + nameLength;
    DmWrite( (void*) bookmarkPtr, sizeof( UInt16 ), (void*) &entries,
        sizeof( UInt16 ) );
    DmWrite( (void*) bookmarkPtr, 2 * sizeof( UInt16 ), (void*) &offset,
        sizeof( UInt16 ) );

    meta = (MetaRecord*) MemHandleLock( GetMetaRecord() );

    bookmarkData.verticalOffset         = meta->verticalOffset;
    bookmarkData.firstVisibleParagraph  = meta->firstVisibleParagraph;
    bookmarkData.firstParagraphY        = meta->firstParagraphY;
    bookmarkData.lastVisibleParagraph   = meta->lastVisibleParagraph;
    bookmarkData.lastParagraphY         = meta->lastParagraphY;
    bookmarkData.recordId               = GetHistoryCurrent();

    /* Write new block of bookmark data */
    DmWrite( bookmarkPtr, endOfRecord + nameLength, &bookmarkData,
        sizeof( BookmarkData ) );
    endOfRecord -= sizeof( BookmarkData );

    /* Reshuffle old blocks with bookmark data */
    for ( i = 1; i < entries; i++ ) {
        MemMove( &bookmarkData, bookmarkPtr + endOfRecord, sizeof( BookmarkData ) );
        DmWrite( bookmarkPtr, endOfRecord + nameLength, &bookmarkData,
            sizeof( BookmarkData ) );
        endOfRecord -= sizeof( BookmarkData );
    }

    /* Write new bookmark name */
    DmStrCopy( bookmarkPtr, offset - nameLength, name );

    MemHandleUnlock( GetMetaRecord() );
    MemHandleUnlock( handle );
}



/* Delete bookmark, return true if list is empty */
Boolean DeleteBookmark
    (
    UInt16 index     /* index in bookmarks list */
    )
{
    BookmarkData  bookmarkData;
    MemHandle     handle;
    UInt16        entries;
    UInt16        offset;
    UInt16        tempOffset;
    UInt16        newSize;
    UInt16        nameLength;
    UInt16        i;
    Char*         bookmarkPtr;
    Char*         readPtr;

    handle = NULL;

    if ( ReturnMetaHandle( INTERNAL_BOOKMARKS_ID, NO_PARAGRAPHS, false, &handle ) != errNone )
        return true;

    bookmarkPtr = (Char*) MemHandleLock( handle );
    ErrFatalDisplayIf( bookmarkPtr == NULL, "DeleteBookmark: MemHandleLock failed" );

    entries = GetEntries( bookmarkPtr );

    if ( entries <= 1 ) {
        MemHandleUnlock( handle );
        ReleaseBookmarkList();
        RemoveBookmarkRecord();
        return true;
    }

    /* Find name string for bookmark  */
    readPtr = bookmarkPtr + BOOKMARK_HEADER_LEN;
    for ( i = 0; i < index; i++ )
        readPtr += StrLen( readPtr ) + 1;

    tempOffset  = readPtr - bookmarkPtr;
    nameLength  = StrLen( readPtr ) + 1;
    readPtr    += nameLength;
    for ( i = 0; i < entries - index - 1; i++ ) {
        UInt16  length;
        Char    tempString[ MAX_BOOKMARK_LEN + 1 ];

        MemMove( tempString, readPtr, StrLen( readPtr ) + 1 );
        length = StrLen( tempString ) + 1;

        DmWrite( bookmarkPtr, tempOffset, tempString, length );

        tempOffset  += length;
        readPtr     += length;
    }

    /* Reshuffle blocks with bookmark data */
    offset  = GetOffset( bookmarkPtr );
    readPtr = bookmarkPtr + offset;
    for ( i = 0; i < index; i++ ) {
        MemMove( &bookmarkData, readPtr, sizeof( BookmarkData ) );

        DmWrite( bookmarkPtr, tempOffset, &bookmarkData, sizeof( BookmarkData ) );
        tempOffset  += sizeof( BookmarkData );
        readPtr     += sizeof( BookmarkData );
    }
    readPtr += sizeof( BookmarkData );
    for ( i = index + 1; i < entries; i++ ) {
        MemMove( &bookmarkData, readPtr, sizeof( BookmarkData ) );

        DmWrite( bookmarkPtr, tempOffset, &bookmarkData, sizeof( BookmarkData ) );
        tempOffset  += sizeof( BookmarkData );
        readPtr     += sizeof( BookmarkData );
    }

    entries--;
    DmWrite( bookmarkPtr, sizeof( UInt16 ), (void*) &entries,
        sizeof( UInt16 ) );
    offset -= nameLength;
    DmWrite( bookmarkPtr, 2 * sizeof( UInt16 ), (void*) &offset,
        sizeof( UInt16 ) );

    newSize = MemHandleSize( handle ) - sizeof( BookmarkData ) - nameLength;

    MemHandleUnlock( handle );

    ResizeRecord( INTERNAL_BOOKMARKS_ID, newSize, &handle );

    return false;
}
