/*
 * browser.c:
 *
 * Copyright (c) 2002 James McKenzie <james@fishsoup.dhs.org>,
 * All rights reserved.
 *
 */

static char rcsid[] =
  "$Id: browser.c,v 1.1 2002/01/08 19:35:11 root Exp root $";

/*
 * $Log: browser.c,v $
 * Revision 1.1  2002/01/08 19:35:11  root
 * Initial revision
 *
 */

#include "const.h"
#include "resourceids.h"
#include "uncompress.h"
#include "util.h"
#include "db.h"
#include "browser.h"
#include "documentdata.h"
#include "categorydata.h"
#include "dbmgrform.h"
#include "session.h"
#include "fetchdb.h"
#include "bhistory.h"


static Boolean
URLToRecordIDWorker (Int16 recnum, Int16 * recordid, Char * url, Int16 * ret)
  PLKRDB_SECTION;
     static Boolean URLToRecordID (Char * url, Int16 * ret) PLKRDB_SECTION;
     static MemHandle
       ResolveLinkByURLWorker (Char * url, Int16 * _recordid) PLKRDB_SECTION;

void
RecordIDToURL (Int16 recordId, Char ** url)
{
  MemHandle linkHandle = NULL;
  Char *text = NULL, *ptr;

  if (ReturnRecordHandle (PLUCKER_LINKS_ID, &linkHandle) == errNone)
    {
      MemHandle uncompressHandle;
      Header *linkDocument;
      Int16 count;
      Int16 offset;
      Int16 *numP;

      count = recordId;
      offset = 0;

      linkDocument = (Header *) MemHandleLock (linkHandle);
      numP = (Int16 *) (linkDocument + 1);

      while (*numP < recordId)
        {
          offset = *numP;
          numP += 2;
        }
      numP += 1;
      count -= offset;

      MemHandleUnlock (linkHandle);

      if (ReturnRecordHandle (*numP, &linkHandle) != errNone)
        {
          *url = NULL;
          return;
        }

      uncompressHandle = NULL;
      linkDocument = (Header *) MemHandleLock (linkHandle);
      if (linkDocument->type == DATATYPE_LINKS_COMPRESSED)
        {
          uncompressHandle = Uncompress (linkDocument);
          MemHandleUnlock (linkHandle);
          linkHandle = NULL;
          if (uncompressHandle)
            text = (Char *) MemHandleLock (uncompressHandle);
        }
      else if (linkDocument->type == DATATYPE_LINKS)
        {
          text = (Char *) (linkDocument + 1);
        }

      if (text != NULL)
        {
          while (--count)
            text += StrLen (text) + 1;
          *url = MemPtrNew (StrLen (text) + 1);
          if (*url)
            StrCopy (*url, text);
        }
      if (linkHandle != NULL)
        MemHandleUnlock (linkHandle);
      if (uncompressHandle != NULL)
        MemHandleUnlock (uncompressHandle);
    }
  if (!text)
    {
      *url = NULL;
      return;
    }

  if ((ptr = *url))
    {
      while (*ptr)
        {
          if (*ptr == '#')
            *ptr = 0;
          ptr++;
        }
    }



}

static Boolean
URLToRecordIDWorker (Int16 recnum, Int16 * recordid, Char * url, Int16 * ret)
{
  MemHandle uncompressHandle = NULL;
  MemHandle linkHandle = NULL;
  Char *text = NULL;
  UInt16 len, q;
  Boolean found = false;
  Header *linkDocument;


  if (ReturnRecordHandle (recnum, &linkHandle) != errNone)
    {
      return false;
    }

  linkDocument = (Header *) MemHandleLock (linkHandle);
  len = linkDocument->size;

  if (linkDocument->type == DATATYPE_LINKS_COMPRESSED)
    {
      uncompressHandle = Uncompress (linkDocument);
      MemHandleUnlock (linkHandle);
      linkHandle = NULL;
      if (uncompressHandle)
        text = (Char *) MemHandleLock (uncompressHandle);
    }
  else if (linkDocument->type == DATATYPE_LINKS)
    {
      text = (Char *) (linkDocument + 1);
    }


  while (len)
    {

      if (!StrCompare (text, url))
        {
          *ret = *recordid;
          found = true;
          break;
        }
      q = StrLen (text) + 1;
      text += q;
      len -= q;
      (*recordid)++;
    }
  if (linkHandle != NULL)
    MemHandleUnlock (linkHandle);
  if (uncompressHandle != NULL)
    MemHandleUnlock (uncompressHandle);
  return found;
}



static Boolean
URLToRecordID (Char * url, Int16 * ret)
{
  MemHandle linkHandle = NULL;
  Boolean found = false;


  if (ReturnRecordHandle (PLUCKER_LINKS_ID, &linkHandle) == errNone)
    {
      Header *linkDocument;
      UInt16 linkrecords;
      Int16 *numP, recordid = 1;

      linkDocument = (Header *) MemHandleLock (linkHandle);
      linkrecords = linkDocument->size / 4;
      numP = (Int16 *) (linkDocument + 1);
      while (linkrecords--)
        {
          /* *(numP) points to the recordid of the _last!_ url in ... */
          /* *(numP+1) points to the record number of the first set of urls */
          /* Work through record *(numP+1) */
          found = URLToRecordIDWorker (*(numP + 1), &recordid, url, ret);
          if (found)
            break;
          numP += 2;
        }
      MemHandleUnlock (linkHandle);
    }
  return found;
}




static MemHandle
ResolveLinkByURLWorker (Char * url, Int16 * _recordid)
{
  Int16 recordId, i;
  MemHandle ret;
  Header *record;
  LocalID dbID;
  Int16 cardNo;
  Err err;
  char name[dmDBNameLength + 1];

  SetWorking ();
  set_progress ("lnk");

/*Ok now we chug through every document seeing if any of */
/*the links match this one, and if they do match then check */
/*to see if the links have any data associated with them*/
/*we have to do most of this ourselfs as we can't use */
/*the onboard functions since we don't want to close ViewerDocument*/

  dbID = Prefs ()->dbID;
  cardNo = Prefs ()->cardNo;
  StrCopy (name, Prefs ()->docName);

  SaveSessionData ();
  CloseDocument ();
  OpenDocList ();

  for (i = 0; i < GetNumOfDocuments (); ++i)
    {

      if (!OpenNewDocument (i))
        {
          /* Check the overall document base - in case we fetched it */
          if (ReturnLastHandle (&ret) == errNone)
            {
              record = (Header *) MemHandleLock (ret);
              if ((record->type == DATATYPE_BASE ) && (!StrCompare ((Char *) (record + 1), url)))
                {
                  MemHandleUnlock (ret);
                  CloseDocList ();
                  if (_recordid)
                    *_recordid = HOME_PAGE_ID;

                  ret = GetRecordHandle (2);

                  SetIdle ();
                  return ret;
                }
              MemHandleUnlock (ret);
            }



          if (URLToRecordID (url, &recordId))
            {
/*This may also just be a URL with no data so see if we can find the record...*/
              if (ReturnRecordHandle (recordId, &ret) == errNone)
                {

                  record = (Header *) MemHandleLock (ret);
                  if (record->type != DATATYPE_LINK_INDEX)
                    {

                      MemHandleUnlock (ret);
                      CloseDocList ();
                      if (_recordid)
                        *_recordid = recordId;

                      SetIdle ();
                      return ret;
                    }
                  MemHandleUnlock (ret);


                }
            }
          CloseDocument ();
        }
    }

  err = OpenDocument (dbID, cardNo);
  if (err == errNone)
    {

      Prefs ()->dbID = dbID;
      Prefs ()->cardNo = cardNo;
      StrCopy (Prefs ()->docName, name);

      InitSessionData ();

    }

  CloseDocList ();
  SetIdle ();

  return NULL;
}

MemHandle
ResolveLinkByURL (char *url, Int16 * recordId)
{
  int tries = 0;
  MemHandle ret;

  for (tries = 0; tries < 4; ++tries)
    {
      if (tries > 1)
        if (!FrmAlert (confirmFetchFailed))
          return NULL;
      if (tries)
        {
          /*Link not on device */
          Char *newurl;

	  /*Fetch - will prompt user if network is down*/
          SetWorking ();
          {
            Char *base = FetchDB (url);
            if (base)
              MemPtrFree (base); /*No longer need to follow these */
/*done in base records*/
          }
          SetIdle ();
        }

      ret = ResolveLinkByURLWorker (url, recordId);
      if (ret)
        {
          set_progress ("zip");
          return ret;
        }

    }
  return NULL;
}


MemHandle
ResolveLinkByRecordID (Int16 * recordId)
{
  Char *url = NULL;
  MemHandle ret;

  /*Get the url in text form */

  RecordIDToURL (*recordId, &url);
  if (!url)
    return NULL;

  ret = ResolveLinkByURL (url, recordId);

  MemPtrFree (url);
  return ret;
}
