/*
 * $Id: control.c,v 1.41 2001/11/04 18:41:27 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "anchor.h"
#include "bookmark.h"
#include "const.h"
#include "control.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "history.h"
#include "image.h"
#include "link.h"
#include "paragraph.h"
#include "resourceids.h"
#include "search.h"
#include "uncompress.h"
#include "browser.h"
#include "bhistory.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define NUMBER_OF_TOOLBAR_OBJECTS   13

#define ADD_BOOKMARK    0
#define VIEW_BOOKMARK   1

#define TOP     0
#define BOTTOM  10

/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
typedef enum {
    REGION1,
    REGION2,
    REGION3,
    REGION4
} RegionType;


static struct {
    Int16 penDownY;
    Int16 penDownX;
    Int16 penMoveX;
    Int16 penMoveY;
    Int16 moveFromX;
    Int16 moveFromY;
    Int16 selectedControl;
} PenData;


typedef struct {
    RectangleType   bounds;
    AnchorStateType state;
} ControlDataType;


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static ControlDataType  controls[ NUMBER_OF_TOOLBAR_OBJECTS ];
static Int16            newX;
static Int16            newY;
static Int16            imageWidth;
static Int16            imageHeight;



/* Jump to requested record */
void JumpToRecord
    (
    const Int16 record, /* record ID */
    const Int16 offset  /* offset to first paragraph */
    )

{
    Boolean successful;

    successful = ViewRecord( record, true, offset );

    if ( successful )
        SetVisitedLink( record );
    else {
        ViewRecord( GetHistoryPrev(), false, NO_OFFSET );
        FrmAlert( warnInsufficientMemory );
    }
}



/* Initialize the data necessary for scrolling images */
void InitializeTbmpData
    (
    Int16 width,    /* width of image */
    Int16 height    /* height of image */
    )
{
    newX        = 0;
    newY        = 0;
    imageWidth  = width;
    imageHeight = height;
}



/* Clear the structure holding pen data between events */
static void ClearPenData( void )
{
    MemSet( (void*) &PenData, sizeof( PenData ), 0 );
}



/* Clear the structure holding control boundaries */
void ClearControlBounds( void )
{
    MemSet( (void*) &controls, sizeof( controls ), 0 );
}



/* Set highlight status for a graphical control object */
static void HighlightGraphicalControl
    (
    const Int16             controlID,  /* control ID of the object */
    const AnchorStateType   state       /* anchor state ( ANCHOR_SELECTED or 
                                           ANCHOR_UNSELECTED ) */
    )
{
    if ( controls[ controlID ].state != state )
        WinInvertRectangle( &controls[ controlID ].bounds, 0 );
}



/* Determine whether an anchor or control object should change highlight status */
static void HighlightControl
    (
    const Int16             controlID,  /* control ID of the object */
    const AnchorStateType   state       /* anchor state ( ANCHOR_SELECTED or 
                                           ANCHOR_UNSELECTED ) */
    )
{
    if ( 0 < controlID )
        HighlightAnchor( controlID - 1, state );
    else if ( controlID < 0 )
        HighlightGraphicalControl( -( controlID + 1 ), state );
}



/* Find control object at given location, return control ID, or NOT_FOUND 
   if no object was found at the given location */
static Int16 FindGraphicalControl
    (
    const Int16 x,
    const Int16 y
    )
{
    Int16 controlID;

    for ( controlID = 0; controlID < NUMBER_OF_TOOLBAR_OBJECTS; controlID++ )
        if ( RctPtInRectangle( x, y, &controls[ controlID ].bounds ) )
            return controlID;

    return NOT_FOUND;
}



/* Find control object at pen location */
static Int16 FindControlObject
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Int16 controlObject;

    controlObject = FindGraphicalControl( event->screenX, event->screenY );

    /* event occurred on a graphical control */
    if ( controlObject != NOT_FOUND )
        return -( controlObject + 1 );

    /* event did not occur on anything in the toolbar */
    if ( event->screenY <= TopLeftY() || TopLeftY() + ExtentY() <= event->screenY )
        return 0;

    controlObject = FindVisibleAnchor( event->screenX, event->screenY );

    /* event occurred on an anchor */
    if ( controlObject != NOT_FOUND )
        return controlObject + 1;

    /* event did not occur on anything */
    return 0;
}



/* Perform action assigned to given control object */
void DoControlAction
    (
    const Int16 control /* control value of the object */
    )
{
    Int16 newRecord;
    Int16 controlID;

    /* Clear location of find pattern */
    ClearFindPatternData();

    if ( 0 < control ) {
        Int16 anchorIndex;
        Int16 reference;
        Int16 offset;
        Int16 image;

        anchorIndex = control - 1;
        reference   = GetVisibleReference( anchorIndex );
        offset      = GetVisibleOffset( anchorIndex );
        image       = GetVisibleImage( anchorIndex );

        if ( Prefs()->linkClick )
            SndPlaySystemSound( sndClick );

        if ( image != 0 ) {
            RectangleType   bounds;
            ListType*       list;
            Int16           x;
            Int16           y;
            Int16           selection;

            bounds  = GetVisibleImagePosition( anchorIndex );
            list    = (ListType*) GetObjectPtr( frmMainImageDialog );

            x = ( bounds.extent.x - 60 ) / 2;
            if ( x < 0 )
                x = 0;
            y = ( bounds.extent.y - 22 ) / 2;
            if ( y < 0 )
                y = 0;
            x += bounds.topLeft.x;
            y += bounds.topLeft.y;
            LstSetPosition( list, x, y );

            selection = LstPopupList( list );
            if ( selection == noListSelection )
                return;
            else if ( selection == 1 )
                reference = image;
        }
        JumpToRecord( reference, offset );
        return;
    }

    if ( control != 0 )
        SndPlaySystemSound( sndClick );

    controlID = -( control + 1 );
    switch ( controlID ) {
        case HOMECONTROL:
            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            ViewRecord( HOME_PAGE_ID, true, NO_OFFSET );
            SetVisitedLink( HOME_PAGE_ID );
            break;

        case LEFTCONTROL:
            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            newRecord = GetHistoryPrev();
            if ( newRecord != NOT_FOUND ) {
                ViewRecord( newRecord, false, NO_OFFSET );
                SetVisitedLink( newRecord );
            } else { 
              /*Do inter document history */
	      BrowserDirection(-1);
	    }
            break;

        case RIGHTCONTROL:
            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            newRecord = GetHistoryNext();
            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            if ( newRecord != NOT_FOUND ) {
                ViewRecord( newRecord, false, NO_OFFSET );
                SetVisitedLink( newRecord );
            } else { 
              /*Do inter document history */
	      BrowserDirection(1);
            }
            break;

        case LIBRARYCONTROL:
            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            FrmGotoForm( frmLibrary );
            break;

        case FINDCONTROL:
            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            FrmPopupForm( frmSearch );
            break;

        case AGAINCONTROL:
            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            SearchAgain();
            break;

        case MENUCONTROL:
            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            EvtEnqueueKey( menuChr, 0, commandKeyMask );
            break;

        case OFFSETCONTROL:
        {
            Int16 selection;

            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            selection = LstPopupList( (ListType*) GetObjectPtr( frmMainPercentList ) );
            if ( selection != noListSelection )
                GotoLocation( selection );

            break;
        }

        case BOOKMARKCONTROL:
        {
            ListType*   list;
            Int16       selection;

            DoAutoscrollToggle ( AUTOSCROLL_OFF );
            list = (ListType*) GetObjectPtr( frmMainBookmarkList );
            if ( CreatePopupBookmarkList( list ) == 0 )
                break;

            selection = LstPopupList( (ListType*) GetObjectPtr( frmMainBookmarkList ) );

            ReleaseBookmarkList();

            if ( selection == ADD_BOOKMARK ) {
                FrmPopupForm( frmAddBookmark );
            }
            else if ( selection == VIEW_BOOKMARK ) {
                FrmPopupForm( frmBookmarks );
            }
            else if ( selection != noListSelection ) {
                newRecord = RestoreBookmarkData( selection );
                if ( newRecord != NOT_FOUND ) {
                    ViewRecord( newRecord, false, NO_OFFSET );
                    SetVisitedLink( newRecord );
                }
            }
            break;
        }        
                    
        case AUTOSCROLLSTARTCONTROL: {
        case AUTOSCROLLSTOPCONTROL: 
		    DoAutoscrollToggle( AUTOSCROLL_TOGGLE );
            break;
        }
                    
        case AUTOSCROLLINCRCONTROL: {
			DoAutoscrollIncr();
            break;    
        }
        
        case AUTOSCROLLDECRCONTROL: {
			DoAutoscrollDecr();
        	break;
        }
    
        default:
            break;
    }
}



/* Perform action for specified select type */
void DoSelectTypeAction
    (
    SelectType selection    /* selection type */
    )
{

    if ( MainFormWinActive() ) {

       if ( selection != SELECT_TOGGLE_AUTOSCROLL && selection != SELECT_INCREASE_AUTOSCROLL
            && selection != SELECT_DECREASE_AUTOSCROLL ) {
            DoAutoscrollToggle( AUTOSCROLL_OFF );
       }

        switch ( selection ) {
            case SELECT_NONE:
                break;

            case SELECT_FULL_PAGE_UP:
                DoPageMove( GetScrollValue() );
                break;

            case SELECT_HALF_PAGE_UP:
                DoPageMove( GetScrollValue() / 2 );
                break;

            case SELECT_FULL_PAGE_DOWN:
                DoPageMove( -GetScrollValue() );
                break;

            case SELECT_HALF_PAGE_DOWN:
                DoPageMove( -GetScrollValue() / 2 );
                break;

            case SELECT_GO_BACK:
                DoControlAction( -( LEFTCONTROL + 1 ) );
                break;

            case SELECT_GO_FORWARD:
                DoControlAction( -( RIGHTCONTROL + 1 ) );
                break;

            case SELECT_GO_HOME:
                DoControlAction( -( HOMECONTROL + 1 ) );
                break;

            case SELECT_GO_TO_TOP:
                GotoLocation( TOP );
                break;

            case SELECT_GO_TO_BOTTOM:
                GotoLocation( BOTTOM );
                break;

            case SELECT_FIND:
                FrmPopupForm( frmSearch );
                break;

            case SELECT_FIND_AGAIN:
                SearchAgain();
                break;

            case SELECT_ADD_BOOKMARK:
                FrmPopupForm( frmAddBookmark );
                break;

            case SELECT_VIEW_BOOKMARKS:
                FrmPopupForm( frmBookmarks );
                break;

            case SELECT_OPEN_LIBRARY:
                FrmGotoForm( frmLibrary );
                break;

            case SELECT_DETAILS:
                FrmPopupForm( frmDetails );
                break;

            case SELECT_PREFS:
                FrmPopupForm( frmPrefs );
                break;

            case SELECT_BUTTON_ACTION:
                FrmPopupForm( frmButton );
                break;

            case SELECT_TAP_ACTION:
                FrmPopupForm( frmControl );
                break;

            case SELECT_GESTURE_ACTION:
                FrmPopupForm( frmGesture );
                break;
                
            case SELECT_TOGGLE_AUTOSCROLL:
                DoAutoscrollToggle( AUTOSCROLL_TOGGLE );
                break;
                
            case SELECT_INCREASE_AUTOSCROLL:
                DoAutoscrollIncr();
                break;
                
            case SELECT_DECREASE_AUTOSCROLL:
                DoAutoscrollDecr();
                            
            default:
                break;
        }
    }
}



/* Perform action for specified screen region */
static void DoScreenAction
    (
    RegionType region   /* screen region */
    )
{
    DoSelectTypeAction( Prefs()->select[ Prefs()->controlMode ][ region ] );
}



/* Respond to pen tap */
void HandlePenDown
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    PenData.penDownX = PenData.moveFromX = event->screenX;
    PenData.penDownY = PenData.moveFromY = event->screenY;

    if ( LargeImage() )
        return;

    PenData.selectedControl = FindControlObject( event );

    if ( PenData.selectedControl != 0 ) {
        HighlightControl( PenData.selectedControl, ANCHOR_SELECTED );
    }
    else if ( Prefs()->controlMode != MODE3 && TopLeftY() < event->screenY &&
              event->screenY < TopLeftY() + ExtentY() ) {
        RegionType region;

        if ( Prefs()->controlMode == MODE1 ) {
            if ( PenData.penDownY <= ( TopLeftY() + ExtentY() / 4 ) )
                region = REGION1;
            else if ( PenData.penDownY <= ( TopLeftY() + ExtentY() / 2 ) )
                region = REGION2;
            else if ( PenData.penDownY <= ( TopLeftY() + 3 * ExtentY() / 4 ) )
                region = REGION3;
            else
                region = REGION4;
        }
        else {
            if ( PenData.penDownY <= ( TopLeftY() + ExtentY() / 2 ) ) {
                if ( PenData.penDownX <= ( TopLeftX() + ExtentX() / 2 ) )
                    region = REGION1;
                else
                    region = REGION2;
            }
            else {
                if ( PenData.penDownX <= ( TopLeftX() + ExtentX() / 2 ) )
                    region = REGION3;
                else
                    region = REGION4;
            }
        }
        DoScreenAction( region );
    }
}



/* Respond to pen movement */
void HandlePenMove
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Int16 mouseDelta;

    if ( LargeImage() && 0 <= PenData.moveFromX ) {
        if ( 0 < event->screenY && event->screenY < MaxExtentY() ) {
            PenData.penMoveX = event->screenX;
            PenData.penMoveY = event->screenY;

            newX += 2 * ( PenData.moveFromX - PenData.penMoveX );
            newY += 2 * ( PenData.moveFromY - PenData.penMoveY );

            if ( ( imageWidth - MaxExtentX() ) <= newX )
                newX = imageWidth - MaxExtentX();
            if ( newX < 0 )
                newX = 0;

            if ( ( imageHeight - MaxExtentY() ) <= newY )
                newY = imageHeight - MaxExtentY();
            if ( newY < 0 )
                newY = 0;

            PenData.moveFromX = PenData.penMoveX;
            PenData.moveFromY = PenData.penMoveY;

            DoImageMove( newX, newY );
        }
        return;
    }
    PenData.penMoveY    = event->screenY;
    mouseDelta          = PenData.penDownY - PenData.penMoveY;
    if ( PenData.selectedControl != 0 && ( ( mouseDelta < -3 ) || ( 3 < mouseDelta ) ) ) {
        HighlightControl( PenData.selectedControl, ANCHOR_UNSELECTED );
        PenData.selectedControl = 0;
    }
    if ( 0 <= PenData.selectedControl ) {
        DoPageMove( PenData.penMoveY - PenData.moveFromY );
        PenData.moveFromY = PenData.penMoveY;
    }
}



/* Respond to pen release */
void HandlePenUp
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    if ( PenData.selectedControl != 0 )
        HighlightControl( PenData.selectedControl, ANCHOR_UNSELECTED );

    if ( PenData.selectedControl == FindControlObject( event ) )
        DoControlAction( PenData.selectedControl );

    ClearPenData();
    PenData.moveFromX = -1;
    PenData.moveFromY = -1;
}



/* Retrieve the bounds of an object */
void GetControlBounds
    (
    const FormType* form,       /* pointer to memory block that contains the form */
    const UInt16    objectId,   /* ID of an object in the form */
    const Int16     controlID   /* control ID of the object */
    )
{
    ErrFatalDisplayIf( objectId != ( 1000 + controlID ), "GetControlBounds: the objects in the form MUST have an ID equal to 1000 + control ID" );

    FrmGetObjectBounds( form, FrmGetObjectIndex( form, objectId ), &controls[ controlID ].bounds );
}



/* Go to location in record */
void GotoLocation
    (
    const Int16 location    /* location in record ( percentage from top measured in steps of 10% ) */
    )
{
    MetaRecord* meta;
    Int16       adjustment;

    meta = (MetaRecord*) MemHandleLock( GetMetaRecord() );
    ErrFatalDisplayIf( meta == NULL, "MemHandleLock failed" );

    adjustment = -( meta->verticalOffset - ExtentY() + location * ( meta->height / 10 + 1 ) );

    MemHandleUnlock( GetMetaRecord() );

    DoPageMove( adjustment );
}



/* Toggle Autoscroll */
void DoAutoscrollToggle
    (
    AutoscrollType toggle
    )
{
    FormType* mainForm;
    
    mainForm = FrmGetFormPtr( Prefs()->toolbar );
    
    if ( toggle == AUTOSCROLL_TOGGLE )
        Prefs()->autoscrollEnabled = ! Prefs()->autoscrollEnabled;
    else if ( toggle == AUTOSCROLL_ON )
        Prefs()->autoscrollEnabled = true;
    else
        Prefs()->autoscrollEnabled = false;
    
    if ( Prefs()->toolbar != frmMainNone && ! LargeImage() &&
         MainFormWinActive() ) { 
	    if ( Prefs()->autoscrollEnabled ) {
	    	FrmHideObject( mainForm, FrmGetObjectIndex( mainForm, bmpAutoscrollStart ) );
	    	FrmShowObject( mainForm, FrmGetObjectIndex( mainForm, bmpAutoscrollStop ) );
	    } else {
	    	FrmHideObject( mainForm, FrmGetObjectIndex( mainForm, bmpAutoscrollStop ) );
	    	FrmShowObject( mainForm, FrmGetObjectIndex( mainForm, bmpAutoscrollStart ) );
		}
    }
}



/* Increase Autoscroll rate */
void DoAutoscrollIncr()
{ 
    if ( 10 <= Prefs()->autoscrollInterval )
        Prefs()->autoscrollInterval -= 10;
}



/* Decrease Autoscroll rate */
void DoAutoscrollDecr()
{ 
    Prefs()->autoscrollInterval += 10;
}



/* Returns whether MainForm's Window is active and OK to draw toolbars on,
   or not (e.g. obscured by menu, etc.) */
Boolean MainFormWinActive()
{
    FormType*   mainForm;
    WinHandle   mainWindow;
    WinHandle   activeWindow;

    mainForm        = FrmGetFormPtr( Prefs()->toolbar );
    mainWindow      = FrmGetWindowHandle( mainForm );
    activeWindow    = WinGetActiveWindow();
    if ( mainWindow == activeWindow )
        return true;
    else
        return false;
}
