/*
 * $Id: documentdata.c,v 1.41 2001/12/14 19:02:31 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "categorydata.h"
#include "const.h"
#include "db.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "history.h"
#include "bookmark.h"
#include "image.h"
#include "os.h"
#include "paragraph.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "uncompress.h"
#include "util.h"


Err OpenDocument( const LocalID dbID, UInt16 cardNo ) PLKRDB_SECTION;
Err ReturnMetaHandle( UInt16 id, const UInt16 num, Boolean update, 
        MemHandle* hand ) PLKRDB_SECTION;
Err ReturnRecordHandle( const UInt16 id, MemHandle* hand ) PLKRDB_SECTION;
Err ReturnNextRecordHandle( UInt16* index, MemHandle* hand, 
        UInt16 * uniqID ) PLKRDB_SECTION;
Err AddBookmarkRecord( MemHandle* hand ) PLKRDB_SECTION;
Err RemoveBookmarkRecord( void ) PLKRDB_SECTION;
Err RenameDocument( Char* newName, Char* oldName, 
        UInt16 oldCardNo ) PLKRDB_SECTION;
UInt16 ResizeRecord( const UInt16 id, const UInt16 size, 
        MemHandle* hand ) PLKRDB_SECTION;
UInt16 GetNumOfRecords( void ) PLKRDB_SECTION;
UInt16 GetReservedID( UInt16 name ) PLKRDB_SECTION;
UInt16 GetInternalID( UInt16 name ) PLKRDB_SECTION;
UInt16 GetDefaultCategories( LocalID dbID, UInt16 cardNo ) PLKRDB_SECTION;
Boolean GetBitStatus( const UInt16 record, Int16 reference ) PLKRDB_SECTION;
void CloseDocument( void ) PLKRDB_SECTION;
void SetBitStatus( const UInt16 record, Int16 reference, 
        Boolean setStatus ) PLKRDB_SECTION;
void DeleteMetaDocument( Char* name ) PLKRDB_SECTION;

static Boolean TextImageRecord( UInt16 uid ) PLKRDB_SECTION;
static DocType GetIndexData( void ) PLKRDB_SECTION;
static Err AddRecord( UInt16 uid, UInt16 size ) PLKRDB_SECTION;
static Err GetInternalIndexData( void ) PLKRDB_SECTION;
static Err InitializeMetaDocument( void ) PLKRDB_SECTION;
static Err OpenMetaDocument( Char* name, UInt32 date ) PLKRDB_SECTION;
static Err RemoveMetaRecord( const UInt16 id ) PLKRDB_SECTION;
static Int16 FindRecord( DmOpenRef dbRef, UInt16 item, MemHandle* handle, 
                Int16* index ) PLKRDB_SECTION;
static void CheckSize( const UInt16 id, Int16 reference, 
                MemHandle handle ) PLKRDB_SECTION;
static void ClearRecordIDs( void ) PLKRDB_SECTION;
static void InitializeBookmarkRecord( MemHandle handle ) PLKRDB_SECTION;
static void InitializeMetaRecord( MemHandle handle, UInt16 uid, UInt16 size ) PLKRDB_SECTION;
static void InitializeIndexRecord( MemHandle handle ) PLKRDB_SECTION;
static void InitializeReferenceRecord( MemHandle handle, UInt16 uid ) PLKRDB_SECTION;
static void InitializeSessionRecord( MemHandle handle ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define REFERENCE_RECORD_SIZE   ( 64 + sizeof( UID ) )
#define SESSION_RECORD_SIZE     sizeof( History )
#define BOOKMARK_RECORD_SIZE    3 * sizeof( UInt16 )
#define BIT_ON                  1
#define INDEX_RECORD            1
#define UNFILED_CATEGORY        1


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static DmOpenRef    ViewerDocument;
static DmOpenRef    MetaDocument;

static UInt16       reservedRecords[ MAX_RESERVED ];
static UInt16       internalRecords[ MAX_INTERNAL ];



/* Clear the arrays holding record IDs */
static void ClearRecordIDs( void )
{
    MemSet( (void*) &reservedRecords, MAX_RESERVED * sizeof( UInt16 ), 0 );
    MemSet( (void*) &internalRecords, MAX_INTERNAL * sizeof( UInt16 ), 0 );
}



/* Check if referenced record is a text/image record */
static Boolean TextImageRecord
    (
    UInt16 uid   /* unique ID to check */
    )
{
    UInt16 i;

    if ( uid == 0 )
        return false;

    /* if it is not one of the reserved internal records (except for the home 
       page) then it is a text/image record */
    for ( i = 1; i < MAX_INTERNAL; i++ )
        if ( uid == internalRecords[ i ] )
            return false;
    return true;
}



/* Return the reserved record's unique ID */
UInt16 GetReservedID
    (
    UInt16 name /* record "name" */
    )
{
    if ( name < MAX_RESERVED )
        return reservedRecords[ name ];
    else
        return 0;
}



/* Return the internal record's unique ID */
UInt16 GetInternalID
    (
    UInt16 name /* record "name" */
    )
{
    if ( name < MAX_INTERNAL )
        return internalRecords[ name ];
    else
        return 0;
}



/* Callback function that compares unique IDs */
Int16 CompareUID
    (
    void*               rec1,           /* record to sort */
    void*               rec2,           /* record to sort */
    Int16               other,          /* any other custom information you want
                                           passed to the comparison function */
    SortRecordInfoPtr   rec1SortInfo,   /* structure that specify unique sorting information */
    SortRecordInfoPtr   rec2SortInfo,   /* structure that specify unique sorting information */
    MemHandle           appInfoH        /* handle to the document's application info block */
    )
{
    Int16 result;

    SET_A4_FROM_A5 

    /*   0 if rec1 = rec2 */
    /* < 0 if rec1 < rec2 */
    /* > 0 if rec1 > rec2 */
    if ( ( (UID*) rec1 )->uid < ( (UID*) rec2 )->uid )
        result = -1;
    else if ( ( (UID*) rec1 )->uid > ( (UID*) rec2 )->uid )
        result = 1;
    else
        result = 0;

    RESTORE_A4 

    return result;
}



/* Find record in document */
static Int16 FindRecord
    (
    DmOpenRef   docRef, /* reference to document */
    UInt16      item,   /* UID to search for */
    MemHandle*  handle, /* upon successful return, the handle to the record */
    Int16*      index   /* upon successful return, the index of the record,
                           pass NULL for this parameter if you don't want to 
                           retrieve this value */
    )
{
    Int16 l;
    Int16 r;

    l = 0;
    r = DmNumRecords( docRef ) - 1;
    while ( l <= r ) {
        UID*    uid;
        Int16   x;

        x = ( r + l ) / 2;

        *handle = DmQueryRecord( docRef, x );
        if ( *handle == NULL )
            break;

        uid = (UID*) MemHandleLock( *handle );
        ErrFatalDisplayIf( uid == NULL, "FindRecord: MemHandleLock failed" );

        if ( item == uid->uid ) {
            MemHandleUnlock( *handle );
            if ( index )
                *index = x;
            return 0;
        }
        if ( item < uid->uid )
            r = x - 1;
        else
            l = x + 1;

        MemHandleUnlock( *handle );
    }
    return 1;
}



/* Initialize the meta record */
static void InitializeMetaRecord
    (
    MemHandle   handle, /* handle to record */
    UInt16      uid,    /* unique ID */
    UInt16      size    /* record size */
    )
{
    MetaRecord*     meta;
    UInt16          depth;
    FontModeType    font;

    meta = (MetaRecord*) MemHandleLock( handle );
    ErrFatalDisplayIf( meta == NULL, "InitializeMetaRecord: MemHandleLock failed" );

    depth   = Prefs()->screenDepth;
    font    = Prefs()->fontMode;

    DmSet( (void*) meta, 0, size, 0 );
    DmWrite( (void*) meta, OFFSETOF( MetaRecord, uid ), (void*) &uid,
        sizeof( UInt16 ) );
    DmWrite( (void*) meta, OFFSETOF( MetaRecord, depth ), 
        (void*) &depth, sizeof( UInt16 ) );
    DmWrite( (void*) meta, OFFSETOF( MetaRecord, font ), 
        (void*) &font, sizeof( FontModeType ) );

    MemHandleUnlock( handle );
}



/* Initialize the session data */
static void InitializeSessionRecord
    (
    MemHandle handle    /* handle to record */
    )
{
    History*    sessionPtr;
    History     history;

    sessionPtr = (History*) MemHandleLock( handle );
    ErrFatalDisplayIf( sessionPtr == NULL, "InitializeSessionRecord: MemHandleLock failed" );

    MemSet( (void*) &history, sizeof( History ), 0 );
    history.uid                     = SESSION_DATA_ID;
    history.addHistory              = false;
    history.records[ 0 ].recordId   = HOME_PAGE_ID;
    DmWrite( (void*) sessionPtr, 0, (void*) &history, sizeof( History ) );

    MemHandleUnlock( handle );
}



/* Initialize the show images or link data record */
static void InitializeReferenceRecord
    (
    MemHandle   handle, /* handle to show images or link record */
    UInt16      uid     /* unique ID */
    )
{
    void* referencePtr;

    referencePtr = MemHandleLock( handle );
    ErrFatalDisplayIf( referencePtr == NULL, "InitializeReferenceRecord: MemHandleLock failed" );

    DmSet( (void*) referencePtr, 0, REFERENCE_RECORD_SIZE, 0 );
    DmWrite( (void*) referencePtr, 0, (void*) &uid, sizeof( UInt16 ) );

    MemHandleUnlock( handle );
}



/* Initialize the bookmarks record */
static void InitializeBookmarkRecord
    (
    MemHandle handle    /* handle to record */
    )
{
    Char*   bookmarkPtr;
    UInt16  uid;
    UInt16  offset;

    bookmarkPtr = (Char*) MemHandleLock( handle );
    ErrFatalDisplayIf( bookmarkPtr == NULL, "InitializeBookmarkRecord: MemHandleLock failed" );

    uid     = INTERNAL_BOOKMARKS_ID;
    offset  = BOOKMARK_RECORD_SIZE;
    DmSet( (void*) bookmarkPtr, 0, BOOKMARK_RECORD_SIZE, 0 );
    DmWrite( (void*) bookmarkPtr, 0, (void*) &uid, sizeof( UInt16 ) );
    DmWrite( (void*) bookmarkPtr, 2 * sizeof( UInt16 ), (void*) &offset,
        sizeof( UInt16 ) );

    MemHandleUnlock( handle );
}



/* Initialize the index record */
static void InitializeIndexRecord
    (
    MemHandle handle    /* handle to record */
    )
{
    MetaIndexRecord*    indexPtr;
    UInt16              uid;
    UInt16              records;

    indexPtr = (MetaIndexRecord*) MemHandleLock( handle );
    ErrFatalDisplayIf( indexPtr == NULL, "InitializeIndexRecord: MemHandleLock failed" );

    uid     = INDEX_RECORD;
    records = MAX_INTERNAL;
    DmWrite( (void*) indexPtr, OFFSETOF( MetaIndexRecord, uid ), (void*) &uid,
        sizeof( UInt16 ) );
    DmWrite( (void*) indexPtr, OFFSETOF( MetaIndexRecord, records ),
        (void*) &records, sizeof( UInt16 ) );
    DmWrite( (void*) indexPtr, OFFSETOF( MetaIndexRecord, internal ),
        (void*) &internalRecords, MAX_INTERNAL * sizeof( UInt16 ) );

    MemHandleUnlock( handle );
}



/* Add record to document */
static Err AddRecord
    (
    UInt16 uid, /* unique ID of record to create */
    UInt16 size /* record size */
    )
{
    Err         err;
    MemHandle   handle;
    UInt16      docIndex;

    docIndex    = dmMaxRecordIndex;
    handle      = DmNewRecord( MetaDocument, &docIndex, size );
    if ( handle == NULL )
        return DmGetLastErr();

    if ( uid == INDEX_RECORD )
        InitializeIndexRecord( handle );
    else if ( TextImageRecord( uid ) )
        InitializeMetaRecord( handle, uid, size );
    else if ( uid == SESSION_DATA_ID )
        InitializeSessionRecord( handle );
    else if ( uid == SHOW_IMAGES_ID ||
              uid == LINK_TABLE_ID )
        InitializeReferenceRecord( handle, uid );
    else if ( uid == INTERNAL_BOOKMARKS_ID )
        InitializeBookmarkRecord( handle );

    err = DmReleaseRecord( MetaDocument, docIndex, true );

    DmInsertionSort( MetaDocument, (DmComparF*) CompareUID, 0 );

    return err;
}



/* Check the compression type and retrieve IDs for reserved records */
static DocType GetIndexData( void )
{
    MemHandle   hand;
    Err         err;
    DocType     docType;

    docType = DOCTYPE_UNKNOWN;

    err = FindRecord( ViewerDocument, INDEX_RECORD, &hand, NULL );
    if ( err == errNone ) {
        IndexRecord*    indexRecord;
        UInt16*         reserved;
        UInt16          i;

        indexRecord = (IndexRecord*) MemHandleLock( hand );
        ErrFatalDisplayIf( indexRecord == NULL, "GetIndexData: MemHandleLock failed" );

        docType     = (DocType) indexRecord->version;
        reserved    = indexRecord->reserved;

        for ( i = 0; i < indexRecord->records; i++ ) {
            UInt16 name;
            UInt16 id;

            name    = *reserved;
            id      = *( reserved + 1 );
            if ( name < MAX_RESERVED )
                reservedRecords[ name ] = id;
            reserved += sizeof( UInt16 );
        }
        MemHandleUnlock( hand );

        if ( docType != DOCTYPE_ZLIB && docType != DOCTYPE_DOC )
            docType = DOCTYPE_UNKNOWN;
    }
    return docType;
}



/* Retrieve ID's for internal records */
static Err GetInternalIndexData( void )
{
    MemHandle   hand;
    Err         err;

    err = FindRecord( MetaDocument, INDEX_RECORD, &hand, NULL );
    if ( err == errNone ) {
        MetaIndexRecord*    indexRecord;
        UInt16*             internal;
        UInt16              i;

        indexRecord = (MetaIndexRecord*) MemHandleLock( hand );
        ErrFatalDisplayIf( indexRecord == NULL, "GetInternalIndexData: MemHandleLock failed" );

        internal = indexRecord->internal;

        for ( i = 0; i < indexRecord->records; i++ ) {
            internalRecords[ i ] = *internal;
            internal++;
        }
        MemHandleUnlock( hand );
    }
    return err;
}



/* Initialize the document containing meta data */
static Err InitializeMetaDocument( void )
{
    Err         err;
    UID*        uid;
    MemHandle   handle;
    UInt16      i;
    UInt16      maxID;
    UInt16      lastIndex;

    lastIndex   = DmNumRecords( ViewerDocument ) - 1;
    handle      = DmQueryRecord( ViewerDocument, lastIndex );
    if ( handle == NULL )
        return 1;

    uid = (UID*) MemHandleLock( handle );
    ErrFatalDisplayIf( uid == NULL, "InitializeMetaDatabase: MemHandleLock failed" );

    maxID = uid->uid;
    MemHandleUnlock( handle );

    internalRecords[ _HOME_PAGE ] = reservedRecords[ HOME_PAGE ];
    for ( i = 1; i < MAX_INTERNAL; i++ )
        internalRecords[ i ] = maxID + i;

    err  = AddRecord( INDEX_RECORD, ( MAX_INTERNAL + 2 ) * sizeof( UInt16 ) );
    err |= AddRecord( SHOW_IMAGES_ID, REFERENCE_RECORD_SIZE );
    err |= AddRecord( SESSION_DATA_ID, SESSION_RECORD_SIZE );
    err |= AddRecord( LINK_TABLE_ID, REFERENCE_RECORD_SIZE );

    return err;
}



/* Open meta document */
static Err OpenMetaDocument
    (
    Char*   name,   /* name of the associated document */
    UInt32  date    /* creation date of the associated document */
    )
{
    Err     err;
    UInt16  metaVersion;
    UInt16  cardNo;
    LocalID metaID;
    Char    metaName[ dmDBNameLength ];

    metaVersion = 0;
    cardNo      = 0;

    StrCopy( metaName, "Plkr-" );
    StrNCat( metaName, name, dmDBNameLength );
    metaID = DmFindDatabase( cardNo, metaName );
    if ( metaID != NULL ) {
        UInt32 metaDate;

        metaDate = 0;

        DmDatabaseInfo( cardNo, metaID, NULL, NULL, &metaVersion, 
            &metaDate, NULL, NULL, NULL, NULL, NULL, NULL, NULL );

        if ( metaDate < date || metaVersion != MetaDocumentVersion )
            DmDeleteDatabase( cardNo, metaID );
        else {
            MetaDocument = DmOpenDatabase( cardNo, metaID, dmModeReadWrite );
            GetInternalIndexData();
            return 0;
        }
    }
    err = DmCreateDatabase( cardNo, metaName, ViewerAppID, MetaDocumentType, false );
    if ( err != errNone ) {
        MSG( "Couldn't create meta document\n" );
        FrmAlert( warnInsufficientMemory );
        return err;
    }
    metaID       = DmFindDatabase( cardNo, metaName );
    metaVersion  = MetaDocumentVersion;
    DmSetDatabaseInfo( cardNo, metaID, NULL, NULL, &metaVersion, 
        &date, NULL, NULL, NULL, NULL, NULL, NULL, NULL );

    MetaDocument = DmOpenDatabase( cardNo, metaID, dmModeReadWrite );

    err = InitializeMetaDocument();
    if ( err != errNone ) {
        CloseDocument();
        MSG( "Couldn't initialize meta document\n" );
        FrmAlert( warnInsufficientMemory );
    }
    return err;
}



/* Open document */
Err OpenDocument
    (
    const LocalID   dbID,   /* document ID */
    UInt16          cardNo  /* card the document is located on */
    )
{
    Err     err;
    UInt32  date;
    UInt16  version;
    DocType docType;
    Char    name[ dmDBNameLength ];

    ClearRecordIDs();
    ResetImageData();

    if ( dbID == NULL || OpenDatabase( dbID, cardNo, &ViewerDocument, &version,
                            name, &date ) != errNone ) {
        MSG( "no Plucker document installed\n" );
        return dmErrCantOpen;
    }
    docType = GetIndexData();
    if ( docType == DOCTYPE_ZLIB ) {
        if ( ZLibSupported() )
            uncompressFunc = UnZip;
        else {
            CloseDocument();
            MSG( "ZLib compressed documents not supported\n" );
            FrmCustomAlert( warnNoZLibSupport, name, NULL, NULL );
            return 1;
        }
    }
    else if ( docType == DOCTYPE_DOC )
        uncompressFunc = UnDoc;
    else {
        CloseDocument();
        MSG( "Not a valid Plucker document\n" );
        FrmCustomAlert( errUnknownType, name, NULL, NULL );
        return 1;
    }
    if ( reservedRecords[ HOME_PAGE ] != 0 ) {
        MemHandle hand;

        err = FindRecord( ViewerDocument, reservedRecords[ HOME_PAGE ], &hand, NULL );
    }
    else
        err = 1;

    if ( err != errNone ) {
        CloseDocument();
        MSG( "couldn't open/initialize document\n" );
        FrmCustomAlert( warnBrokenDocument, name, NULL, NULL );
    }
    else
        OpenMetaDocument( name, date );

    return err;
}

/* Close document */
void CloseDocument( void )
{
    ResetRecordReferences();

    if ( ViewerDocument == NULL )
        return;

    CloseDatabase( ViewerDocument );
    ViewerDocument = NULL;

    if ( MetaDocument == NULL )
        return;

    CloseDatabase( MetaDocument );
    MetaDocument = NULL;
}



/* Delete meta document */
void DeleteMetaDocument
    (
    Char* name  /* name of the associated document */
    )
{
    UInt16  cardNo;
    LocalID dbID;
    Char    metaName[ dmDBNameLength ];

    if ( MetaDocument != NULL )
        return;

    StrCopy( metaName, "Plkr-" );
    StrNCat( metaName, name, dmDBNameLength );

    cardNo  = 0;
    dbID    = DmFindDatabase( cardNo, metaName );
    if ( dbID != NULL )
        DmDeleteDatabase( cardNo, dbID );
}



/* Rename document and its associated meta document */
Err RenameDocument
    (
    Char*   newName,    /* new document name */
    Char*   oldName,    /* old document name */
    UInt16  oldCardNo   /* card that the old document resides on */
    )
{
    Err     err;
    Char    metaNewName[ dmDBNameLength ];
    Char    metaOldName[ dmDBNameLength ];
    LocalID dbID;
    UInt16  cardNo;

    cardNo  = 0;
    err     = 1;

    /* If a document with the given name doesn't already exists, rename it */
    dbID = DmFindDatabase( oldCardNo, newName );
    if ( dbID == NULL ) {
        dbID    = DmFindDatabase( oldCardNo, oldName );
        err     = DmSetDatabaseInfo( oldCardNo, dbID, newName, NULL, NULL, NULL, 
                    NULL, NULL, NULL, NULL, NULL, NULL, NULL );
    }
    if ( err != errNone )
        return err;

    /* Remove any "left-overs" */
    StrCopy( metaNewName, "Plkr-" );
    StrNCat( metaNewName, newName, dmDBNameLength );
    dbID = DmFindDatabase( cardNo, metaNewName );
    if ( dbID != NULL )
        DmDeleteDatabase( cardNo, dbID );

    /* Rename meta document ( if it exists ) */
    StrCopy( metaOldName, "Plkr-" );
    StrNCat( metaOldName, oldName, dmDBNameLength );
    dbID = DmFindDatabase( cardNo, metaOldName );
    if ( dbID != NULL )
        DmSetDatabaseInfo( cardNo, dbID, metaNewName, NULL, NULL, NULL, 
            NULL, NULL, NULL, NULL, NULL, NULL, NULL );

    return 0;
}



/* Return a handle to a meta record */
Err ReturnMetaHandle
    (
    UInt16          id,     /* unique ID to search for */
    const UInt16    num,    /* number of paragraphs in record */
    Boolean         update, /* if true, meta data should be updated */
    MemHandle*      hand    /* upon successful return, the handle to the record */
    )
{
    Err err;

    ErrFatalDisplayIf( hand == NULL, "ReturnMetaHandle: NULL argument" );

    if ( MetaDocument == NULL || id == 0 )
        return 1;

    err = FindRecord( MetaDocument, id, hand, NULL );
    if ( err == errNone && update && TextImageRecord( id ) ) {
        MetaRecord* meta;

        /* if bit depth or font has changed the height should be re-calculated */
        meta = (MetaRecord*) MemHandleLock( *hand );
        if ( meta->depth != Prefs()->screenDepth ||
             meta->font  != Prefs()->fontMode ) {
            Int16           height;
            UInt16          depth;
            FontModeType    font;

            height = 0;
            DmWrite( (void*) meta, OFFSETOF( MetaRecord, height ), (void*) &height, sizeof( Int16 ) );

            depth = Prefs()->screenDepth;
            DmWrite( (void*) meta, OFFSETOF( MetaRecord, depth ), (void*) &depth, sizeof( UInt16 ) );

            font = Prefs()->fontMode;
            DmWrite( (void*) meta, OFFSETOF( MetaRecord, font ), (void*) &font, sizeof( FontModeType ) );
        }
        MemHandleUnlock( *hand );
    }
    else if ( err != errNone && id != internalRecords[ _INTERNAL_BOOKMARKS ] ) {
        UInt16 size;

        size = sizeof( MetaRecord ) + num * sizeof( MetaParagraph );
        err = AddRecord( id, size );
        if ( err == errNone )
            err = FindRecord( MetaDocument, id, hand, NULL );
    }
    return err;
}



/* Return a handle to a record */
Err ReturnRecordHandle
    (
    const UInt16    id,     /* unique ID to search for */
    MemHandle*      hand    /* upon successful return, the handle to the record */
    )
{
    Err err;

    if ( ViewerDocument == NULL || id == 0 )
        return 1;

    ErrFatalDisplayIf( hand == NULL, "ReturnRecordHandle: NULL argument" );

    err = FindRecord( ViewerDocument, id, hand, NULL );

    return err;
}


/* Return a handle to a record */
Err ReturnLastHandle
    (
    MemHandle*      hand    /* upon successful return, the handle to the record */
    )
{

    if ( ViewerDocument == NULL)
        return 1;

    ErrFatalDisplayIf( hand == NULL, "ReturnRecordHandle: NULL argument" );

    *hand = DmQueryRecord( ViewerDocument, DmNumRecords(ViewerDocument)-1 );
    if ( *hand == NULL ) return 1;

    return 0;
}



/* Return a handle to next record */
Err ReturnNextRecordHandle
    (
    UInt16*     index,  /* pointer to index of a known record */
    MemHandle*  hand,   /* upon successful return, the handle to the record */
    UInt16*     uniqID  /* upon successful return, the record's unique ID */
    )
{
    Header* record;
    UInt16  lastIndex;

    ErrFatalDisplayIf( hand == NULL, "ReturnNextRecordHandle: NULL argument" );

    if ( *index == 0 )
        *index = 1;

    lastIndex = DmNumRecords( ViewerDocument ) - 1;
    for ( ;; ) {
        if ( *index <= lastIndex )
            *hand = DmQueryRecord( ViewerDocument, *index );
        else
            return 1;

        if ( *hand == NULL )
            return 1;

        record = (Header*) MemHandleLock( *hand );
        ErrFatalDisplayIf( record == NULL, "ReturnNextRecordHandle: MemHandleLock failed" );

        if ( record->type == DATATYPE_PHTML || 
             record->type == DATATYPE_PHTML_COMPRESSED ) {
            *uniqID = record->uid;
            MemHandleUnlock( *hand );
            return 0;
        }
        MemHandleUnlock( *hand );
        ( *index )++;
    }
}



/* Remove meta record */
static Err RemoveMetaRecord
    (
    const UInt16 id /* unique ID to remove */
    )
{
    Err         err;
    MemHandle   hand;
    Int16       docIndex;

    if ( MetaDocument == NULL )
        return 1;

    err = FindRecord( MetaDocument, id, &hand, &docIndex );
    if ( err == errNone )
        err = DmRemoveRecord( MetaDocument, docIndex );

    return err;
}



/* Return number of records in document */
UInt16 GetNumOfRecords( void )
{
    return DmNumRecords( ViewerDocument );
}



/* Resize record */
Err ResizeRecord
    (
    const UInt16    id,     /* unique ID to resize */
    const UInt16    size,   /* new size */
    MemHandle*      hand    /* upon successful return, the handle to the record */
    )
{
    Err     err;
    Int16   docIndex;

    if ( id == 0 )
        return 1;

    err = FindRecord( MetaDocument, id, hand, &docIndex );
    if ( err == errNone ) {
        *hand = DmResizeRecord( MetaDocument, docIndex, size );
        DmInsertionSort( MetaDocument, (DmComparF*) CompareUID, 0 );
    }
    return err;
}



/* Check that the size of the record is enough to hold the required data otherwise resize it */
static void CheckSize
    (
    const UInt16    id,         /* record ID to check */
    Int16           reference,  /* reference for the bit to check */
    MemHandle       handle      /* handle to record */
    )
{
  UInt16 size;
  UInt16 origSize;

  size      = reference / 8 + 1 + sizeof( UID );
  origSize  = MemHandleSize( handle );

  if ( origSize < size ) {
    void* recPtr;

    ResizeRecord( id, size, &handle );

    recPtr = MemHandleLock( handle );
    DmSet( recPtr, origSize, size - origSize, 0 );
    MemHandleUnlock( handle );
  }
}



/* Get the status for a specific bit in the given record */
Boolean GetBitStatus
    (
    const UInt16    id,         /* record ID to check */
    Int16           reference   /* reference for the bit to check */
    )
{
  MemHandle handle;
  Char*     recordPtr;
  Int16     value;

  handle = NULL;

  if ( ReturnMetaHandle( id, 0, false, &handle ) != errNone )
    return false;

  CheckSize( id, reference, handle );

  recordPtr = (Char*) MemHandleLock( handle );
  ErrFatalDisplayIf( recordPtr == NULL, "GetBitStatus: MemHandleLock failed" );

  recordPtr += sizeof( UID );

  reference    -= 1;
  value         = ( *( recordPtr + reference / 8 ) >> reference % 8 ) & 0x01;

  MemHandleUnlock( handle );

  return ( value == BIT_ON );
}



/* Set the status of a bit in the specified record */
void SetBitStatus
    (
    const UInt16    id,         /* record ID to set */
    Int16           reference,  /* reference to set */
    Boolean         setStatus   /* indicates whether the bit should be set or unset */
    )
{
    MemHandle   handle;
    Char*       recordPtr;
    Char        bitStatus;
    Char        byte;
    Char        bit;
    Int16       byteOffset;

    handle = NULL;
    if ( ReturnMetaHandle( id, 0, false, &handle ) != errNone )
        return;

    CheckSize( id, reference, handle );

    recordPtr = (Char*) MemHandleLock( handle );
    ErrFatalDisplayIf( recordPtr == NULL, "SetBitStatus: MemHandleLock failed" );

    reference -= 1;
    byteOffset = reference / 8 + sizeof( UID );

    byte    = *( recordPtr + byteOffset );
    bit     = 0x01 << reference % 8;

    if ( setStatus )
        bitStatus = byte | bit;
    else
        bitStatus = byte & ~bit;

    DmWrite( recordPtr, byteOffset, (void*) &bitStatus, 1 );

    MemHandleUnlock( handle );
}



/* Add bookmark record to document */
Err AddBookmarkRecord
    (
    MemHandle* hand /* upon successful return, the handle to the record */
    )
{
    Err err;

    err = AddRecord( internalRecords[ _INTERNAL_BOOKMARKS ], BOOKMARK_RECORD_SIZE );
    if ( err == errNone )
        err = ReturnMetaHandle( internalRecords[ _INTERNAL_BOOKMARKS ], 
                0, false, hand );

    return err;
}



/* Remove bookmark record from document */
Err RemoveBookmarkRecord( void )
{
    return RemoveMetaRecord( internalRecords[ _INTERNAL_BOOKMARKS ] );
}



/* Retrieve the default category/categories from the document */
UInt16 GetDefaultCategories
    (
    LocalID dbID,   /* document ID */
    UInt16  cardNo  /* card the document is located on */
    )
{
    UInt16      categories;
    MemHandle   handle;
    Err         err;

    if ( ViewerDocument != NULL )
        return UNFILED_CATEGORY;

    ViewerDocument = DmOpenDatabase( cardNo, dbID, dmModeReadOnly );
    if ( ViewerDocument == NULL )
        return UNFILED_CATEGORY;

    categories = 0;

    GetIndexData();

    err = FindRecord( ViewerDocument, CATEGORY_ID, &handle, NULL );
    if ( err != errNone )
        categories = UNFILED_CATEGORY;
    else {
        Header* categoryRecord;
        Char*   name;
        Char    categoryName[ dmCategoryLength ];
        UInt16  totalSize;
        UInt16  size;
        UInt8   index;

        categoryRecord = (Header*) MemHandleLock( handle );
        ErrFatalDisplayIf( categoryRecord == NULL, "GetDefaultCategories: MemHandleLock failed" );

        size        = 0;
        totalSize   = categoryRecord->size;
        name        = (Char*) ( categoryRecord + 1 );
        do {
            // create NULL terminated category name of valid length
            StrNCopy( categoryName, name, dmCategoryLength );
            categoryName[ dmCategoryLength - 1 ] = '\0';

            index = GetCategoryIndex( categoryName );
            if ( index == dmAllCategories ) {
                index = AddCategoryToFreePosition( categoryName );
            }
            if ( index != dmAllCategories )
                categories |= ( 1 << index );

            size    += StrLen( name ) + 1;
            name    += StrLen( name ) + 1;
        } while ( size < totalSize );

        MemHandleUnlock( handle );
    }
    CloseDatabase( ViewerDocument );
    ViewerDocument = NULL;

    if ( categories == 0 )
        return UNFILED_CATEGORY;
    else
        return categories;
}
