/*
 * $Id: externalform.c,v 1.21 2001/09/22 11:13:50 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "uncompress.h"
#include "util.h"
#include "browser.h"
#include "history.h"

void SetLinkIndex( Int16 index ) PLKRDB_SECTION;
void AddURLToField( FieldType* fldPtr, Int16 index ) PLKRDB_SECTION;
void WriteMemo( FieldType* field ) PLKRDB_SECTION;

static void ExternalLinksFormInit( void ) PLKRDB_SECTION;



/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static Int16 linkIdx;



/* Set current link in the ExternalLinks record */
void SetLinkIndex
    (
    Int16 index /* index of record */
    )
{
    linkIdx = index;
}



/* Initialize field with URL string */
void AddURLToField
    (
    FieldType*  fldPtr, /* pointer to field */
    Int16       index   /* index in URL record */
    )
{
    MemHandle   linkHandle;
    Char*       text;

    linkHandle = NULL;
    text = NULL;
    if ( ReturnRecordHandle( PLUCKER_LINKS_ID, &linkHandle ) == errNone ) {
        MemHandle   uncompressHandle;
        Header*     linkDocument;
        Int16       count;
        Int16       offset;
        Int16*      numP;

        count   = index;
        offset  = 0;

        linkDocument = (Header*) MemHandleLock( linkHandle );
        numP = (Int16*) ( linkDocument + 1 );

        while ( *numP < index ) {
            offset  = *numP;
            numP   += 2;
        }
        numP    += 1;
        count   -= offset;

        MemHandleUnlock( linkHandle );

        if ( ReturnRecordHandle( *numP, &linkHandle ) != errNone )
            return;

        uncompressHandle    = NULL;
        linkDocument        = (Header*) MemHandleLock( linkHandle );
        if ( linkDocument->type == DATATYPE_LINKS_COMPRESSED ) {
            uncompressHandle = Uncompress( linkDocument );
            MemHandleUnlock( linkHandle );
            linkHandle = NULL;
            if ( uncompressHandle )
                text = (Char*) MemHandleLock( uncompressHandle );
        }
        else if ( linkDocument->type == DATATYPE_LINKS ) {
            text = (Char*) ( linkDocument + 1 );
        }

        if ( text != NULL ) {
            while ( --count )
                text += StrLen( text ) + 1;
            FldInsert( fldPtr, text, StrLen( text ) );
        }
        if ( linkHandle != NULL )
            MemHandleUnlock( linkHandle );
        if ( uncompressHandle != NULL )
            MemHandleUnlock( uncompressHandle );
    }
    if ( text == NULL ) {
        Char buff[ 100 ];

        SysCopyStringResource( buff, strExternNoURL );
        FldInsert( fldPtr, buff, StrLen( buff ) );
    }
}



/* Initialize the externallinks form */
static void ExternalLinksFormInit( void )
{
    FormType*   externallinksForm;
    FieldType*  field;

    field = (FieldType*) GetObjectPtr( frmExternalLinksLink );
    if (linkIdx>=0) AddURLToField( field, linkIdx );
        else FldInsert( field,"http://www.",11);

    externallinksForm = FrmGetFormPtr( frmExternalLinks );
    FrmDrawForm( externallinksForm );
}


void GoToURL(FieldType *field)
{
    UInt16 length = FldGetTextLength( field );
    Char *text = FldGetTextPtr( field );
    Char *url;
    MemHandle mh;
    Int16 recordId;

    url=MemPtrNew(length+1);
    MemMove(url,text,length);
    url[length]=0;


    mh=ResolveLinkByURL(url,&recordId);
    MemPtrFree(url);
    if (mh) {
         AddToHistory(recordId);
	 FrmGotoForm( Prefs()->toolbar );
    }
    FrmDrawForm(FrmGetActiveForm());
}

/* Write the text from a TextField to a Memo */
void WriteMemo
    (
    FieldType* field    /* field to get text from */
    )
{
    static UInt16 index     = dmMaxRecordIndex;
    static UInt32 secs      = 0;
    static UInt32 offset    = 0;

    Char        header[ 40 ];
    DmOpenRef   MemopadDB;
    MemHandle   recH;
    UInt16      mode;
    UInt16      length;

    length = FldGetTextLength( field );
    if ( length == 0 )
        return;

    if ( secs == 0 ) {
        DateTimeType dateTime;

        secs = TimGetSeconds();
        TimSecondsToDateTime( secs, &dateTime );
        StrPrintF( header, "%s %4d-%02d-%02d %02d:%02d\n", "Plucker URLs", 
            dateTime.year, dateTime.month, dateTime.day, dateTime.hour, dateTime.minute );
    }
    else {
        StrCopy( header, "\n" );
    }

    mode = dmModeReadWrite;
    if ( ( MemopadDB = DmOpenDatabaseByTypeCreator( 'DATA', 'memo', mode ) ) == NULL )
        return;

    if ( index == dmMaxRecordIndex )
        recH = DmNewRecord( MemopadDB, &index, StrLen( header ) + length + 1 );
    else
        recH = DmResizeRecord( MemopadDB, index, offset + StrLen( header ) + length + 1 );

    /* Should we pop up an alert form here?? */
    if ( recH != NULL ) {
        Char    null;
        Char*   text;
        MemPtr  recP;
        UInt16  attr;
        Int16   category;

        null = '\0';
        text = FldGetTextPtr( field );
        recP = MemHandleLock( recH );

        DmWrite( recP, offset, header, StrLen( header ) );
        DmWrite( recP, offset + StrLen( header ), text, length );
        DmWrite( recP, offset + StrLen( header ) + length, &null, 1 );

        MemHandleUnlock( recH );

        DmRecordInfo( MemopadDB, index, &attr, NULL, NULL );
        attr &= ~dmRecAttrCategoryMask;

        category = CategoryFind( MemopadDB, "Plucker" );

        if ( category == dmAllCategories )
            attr |= dmUnfiledCategory;
        else
            attr |= category;

        DmSetRecordInfo( MemopadDB, index, &attr, NULL );
        DmReleaseRecord( MemopadDB, index, true );

        offset += StrLen( header ) + length;
    }
    DmCloseDatabase( MemopadDB );
}



/* Event handler for the externallinks form */
Boolean ExternalLinksFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean     handled;
    FieldType*  field;

    SET_A4_FROM_A5 

    handled = false;
    field   = (FieldType*) GetObjectPtr( frmExternalLinksLink );

    switch ( event->eType ) {
        case ctlSelectEvent:
            if ( event->data.ctlEnter.controlID == frmExternalLinksCopy ) 
                WriteMemo( field );
            else if ( event->data.ctlEnter.controlID == frmExternalLinksFetch ) {
		GoToURL(field);
                break;
            } else if ( event->data.ctlEnter.controlID != frmExternalLinksBack )
                break;
            FrmGotoForm( Prefs()->toolbar );
            handled = true;
            break;

        case frmOpenEvent:
            ExternalLinksFormInit();
            handled = true;
            break;

        case frmCloseEvent:
            handled = false;
            break;

        default:
            handled = false;
    }

    RESTORE_A4 

    return handled;
}
