/*
 * fetchdb.c:
 *
 * developed from Net.c in the Palm LiveJournal project
 *
 * Copyright (C) 2000-2001 Kev Vance <kvance@tekktonik.net>
 * Copyright (c) 2002 James McKenzie <james@fishsoup.dhs.org>,
 * All rights reserved.
 *
 */

/* static char rcsid[] = "$Id:$"; */

/*
 * $Log:$
 */

/* 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <PalmOS.h>
#include <System/NetMgr.h>
#include <System/DateTime.h>

#include "resourceids.h"
#include "util.h"
#include "browser.h"
#include "fetchdb.h"
#include "documentdata.h"
#include "session.h"
#include "prefsdata.h"



#define MAXBUF 512
#define CACHE_SIZE 20
#define CHUNK_SIZE 2048
#define LINE_LEN  100

static RectangleType bounds = { {5, 20}, {140, 130} };
static Char linbuf[LINE_LEN];

static unsigned char *buffer = NULL;
static NetSocketRef sock;
static Int32 counter, docsize;
static UInt16 AppNetRefnum;
static Int32 AppNetTimeout;

static int yesfetch = 0;

static void
UrlCat (Char * x, Char * y)
  PLKRDB_SECTION;
     static int CompleteRead (Char * buf, UInt32 len) PLKRDB_SECTION;
     static int CompleteDmRead (void *recordP, UInt32 len) PLKRDB_SECTION;
     static Char *ReadInDB (void) PLKRDB_SECTION;
     static int CheckSig (void) PLKRDB_SECTION;
     static void Blat (Char * msg, Int16 y) PLKRDB_SECTION;
     static void empty_counter (void) PLKRDB_SECTION;
     static void breakpoint (char *, int) PLKRDB_SECTION;
     static void update_progress (void) PLKRDB_SECTION;
     static int IsNetUp (void) PLKRDB_SECTION;


     void set_progress (Char * t)
{
  ControlType *ct;
  if (FrmGetActiveFormID () != frmMainTop)
    return;

  ct = (ControlType *) GetObjectPtr (frmMainPercentPopup);
  if (!ct)
    return;
  CtlSetLabel (ct, t);
}

static void
update_progress (void)
{
  RectangleType rt;
  Int32 pct;
  Char offText[8];

  rt.topLeft.x = bounds.topLeft.x + 1;
  rt.topLeft.y = 60;
  rt.extent.x = (((Int32) (bounds.extent.x - 2)) * counter) / docsize;
  rt.extent.y = 10;

  WinDrawRectangle (&rt, 0);

  pct = (counter * 100) / docsize;
  StrPrintF (offText, "%ld%%", pct);
  set_progress (offText);

}

static void
empty_counter (void)
{
  RectangleType rt;

  rt.topLeft.x = bounds.topLeft.x + 1;
  rt.topLeft.y = 60;
  rt.extent.x = bounds.extent.x - 2;
  rt.extent.y = 10;

  WinDrawRectangleFrame (simpleFrame, &rt);

}


static void
UrlCat (Char * x, Char * y)
{
  int i, j, hex;
  Err err;

  j = 0;
  *x = 0;

  for (i = 0; i < StrLen (y); i++)
    {
      if (MemPtrSize (x) < j + 4)
        {
          /* Out of memory, request more */
          err = MemPtrResize (x, MemPtrSize (x) + 100);
          if (err)
            /* Can't get more memory, abort! */
            break;
        }
      if ((y[i] >= 'a' && y[i] <= 'z') || // a-z
          (y[i] >= 'A' && y[i] <= 'Z') || // A-Z
          (y[i] >= '0' && y[i] <= '9'))
        {                       // 0-9

          x[j++] = y[i];
          x[j] = '\0';
        }
      else if (y[i] == ' ')
        {
          x[j++] = '+';
          x[j] = '\0';
        }
      else
        {
          x[j++] = '%';
          /* Left hex digit */
          hex = y[i] / 16;
          if (hex < 10)
            hex += '0';
          else
            hex += 'A' - 10;
          x[j++] = hex;
          /* Right hex digit */
          hex = y[i] % 16;
          if (hex < 10)
            hex += '0';
          else
            hex += 'A' - 10;
          x[j++] = hex;
          x[j] = '\0';
        }
    }
}


static int
CompleteRead (Char * buf, UInt32 len)
{
  Int32 i;
  UInt32 red = 0;
  Err ifErrs;

  while (len)
    {
      i =
        NetLibReceive (AppNetRefnum, sock, buf, len, 0, NULL, NULL,
                       AppNetTimeout, &ifErrs);
      if (i <= 0)
        break;
      buf += i;
      red += i;
      len -= i;
      counter += i;
    }
  return red;
}


static int
CompleteDmRead (void *recordP, UInt32 len)
{
  UInt32 off = 0;
  Int32 i;
  Err ifErrs;

  while (len)
    {
      i =
        NetLibDmReceive (AppNetRefnum, sock, recordP, off,
                         len > CHUNK_SIZE ? CHUNK_SIZE : len, 0, NULL, NULL,
                         AppNetTimeout, &ifErrs);
      if (i <= 0)
        break;
      off += i;
      len -= i;
      counter += i;
      update_progress ();
    }

  return off;
}

Char *
ExtractBaseLink (Int16 cardNo, LocalID id)
{
  Char *ret = NULL, *ptr;
  SaveSessionData ();
  CloseDocument ();
  OpenDocument (id, cardNo);

  RecordIDToURL (HOME_PAGE_ID, &ret);

  CloseDocument ();
  OpenDocument (Prefs ()->dbID, Prefs ()->cardNo);
  InitSessionData ();


  if ((ptr = ret))
    {
      while (*ptr)
        {
          if (*ptr == '#')
            *ptr = 0;
          ptr++;
        }
    }

  return ret;
}


static Char *
ReadInDB (void)
{
  DmOpenRef db;
  UInt16 r, nrecs, size;
  UInt32 uid;
  MemHandle mh;
  void *p;
  LocalID id;
  Int16 cardNo = 0;
  StrPrintF (buffer, "Zplk_cache_%d", (Prefs ()->cacheid)++);
  Prefs ()->cacheid %= CACHE_SIZE;

  id = DmFindDatabase (cardNo, buffer);
  if (id)
    DmDeleteDatabase (cardNo, id);

  DmCreateDatabase (cardNo, buffer, 'Plkr', 'Data', false);
  id = DmFindDatabase (cardNo, buffer);
  db = DmOpenDatabase (cardNo, id, dmModeReadWrite);


  counter = 0;

  if (CompleteRead (buffer, 6) != 6) /*Size in bytes, number of records */
    goto error;

  docsize = buffer[0];
  docsize += ((UInt32) buffer[1]) << 8L;
  docsize += ((UInt32) buffer[2]) << 16L;
  docsize += ((UInt32) buffer[3]) << 24L;
  docsize++;

  nrecs = buffer[4];
  nrecs += ((UInt16) buffer[5]) << 8;

  for (r = 0; r < nrecs; ++r)
    {
      if (CompleteRead (buffer, 6) != 6) /*UID, then size */
        goto error;

      uid = buffer[0];
      uid += ((UInt32) buffer[1]) << 8L;
      uid += ((UInt32) buffer[2]) << 16L;
      uid += ((UInt32) buffer[3]) << 24L;
      size = buffer[4];
      size += ((UInt16) buffer[5]) << 8;

      mh = DmNewRecord (db, &r, size);

      p = MemHandleLock (mh);
      CompleteDmRead (p, size);
      MemHandleUnlock (mh);
      DmReleaseRecord (db, r, true);
      DmSetRecordInfo (db, r, NULL, &uid);

    }
  DmCloseDatabase (db);
  return ExtractBaseLink (cardNo, id);

error:
  set_progress ("err");
  DmCloseDatabase (db);
  return NULL;

}

static int
CheckSig (void)
{
  if (CompleteRead (buffer, 4) != 4)
    return 0;

  if ((buffer[0] != 0x54) || (buffer[1] != 0xf1) || (buffer[2] != 0xa1)
      || (buffer[3] != 0x0e))
    return 0;
  return 1;

}

static void
Blat (Char * msg, Int16 y)
{
  FontID oldFont;

  oldFont = FntSetFont (stdFont);
  StrNCopy (linbuf, msg, LINE_LEN);
  TrimText (linbuf, bounds.extent.x - 2);
  WinDrawChars (linbuf, StrLen (linbuf), bounds.topLeft.x, y);
  FntSetFont (oldFont);



}

static int
IsNetUp (void)
{
#if 0
  Err err;
  UInt32 ifCreator;
  UInt16 index;
  UInt16 ifInstance;
  UInt16 len;
  UInt8 up;

#define SAFETY 20

  for (index = 0; index < SAFETY; index++)
    {
      if (NetLibIFGet (AppNetRefnum, index, &ifCreator, &ifInstance))
        return 0;


      if (ifCreator == netIFCreatorLoop)
        continue;

      up = 0;
      len = sizeof (up);

      NetLibIFSettingGet (AppNetRefnum, ifCreator, ifInstance, netIFSettingUp,
                          &up, &len);
      if (up)
        return 1;
    }
  return 0;
#else
  return yesfetch;
#endif
}

Char *
FetchDB (Char * url)
{
  NetHostInfoPtr nhinfo;
  NetHostInfoBufType nh;
  NetSocketAddrINType sain;
  Char *baseurl = NULL;
  Char *m1, *m2, *host;
  UInt16 port = 80;
  UInt16 opencount;

  Err ifErrs;
  Int16 err;

  Int32 i, j;

  m1 = StrChr (Prefs ()->proxyName, '/');
  if (!m1)
    return NULL;
  m1 += 1;

  docsize = 1000;
  counter = 0;
  set_progress (">>>");


  WinEraseRectangle (&bounds, 0);

  Blat ("Fetching:", 20);
  Blat (url, 40);
  empty_counter ();


  AppNetTimeout = SysTicksPerSecond () * 45;

  if (SysLibFind ("Net.lib", &AppNetRefnum))
    return NULL;


  /* Prompt user if nework is down */
  if ((!IsNetUp ()) && (FrmAlert (confirmFetchFromNet)))
    return NULL;

  yesfetch = 1;

  err = NetLibOpen (AppNetRefnum, &ifErrs);
  if (err || ifErrs)
    {
      NetLibClose (AppNetRefnum, true);
      return NULL;
    }
  sock = NetLibSocketOpen (AppNetRefnum, netSocketAddrINET,
                           netSocketTypeStream, 0, AppNetTimeout, &ifErrs);

  if (ifErrs)
    {
      NetLibClose (AppNetRefnum, false);
      set_progress ("skt");

      return NULL;
    }

  host = MemPtrNew (proxyNameLength);
  if (!host)
    return NULL;
  StrCopy (host, Prefs ()->proxyName);

  m2 = StrChr (host, '/');
  *m2 = 0;
  m2 = StrChr (host, ':');
  if (m2)
    {
      port = StrAToI (m2 + 1);
      *m2 = 0;
    }



  /* Lookup/Connect to the server */
  nhinfo = NetLibGetHostByName (AppNetRefnum, host, &nh, AppNetTimeout,
                                &ifErrs);


  if (ifErrs)
    {
      NetLibClose (AppNetRefnum, false);
      set_progress ("dns");
      MemPtrFree (host);
      return NULL;
    }
  /* Fill in the sain structure */
  MemSet (&sain, sizeof (sain), 0);
  sain.addr = nh.address[0];
  sain.family = netSocketAddrINET;
  sain.port = NetHToNS (port);

  /* Connect to the LJ server */
  err = NetLibSocketConnect (AppNetRefnum, sock,
                             (NetSocketAddrType *) & sain, sizeof (sain),
                             AppNetTimeout, &ifErrs);
  if (err || ifErrs)
    {
      set_progress ("cnt");
      NetLibClose (AppNetRefnum, false);
      MemPtrFree (host);
      return NULL;
    }

  buffer = MemPtrNew (MAXBUF);

  if (buffer == NULL)
    {
      NetLibSocketClose (AppNetRefnum, sock, AppNetTimeout, &ifErrs);
      NetLibClose (AppNetRefnum, false);
      MemPtrFree (host);
      return NULL;
    }

  UrlCat (buffer, url);

  /* Send HTTP headers */
  NetLibSend (AppNetRefnum, sock, "GET ", 4, 0, 0, 0, AppNetTimeout, &ifErrs);
  NetLibSend (AppNetRefnum, sock, "/cgi-bin/pluck.cgi", 18, 0, 0, 0,
              AppNetTimeout, &ifErrs);
  NetLibSend (AppNetRefnum, sock, "?", 1, 0, 0, 0, AppNetTimeout, &ifErrs);
  NetLibSend (AppNetRefnum, sock, buffer, StrLen (buffer), 0, 0, 0,
              AppNetTimeout, &ifErrs);
  NetLibSend (AppNetRefnum, sock, " HTTP/1.0\nHost: ", 16, 0, 0, 0,
              AppNetTimeout, &ifErrs);
  NetLibSend (AppNetRefnum, sock, host, StrLen (host), 0, 0, 0,
              AppNetTimeout, &ifErrs);
  NetLibSend (AppNetRefnum, sock, "\n\n", 2, 0, 0, 0, AppNetTimeout, &ifErrs);

  MemPtrFree (host);

  AppNetTimeout = SysTicksPerSecond () * 45;

  j = 0;
  buffer[0] = 0;
  do
    {
      i =
        NetLibReceive (AppNetRefnum, sock, buffer + j, 1, 0, NULL, NULL,
                       AppNetTimeout, &ifErrs);
      if (buffer[j] == '\n')
        {
          buffer[j] = 0;
          break;
        }
      if (i > 0)
        j += i;
    }
  while ((i > 0) && (j < (MAXBUF - 2)));

  buffer[j] = 0;

  /*StrStr underflows for some reason on 3.5.0 if *buffer=0 */
  if ((i <= 0) || ((StrLen (buffer) > 5) && (!StrStr (buffer, " 200 "))))
    {
      set_progress ("200");
      goto error;
    }


  j = 0;
  do
    {
      i =
        NetLibReceive (AppNetRefnum, sock, buffer, 1, 0, NULL, NULL,
                       AppNetTimeout, &ifErrs);
      if (i > 0)
        {
          switch (j)
            {
            case 0:
              if (buffer[0] == '\r')
                j++;
              break;
            case 1:
              if (buffer[0] == '\n')
                j++;
              else
                j = 0;
              break;
            case 2:
              if (buffer[0] == '\r')
                j++;
              else
                j = 0;
              break;
            case 3:
              if (buffer[0] == '\n')
                j++;
              else
                j = 0;
              break;
            }
        }
    }
  while ((i > 0) && (j != 4));

  if ((i <= 0) || (j != 4))
    {
      set_progress ("pse");
      goto error;
    }


  if (!CheckSig ())
    {
      set_progress ("sig");
      goto quit;
    }
  if (!(baseurl = ReadInDB ()))
    goto quit;
  if (!CheckSig ())
    {
      set_progress ("sig");
      if (baseurl)
        {
          MemPtrFree (baseurl);
          baseurl = NULL;
        }
      goto quit;
    }
  goto quit;

error:
quit:
  NetLibSocketClose (AppNetRefnum, sock, AppNetTimeout, &ifErrs);
  NetLibClose (AppNetRefnum, false);

  MemPtrFree (buffer);
  WinEraseRectangle (&bounds, 0);

  return baseurl;
}
