/*
 * $Id: os.c,v 1.29 2001/11/26 20:09:02 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 */

#include "debug.h"
#include "os.h"
#include "resourceids.h"
#include "screen.h"
#include "prefsdata.h"

#include "SysZLib.h"
#include "AutoCF.h"
#include "ffslib.h"


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
static FontID os3StandardStyles[ MAXSTYLES ] = {
    stdFont, largeBoldFont, largeBoldFont, largeFont,
    largeFont, boldFont, boldFont, boldFont, narrowFixedFont
};

static FontID os3BoldStyles[ MAXSTYLES ] = {
    boldFont, largeBoldFont, largeBoldFont, largeBoldFont,
    largeBoldFont, boldFont, boldFont, boldFont, narrowFixedFont
};

static FontID os3LargeStyles[ MAXSTYLES ] = {
    largeFont, largeBoldFont, largeBoldFont, largeFont,
    largeFont, largeBoldFont, largeBoldFont, largeBoldFont, narrowFixedFont
};

static FontID os3LargeBoldStyles[ MAXSTYLES ] = {
    largeBoldFont, largeBoldFont, largeBoldFont, largeBoldFont,
    largeBoldFont, largeBoldFont, largeBoldFont, largeBoldFont, narrowFixedFont
};

static FontID os3NarrowStyles[ MAXSTYLES ] = {
    narrowFont, largeBoldFont, largeBoldFont, largeFont,
    largeFont, boldFont, boldFont, boldFont, narrowFixedFont
};

static FontID os2StandardStyles[ MAXSTYLES ] = {
    stdFont, boldFont, boldFont, boldFont,
    boldFont, stdFont, stdFont, stdFont, stdFont
};

static FontID os2BoldStyles[ MAXSTYLES ] = {
    boldFont, boldFont, boldFont, boldFont,
    boldFont, boldFont, boldFont, boldFont, stdFont
};


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static FontID*      Styles              = NULL;
static UInt32       osVersion           = 0;
static Boolean      zlibStatus          = false;
static Boolean      autoCFStatus        = false;
static Boolean      beamStatus          = false;
static Boolean      ffsStatus           = false;
static UInt32       maxScreenDepth      = 0;
static UInt32       defaultScreenDepth  = 0;
static UInt16       FfsRef              = 0;
static FontType*    narrow              = NULL;
static FontType*    narrowFixed         = NULL;



/* Return Ffs library ref */
UInt16 GetFsfRef( void )
{
    return FfsRef;
}



/* Return font style */
FontID GetStyle
    (
    const Int16 style   /* numerical representation for font style */
    )
{
    return Styles[ style ];
}



/* Return OS version */
UInt32 GetOSVersion( void )
{
    return osVersion;
}



/* Return max screen depth for this device */
UInt32 GetMaxScreenDepth( void )
{
    return maxScreenDepth;
}



/* Return if device is using the default screen depth  */
Boolean DefaultScreenMode( void )
{
    return ( Prefs()->screenDepth == defaultScreenDepth );
}



/* Return status for ZLib support */
Boolean ZLibSupported( void )
{
    return zlibStatus;
}



/* Return status for AutoCF support */
Boolean AutoCFSupported( void )
{
    return autoCFStatus;
}



/* Return status for beam support */
Boolean BeamSupported( void )
{
    return beamStatus;
}



/* Return status for Ffs library support */
Boolean FfsSupported( void )
{
    return ffsStatus;
}

/* Set selected font style */
void SetFontStylesOS3( void )
{
    switch ( Prefs()->fontMode ) {
        case FONT_BOLD:
            Styles = os3BoldStyles;
            break;

        case FONT_LARGE:
            Styles = os3LargeStyles;
            break;

        case FONT_LARGEBOLD:
            Styles = os3LargeBoldStyles;
            break;

        case FONT_NARROW:
            Styles = os3NarrowStyles;
            break;

        default:
            Styles = os3StandardStyles;
            break;
    }
}



/* Set selected font style */
void SetFontStylesOS2( void )
{
    switch ( Prefs()->fontMode ) {
        case FONT_BOLD:
            Styles = os2BoldStyles;
            break;

        default:
            Styles = os2StandardStyles;
            break;
    }
}



/* PalmOS 3.x specific operations */
static void OS3( void )
{
    UInt32 supportedDepths;
    UInt32 version;
    UInt16 i;

    WinScreenMode( winScreenModeGetSupportedDepths, NULL, NULL, 
        &supportedDepths, NULL );

    for ( i = 1; supportedDepths; i++ ) {
        if ( ( supportedDepths & 0x01 ) == 0x01 )
            maxScreenDepth = i;
        supportedDepths >>= 1;
    }
    WinScreenMode( winScreenModeGetDefaults, NULL, NULL, 
        &defaultScreenDepth, NULL );

    if ( sysGetROMVerMinor( osVersion ) < 5 )
        depthCheck = TooHighBitDepth_Default;
    else
        depthCheck = TooHighBitDepth_Max;

    screenMode  = SetScreenModeOS3;
    defaultMode = SetDefaultScreenModeOS3;
    fontStyles  = SetFontStylesOS3;

    if ( ( FtrGet( TRG_CREATOR, AUTOCF_FEATURE_ID, &version ) == errNone ) &&
         AUTOCF_LOADED )
        autoCFStatus = true;
    else
        autoCFStatus = false;

    beamStatus = true; /* TODO: Add check for beam support since it could be missing on an OS3+ device */

    if ( ( SysLibFind( "Z.lib", &ZLibRef ) == errNone ) || 
         ( SysLibLoad( 'libr', 'ZLib', &ZLibRef ) == errNone ) ) {
        ZLibOpen( ZLibRef );
        zlibStatus = true;
    }
    else {
        zlibStatus  = false;
        ZLibRef     = NULL;
    }
    if ( ( SysLibFind( FfsLibName, &FfsRef ) == errNone ) || 
         ( SysLibLoad( FfsLibTypeID, FfsLibCreatorID, &FfsRef ) == errNone ) ) {
        FfsLibOpen( FfsRef );
        ffsStatus = true;
    }
    else {
        ffsStatus   = false;
        FfsRef      = 0;
    }
    narrow = (FontType*) MemHandleLock( DmGetResource( 'NFNT', narrowFontID ) );
    FntDefineFont( (FontID) narrowFont, narrow );

    narrowFixed = MemHandleLock( DmGetResource( 'NFNT', narrowFixedFontID ) );
    FntDefineFont( (FontID) narrowFixedFont, narrowFixed );
}



/* PalmOS 4.x specific operations */
static void OS4( void )
{
    OS3();
    depthCheck = TooHighBitDepth_Max;
}



/* PalmOS 2.x specific operations. */
static void OS2( void )
{
    depthCheck          = TooHighBitDepth_Default;
    screenMode          = SetScreenModeOS2;
    defaultMode         = SetDefaultScreenModeOS2;
    fontStyles          = SetFontStylesOS2;
    maxScreenDepth      = 1;
    defaultScreenDepth  = 1;
}



/* Handle start of OS specific features */
Err OS_Start( void )
{
    FtrGet( sysFtrCreator, sysFtrNumROMVersion, &osVersion );

    switch ( sysGetROMVerMajor( osVersion ) ) {
        case 0x01:
            FATAL( _( "wrong ROM version detected ( %lx )\n", osVersion ) );
            FrmAlert( infoWrongROMVersion );
            return sysErrRomIncompatible;

        case 0x02:
            OS2();
            break;

        case 0x03:
            OS3();
            break;

        case 0x04:
            OS4();
            break;

        default:
            OS2();
    }
    return 0;
}



/* Handle stop of OS specific features. */
void OS_Stop( void )
{
    if ( narrowFixed != NULL )
        MemPtrUnlock( narrowFixed );

    if ( narrow != NULL )
        MemPtrUnlock( narrow );

    ZLTeardown;
    if ( FfsRef != 0 ) {
        if ( FfsLibClose( FfsRef ) == errNone )
            SysLibRemove( FfsRef );
        FfsRef = 0;
    }
}
