/*
 * $Id: paragraph.h,v 1.26 2001/11/04 18:44:54 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifndef __PLUCKER_PARAGRAPH_H
#define __PLUCKER_PARAGRAPH_H

#include "viewer.h"
#include "document.h"
#include "util.h"

/* These numbers are the hex representation of the oct function codes
   found in PluckerDocs.py (eg 0123 in PluckerDocs.py is 0x53 here). */
typedef enum {
    ANCHOR_START        = 0x0A,
    NAMED_ANCHOR_START  = 0x0C,
    ANCHOR_END          = 0x08,
    SET_STYLE           = 0x11,
    INLINE_IMAGE        = 0x1A,
    SET_MARGIN          = 0x22,
    ALIGNMENT           = 0x29,
    HRULE               = 0x33,
    NEWLINE             = 0x38,
    ITALIC_START        = 0x40,
    ITALIC_END          = 0x48,
    SET_FORECOLOR       = 0x53,
    MULTI_IMAGE         = 0x5C,
    UNDERLINE_START     = 0x60,
    UNDERLINE_END       = 0x68,
    STRIKE_START        = 0x70,
    STRIKE_END          = 0x78
} ParagraphFunction;


typedef enum {
    TOKEN_CHARACTER     = -2, 
    TOKEN_PARAGRAPH_END = -1,
    TOKEN_FUNCTION      = 0
} TokenType;


typedef enum {
    ALIGNMENT_LEFT      = 0, 
    ALIGNMENT_RIGHT     = 1,
    ALIGNMENT_CENTER    = 2, 
    ALIGNMENT_JUSTIFY   = 3
} AlignmentType;


/*
    A paragraph as it appears in the input data stream. The height of the
    paragraph is calculated on the device.

    The "previous" is a relative offset to the previous paragraph; normally
    it is the same as the previous paragraph's length. (The offset is always
    positive; subtract the offset from the current paragraph's pointer to get
    the previous paragraph's pointer).

    Notes:
    
    (1) A paragraph contains either characters or "functions". A function is
    introduced by a NUL (\0), followed by a function code, followed by up
    to 7 bytes of data (at the moment). The function code's least significant
    3 bits represent the remaining function code length; the most significant
    5 bits represent the function code. For example, the 'ForeColor start'
    function is 0x83, or 10000011. Breaking it down, 10000 is the unique
    function code, and 011 tells that there is 3 arguments to the function--
    in this case, red, green, and blue.
    
    (2) Unlike most other places in this program, function code length does
    NOT include the introductory bytes (as opposed to objectLength which
    includes the length of the objectLength field). So to get the next
    character/function, you add the length of the function operators + 2 to 
    the pointer to the start of the function.

    Defined function codes:

    00000xxx    0x00  Dummy code (used internally)
    00001010    0x0A  Anchor begins (record ID)
    00001100    0x0C  Named anchor begins (record ID and paragraph offset)
    00001000    0x08  Anchor ends. (==anchor begin with no data)
    00010001    0x11  Set style. (Style is always #0 at start of paragraph)
    00011010    0x1A  Include image (record ID)
    00100010    0x22  Set left and/or right margin (margins in pixels)
    00101001    0x29  Alignment of text (left = 0, right = 1, center = 2)
    00110011    0x33  Horizontal rule
    00111000    0x38  New line
    01000000    0x40  Italic text begins
    01001000    0x48  Italic text ends
    01010011    0x53  Set forecolor (red, green, blue)
    01011100    0x5C  Multi-image
    01100000    0x60  Underlined text begins
    01101000    0x68  Underlined text ends
    01110000    0x70  Strikethrough text starts
    01111000    0x78  Strikethrough text ends

    Note about attributes:
    For Paragraph objects, the attributes consist of the following bits:
    MSB \
    x    \
    x     + Unused.
    x    /
    x   /
    x   \ 
    x    + These bits indicates the extra paragraph spacing required
    LSB /  above the paragraph (value * DEFAULT_PARAGRAPH_SPACING (
           defined in const.h) pixels of spacing).
*/

typedef struct {
    Int16 size;         /* Size of text */
    Int16 attributes;   /* Paragraph info (see above) */                           
} Paragraph;


typedef struct {
    Int16 height;       /* Height of the paragraph in pixels (see above) */
} MetaParagraph;


/*
   ParagraphContext is used to help parse paragraphs.
 */
typedef struct {
    Char*           last;       /* Points to last character in paragraph + 1 */
    Char*           position;   /* Points to next character in paragraph */
    UInt8*          function;   /* Points to function if last token was a 
                                   function */
    Int16           fontHeight; /* Height of line in current font */
    Int16           left;       /* Left margin */
    Int16           right;      /* Right margin */
    Int16           maxPixels;  /* Max number of pixels for line */
    Int16           linePixels; /* Number of pixels for current line */
    AlignmentType   type;       /* Alignment type */
} ParagraphContext;


void DrawParagraph( TextContext* tContext, Paragraph* paragraph,
        Header* record );
void SetFindPatternData( const Int16 pos, const Int16 len );
void ClearFindPatternData( void );
void DoEraseRectangle(UInt8 *u,RectangleType *rt);


/* Return pointer to paragraph with given index */
#define GET_PARAGRAPH( RECORD, INDEX )      ( (Paragraph*) ( (UInt8*) ( ( RECORD ) + 1 ) + 4 + ( INDEX ) * sizeof( Paragraph ) ) )

/* Return pointer to meta paragraph with given index */
#define GET_METAPARAGRAPH( RECORD, INDEX )  ( (MetaParagraph*) ( (UInt8*) ( ( RECORD ) + 1 ) + ( INDEX ) * sizeof( MetaParagraph ) ) )

/* Return pointer to data in given record */
#define GET_DATA( RECORD )                  ( (UInt8*) ( ( RECORD ) + 1 ) + ( RECORD )->paragraphs * sizeof( Paragraph ) )

#endif
