/*
 * $Id: search.c,v 1.22 2001/09/16 15:07:38 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <CharAttr.h>

#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "paragraph.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "resultform.h"
#include "search.h"
#include "uncompress.h"
#include "util.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define SEARCH_AGAIN    false

#define SELECT_OK       0


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static Boolean  foundString;
static UInt16   recordNum;
static UInt16   position;



/* Return caseless but still accented value for character */
static Char GetCaselessValue
    (
    Char c
    )
{
    if ( IsUpper( GetCharAttr(), c ) )
        return ( c + ' ' );
    else
        return c;
}



/* Find contextual information */
static void FindContext
    (
    Char*   text,       /* text string */
    Int16   size,       /* size of text */
    UInt16  pos,        /* position of the pattern */
    UInt16  function,   /* position of function before pattern */
    UInt16  len,        /* length of pattern */
    Char*   result      /* will contain the result string after return
                           of function */
    )
{
    UInt16 start;
    UInt16 stop;
    UInt16 i;

    start   = function + 2 + ( text[ function + 1 ] & 0x07 );
    stop    = size;
    for ( i = pos; i < pos + 30; i++ ) {
        if ( text[ i ] == '\0' ) {
            stop = i - 1;
            break;
        }
    }
    if ( 32 <= ( stop - start + 1 ) ) {
        if ( start == pos ) {
            if ( 32 <= ( stop - pos ) )
                stop = pos + 30;
        }
        else if ( stop == ( pos + len ) ) {
            if ( 32 <= ( pos - start ) )
                start = pos - ( 31 - len + 1 );
        }
        else {
            if ( 16 <= ( pos - start ) )
                start = pos - 15;
            if ( 16 <= ( stop - pos ) )
                stop = pos + 15;
        }
    }
    StrNCopy( result, &text[ start ], stop - start + 1 );
}



/* Find pattern in text string */
static Boolean DoSearch
    (
    Char*   text,           /* text string */
    UInt16  size,           /* size of text */
    Char*   pattern,        /* pattern to search for */
    Boolean caseSensitive,  /* indicates if the search should be
                               case-sensitive or not */
    UInt16* pos,            /* start position, will contain the position
                               of the pattern if found */
    UInt16* function        /* position of function before pattern,
                               set to NULL if you are not interested
                               in this information */
    )
{
    Int16 m;
    Int16 i;
    Int16 j;

    m = StrLen( pattern );

    if ( function != NULL )
        *function = 0;

    for (i = *pos, j = 0; j < m && i < size; i++, j++) {
        if ( text[ i ] == '\0' ) {
            if ( function != NULL )
                *function = i;
            i  +=  1 + ( text[ i + 1 ] & 0x07 );
            j   = -1;
            continue;
        }
        if ( caseSensitive ) {
            if ( text[ i ] != pattern[ j ] ) {
                i  -=  j;
                j   = -1;
            }
        }
        else {
            if ( GetCaselessValue( text[ i ] ) != GetCaselessValue( pattern[ j ] ) ) {
                i  -=  j;
                j   = -1;
            }
        }
    }
    if ( j == m ) {
        *pos = i - m;
        return true;
    }
    else
        return false;
}



/* Search in current page */
Boolean SearchInPage
    (
    Char*   pattern,    /* pattern to search for */
    Boolean startAtTop  /* indicates if the search should start
                           at the top of the page or from the
                           current position */
    )
{
    Header* record;
    Char*   text;
    UInt16  size;
    Boolean match;
    Boolean status;

    match   = false;
    status  = false;

    record = (Header*) MemHandleLock( GetCurrentRecord() );
    ErrFatalDisplayIf( record == NULL, "SearchInPage: MemHandleLock failed" );

    if ( record->type == DATATYPE_TBMP || record->type == DATATYPE_TBMP_COMPRESSED ) {
        MemPtrUnlock( record );
        return false;
    }

    if ( record->type == DATATYPE_PHTML_COMPRESSED ) {
        text = (Char*) MemHandleLock( GetUncompressTextHandle() );
        size = MemPtrSize( text );
    }
    else {
        text = ((Char*) GET_DATA( record ))+4;
        size = record->size;
    }

    if ( startAtTop ) {
        position    = 0;
        foundString = false;
    }
    else
        position++;

    ClearFindPatternData();

    match = DoSearch( text, size, pattern, Prefs()->caseSensitive, &position, NULL );

    if ( match ) {
        SetFindPatternData( position, StrLen( pattern ) );
        foundString = true;
        status      = true;
    }
    else if ( foundString ) {
        if ( FrmAlert( confirmEndOfDoc ) == SELECT_OK ) {
            SearchInPage( pattern, true );
            status = true;
        }
    }
    else {
        FrmAlert( errStringNotFound );
        status = false;
    }

    if ( record->type == DATATYPE_PHTML_COMPRESSED )
        MemHandleUnlock( GetUncompressTextHandle() );

    MemPtrUnlock( record );

    return status;
}



/* Search in all pages */
Boolean SearchInAllPages
    (
    Char*       pattern,            /* pattern to search for */
    Boolean     startFromBeginning, /* start from beginning of record */
    Boolean*    findMore            /* indicates if there are remaining
                                       records to search through */
    )
{
    Boolean done;
    UInt16  numOfRecords;

    if ( startFromBeginning ) {
        recordNum   = 0;
        position    = 0;
    }

    *findMore   = true;
    done        = false;

    numOfRecords = GetNumOfRecords();

    for ( ;; ) {
        MemHandle   uncompressHandle;
        MemHandle   handle;
        Header*     record;
        Char*       text;
        Char        header[ 5 ];
        UInt16      uniqueID;
        Int32       percent;
        Int16       size;
        Err         err;

        if ( EvtSysEventAvail( true ) ) {
            done = false;
            break;
        }

        percent = recordNum * 100L / numOfRecords + 1;
        StrPrintF( header, "%3ld%%", percent );

        done = DrawResultHeader( header );

        if ( done )
            break;

        err = ReturnNextRecordHandle( &recordNum, &handle, &uniqueID );

        if ( err != errNone ) {
            recordNum   = 0;
            *findMore   = false;
            done        = true;
            break;
        }
        record = (Header*) MemHandleLock( handle );
        ErrFatalDisplayIf( record == NULL, "SearchInAllPages: MemHandleLock failed" );

        uncompressHandle = NULL;

        if ( record->type == DATATYPE_PHTML_COMPRESSED ) {
            uncompressHandle = Uncompress( record );
            if ( uncompressHandle == NULL )
                goto nextRecord;
            text = (Char*) MemHandleLock( uncompressHandle );
            size = MemPtrSize( text );
        }
        else {
            text = ((Char*) GET_DATA( record ))+4;
            size = record->size;
        }

        for ( ;; ) {
            Boolean match;
            UInt16  function;

            match = DoSearch( text, size, pattern, Prefs()->caseSensitive, &position, &function );

            if ( match ) {
                Char result[ 32 ];
                MemSet( result, 32, 0 );

                FindContext( text, size, position, function, StrLen( pattern ), result );
                done = DrawResultString( uniqueID, position, StrLen( pattern ), result );
                position++;
            }
            else
                break;

            if ( done )
                break;
        }
        if ( record->type == DATATYPE_PHTML_COMPRESSED )
            MemHandleUnlock( uncompressHandle );

    nextRecord:

        MemHandleUnlock( handle );

        if ( done )
            break;

        recordNum++;
        position = 0;
    }
    return done;
}



/* Search again in current page */
void SearchAgain( void )
{
    Boolean contSearch;
    Char    pattern[ 16 ];

    GetSearchString( pattern );

    if ( StrLen( pattern ) == 0 )
        return;

    contSearch = SearchInPage( pattern, SEARCH_AGAIN );

    if ( contSearch )
        DoPageMove( GetSearchAdjustment() );
}
